/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cdate.c                 */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Tue Feb  4 11:51:17 2003                          */
/*    Last change :  Wed Jun  9 04:56:47 2004 (serrano)                */
/*    Copyright   :  2003-04 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    C implementation of time & date                                  */
/*=====================================================================*/
#include <bigloo.h>
#include <time.h>

/*---------------------------------------------------------------------*/
/*    static obj_t                                                     */
/*    tm_to_date ...                                                   */
/*---------------------------------------------------------------------*/
static obj_t
tm_to_date( struct tm *tm, time_t *t ) {
   obj_t date;

   date = GC_MALLOC_ATOMIC( BGL_DATE_SIZE );
   date->date_t.header = MAKE_HEADER( DATE_TYPE, 0 );

   date->date_t.timezone = BGL_TIMEZONE;
      
   date->date_t.sec = tm->tm_sec;
   date->date_t.min = tm->tm_min;
   date->date_t.hour = tm->tm_hour;
      
   date->date_t.mday = tm->tm_mday;
   date->date_t.mon = tm->tm_mon + 1;
   date->date_t.year = tm->tm_year + 1900;
   date->date_t.wday = tm->tm_wday + 1;
   date->date_t.yday = tm->tm_yday + 1;

   date->date_t.isdst = tm->tm_isdst;

   return BGL_HEAP_DEBUG_MARK_OBJ( BREF( date ) );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_seconds_to_date ...                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bgl_seconds_to_date( long sec ) {
   return tm_to_date( localtime( (time_t *)&sec ), (time_t *)&sec );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_make_date ...                                                */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bgl_make_date( int s, int m, int hr, int mday, int mon, int year, long tz, bool_t istz, int isdst ) {
   struct tm tm;
   time_t t;
   long delta = istz ? tz - BGL_TIMEZONE : 0;

   tm.tm_sec = s;
   tm.tm_min = m;
   tm.tm_hour = hr;
   tm.tm_mday = mday;
   tm.tm_mon = mon - 1;
   tm.tm_year = year - 1900;
   tm.tm_isdst = isdst;

   t = mktime( &tm );
   
   return bgl_seconds_to_date( (long)t + delta );
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    bgl_date_to_seconds ...                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
long
bgl_date_to_seconds( obj_t date ) {
   struct tm t;
   time_t n;

   t.tm_sec = BGL_DATE( date ).sec;
   t.tm_min = BGL_DATE( date ).min;
   t.tm_hour = BGL_DATE( date ).hour;
   t.tm_mday = BGL_DATE( date ).mday;
   t.tm_mon = BGL_DATE( date ).mon - 1;
   t.tm_year = BGL_DATE( date ).year - 1900;
   t.tm_isdst = -1;

   n = mktime( &t );
   
   return (long)n;
}

/*---------------------------------------------------------------------*/
/*    long                                                             */
/*    bgl_current_seconds ...                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
long
bgl_current_seconds() {
   return (long)( time( 0L ) );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_seconds_to_utc_string ...                                    */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bgl_seconds_to_utc_string( long sec ) {
   struct tm *t;
   char *s;

   t = gmtime( (time_t *)&sec );
   s = asctime( t );

   return string_to_bstring_len( s, (int)strlen( s ) - 1 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_seconds_to_string ...                                        */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DEF
obj_t
bgl_seconds_to_string( long sec ) {
   char *s = ctime( (time_t *)&sec );
   
   return string_to_bstring_len( s, (int)strlen( s ) - 1 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_seconds_format ...                                           */
/*---------------------------------------------------------------------*/
obj_t
bgl_seconds_format( long sec, obj_t fmt ) {
   char *buffer;
   struct tm *p;
   int len = (int)STRING_LENGTH( fmt ) + 256;

   buffer = (char *)GC_MALLOC_ATOMIC( len + 1 );
   p = localtime( (time_t *)&sec );
   
   len = (int)strftime( buffer, len, BSTRING_TO_STRING( fmt ), p );

   if( len > 0 )
      return string_to_bstring_len( buffer, len );
   else
      FAILURE( string_to_bstring( "seconds-format" ),
	       string_to_bstring( "buffer too short!" ),
	       BINT( 256 ) );
}

/*---------------------------------------------------------------------*/
/*    static obj_t                                                     */
/*    make_names ...                                                   */
/*---------------------------------------------------------------------*/
static obj_t
make_names( int range, char *fmt ) {
   obj_t names = (obj_t)create_vector( range );
   char buf[ 40 ];
   struct tm tm;
   int i;

   for( i = 0; i < range; i++ ) {
      tm.tm_wday = i;
      tm.tm_mon = i;
      strftime( buf, 40, fmt, &tm );
      VECTOR_SET( names, i, string_to_bstring( buf ) );
   }

   return names;
}
   
/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_day_name ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
bgl_day_name( int day ) {
   static obj_t names = BNIL;
   
   if( names == BNIL ) names = make_names( 7, "%A" );

   return VECTOR_REF( names, day-1 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_day_aname ...                                                */
/*---------------------------------------------------------------------*/
obj_t
bgl_day_aname( int day ) {
   static obj_t names = BNIL;
   
   if( names == BNIL ) names = make_names( 7, "%a" );

   return VECTOR_REF( names, day-1 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_month_name ...                                               */
/*---------------------------------------------------------------------*/
obj_t
bgl_month_name( int month ) {
   static obj_t names = BNIL;
   
   if( names == BNIL ) names = make_names( 12, "%B" );

   return VECTOR_REF( names, month-1 );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bgl_month_aname ...                                              */
/*---------------------------------------------------------------------*/
obj_t
bgl_month_aname( int month ) {
   static obj_t names = BNIL;
   
   if( names == BNIL ) names = make_names( 12, "%b" );

   return VECTOR_REF( names, month-1 );
}
