/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/csymbol.c               */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Feb 12 14:51:41 1992                          */
/*    Last change :  Tue Dec  5 07:23:46 2000 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Symbol handling (creation and hash tabling).                     */
/*=====================================================================*/
#include <string.h>
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Quelques prototypes                                              */
/*---------------------------------------------------------------------*/
extern obj_t string_to_symbol( char * );
static obj_t make_symbol( char * );
extern obj_t make_vector();
extern long get_hash_power_number( char *, unsigned long );

/*---------------------------------------------------------------------*/
/*    Quelques variables extern                                        */
/*---------------------------------------------------------------------*/
obj_t c_symtab;

/*---------------------------------------------------------------------*/
/*    init_symbol_table ...                                            */
/*---------------------------------------------------------------------*/
void init_symbol_table()
{
   c_symtab = make_vector( SYMBOL_HASH_TABLE_SIZE, BNIL );
}
          
/*---------------------------------------------------------------------*/
/*    make_symbol ...                                                  */
/*    -------------------------------------------------------------    */
/*    Symbols are not allocated in the first generation because        */
/*    they can't be collected.                                         */
/*---------------------------------------------------------------------*/
static obj_t
make_symbol( char *name )
{
   obj_t symbol;

   symbol = GC_MALLOC( SYMBOL_SIZE );
   
   symbol->symbol_t.header = MAKE_HEADER( SYMBOL_TYPE, SYMBOL_SIZE );
   symbol->symbol_t.string = string_to_bstring( name );
   symbol->symbol_t.cval   = BNIL;

   return BGL_HEAP_DEBUG_MARK_OBJ( BREF( symbol ) );
}
   
/*---------------------------------------------------------------------*/
/*    string_to_symbol ...                                             */
/*    char * --> obj_t                                                 */
/*---------------------------------------------------------------------*/
obj_t
string_to_symbol( char *name )
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      obj_t symbol = make_symbol( name );
      obj_t pair   = MAKE_PAIR( symbol, BNIL );
      
      VECTOR_SET( c_symtab, hash_number, pair );
      
      return symbol;
   }
   else
   {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( (char *)BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ),
		     name ) )
         back = run, run = CDR( run );
      
      if( !NULLP( run ) )
         return CAR( run );
      else
      {
         obj_t symbol = make_symbol( name );
	 obj_t pair   = MAKE_PAIR( symbol, BNIL );
	 
         SET_CDR( back, pair );

         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    symbol_exists_p ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne vrai s'il existe un symbol de nom        */
/*    `string' et faux sinon                                           */
/*---------------------------------------------------------------------*/
int
symbol_exists_p( char *name )
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, SYMBOL_HASH_TABLE_SIZE_SHIFT );
   bucket = VECTOR_REF( c_symtab, hash_number );
   
   if( NULLP( bucket ) )
   {
      return 0;
   }
   else
   {
      obj_t run = bucket;
      
      while( !NULLP( run ) &&
	     strcmp( (char *)BSTRING_TO_STRING( SYMBOL( CAR( run ) ).string ),
		     name ) )
         run = CDR( run );

      return !NULLP( run );
   }
}
