/** 
 * @namespace   biew
 * @file        bmfile.c
 * @brief       This file has developed as Buffered stream Manager and presents
 *              front end interface to bbio library.
 * @version     -
 * @remark      this source file is part of Binary vIEW project (BIEW).
 *              The Binary vIEW (BIEW) is copyright (C) 1995 Nick Kurshev.
 *              All rights reserved. This software is redistributable under the
 *              licence given in the file "Licence.en" ("Licence.ru" in russian
 *              translation) distributed in the BIEW archive.
 * @note        Requires POSIX compatible development system
 *
 * @author      Nick Kurshev
 * @since       1995
 * @note        Development, fixes and improvements
**/
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "bmfile.h"
#include "bconsole.h"
#include "biewlib/bbio.h"

static BGLOBAL file = NULL,sc_file = NULL;
static char FName[FILENAME_MAX+1];

BGLOBAL __FASTCALL__ biewOpenRO(char *fname,unsigned cache_size)
{
  BGLOBAL fret;
  fret = bioOpen(fname,FO_READONLY | SO_DENYNONE,cache_size);
  if(fret == &bNull)
    fret = bioOpen(fname,FO_READONLY | SO_COMPAT,cache_size);
  return fret;
}

BGLOBAL __FASTCALL__ biewOpenRW(char *fname,unsigned cache_size)
{
  BGLOBAL fret;
  fret = bioOpen(fname,FO_READWRITE | SO_DENYNONE,cache_size);
  if(fret == &bNull)
    fret = bioOpen(fname,FO_READWRITE | SO_COMPAT,cache_size);
  return fret;
}

int __FASTCALL__ BMOpen(char * fname)
{
  strcpy(FName,fname);
  file = biewOpenRO(fname,BBIO_CACHE_SIZE);
  if(file == &bNull)
  {
    errnoMessageBox("Can not open file",NULL,errno);
    return -1;
  }
  sc_file = bioDupEx(file,BBIO_SMALL_CACHE_SIZE);
  if(sc_file == &bNull)
  {
    errnoMessageBox("Can not dup file",NULL,errno);
    return -1;
  }
  bioSetOptimization(file,BIO_OPT_RANDOM);
  bioSetOptimization(sc_file,BIO_OPT_RANDOM);
  return 0;
}

int __FASTCALL__ BMHandle( void )
{
  return bioHandle(file);
}

BGLOBAL __FASTCALL__ BMbioHandle( void )
{
  return file;
}

void __FASTCALL__ BMClose( void )
{
  if(file != &bNull) bioClose(file);
  file = &bNull;
  if(sc_file != &bNull) bioClose(sc_file);
  sc_file = &bNull;
}

void __FASTCALL__ BMSeek(long pos,int RELATION)
{
  bioSeek(file,pos,RELATION);
}

void __FASTCALL__ BMReRead( void )
{
  bioReRead(file);
}

uint8_t  __FASTCALL__ BMReadByte( void )
{
  return bioReadByte(file);
}

uint16_t __FASTCALL__ BMReadWord( void )
{
 return bioReadWord(file);
}

uint32_t  __FASTCALL__ BMReadDWord( void )
{
  return bioReadDWord(file);
}

unsigned long  __FASTCALL__ BMGetCurrFilePos( void )
{
  return bioTell(file);
}

unsigned long __FASTCALL__ BMGetFLength( void )
{
  return bioFLength(file);
}

tBool __FASTCALL__ BMReadBuffer(void * buffer,unsigned len)
{
  return bioReadBuffer(file,buffer,len);
}

uint8_t  __FASTCALL__ BMReadByteEx(long pos,int RELATION)
{
 bioSeek(file,pos,RELATION);
 return bioReadByte(file);
}

uint16_t __FASTCALL__ BMReadWordEx(long pos,int RELATION)
{
 bioSeek(file,pos,RELATION);
 return bioReadWord(file);
}

uint32_t  __FASTCALL__ BMReadDWordEx(long pos,int RELATION)
{
 bioSeek(file,pos,RELATION);
 return bioReadDWord(file);
}

tBool __FASTCALL__ BMReadBufferEx(void * buffer,unsigned len,long pos,int RELATION)
{
 bioSeek(file,pos,RELATION);
 return bioReadBuffer(file,buffer,len);
}

tBool __FASTCALL__ BMWriteByte(uint8_t byte)
{
  return bioWriteByte(file,byte);
}

tBool __FASTCALL__ BMWriteWord(uint16_t word)
{
  return bioWriteWord(file,word);
}

tBool __FASTCALL__ BMWriteDWord(uint32_t dword)
{
  return bioWriteDWord(file,dword);
}

tBool __FASTCALL__ BMWriteBuff(void * buff,unsigned len)
{
  tBool ret;
  ret = bioWriteBuffer(file,buff,len);
  bioFlush(file);
  return ret;
}

tBool __FASTCALL__ BMWriteByteEx(long pos,int RELATION,uint8_t byte)
{
  bioSeek(file,pos,RELATION);
  return bioWriteByte(file,byte);
}

tBool __FASTCALL__ BMWriteWordEx(long pos,int RELATION,uint16_t word)
{
  bioSeek(file,pos,RELATION);
  return bioWriteWord(file,word);
}

tBool __FASTCALL__ BMWriteDWordEx(long pos,int RELATION,uint32_t dword)
{
  bioSeek(file,pos,RELATION);
  return bioWriteDWord(file,dword);
}

tBool  __FASTCALL__ BMWriteBuffEx(long pos,int RELATION,void * buff,unsigned len)
{
  bioSeek(file,pos,RELATION);
  return bioWriteBuffer(file,buff,len);
}

char * __FASTCALL__ BMName( void )
{
  return bioFileName(file);
}

tBool __FASTCALL__ BMEOF( void )
{
  return bioEOF(file);
}

int __FASTCALL__ bmHandle( void )
{
  return bioHandle(sc_file);
}

BGLOBAL __FASTCALL__ bmbioHandle( void )
{
  return sc_file;
}

void __FASTCALL__ bmSeek(long pos,int RELATION)
{
  bioSeek(sc_file,pos,RELATION);
}

void __FASTCALL__ bmReRead( void )
{
  bioReRead(sc_file);
}

uint8_t  __FASTCALL__ bmReadByte( void )
{
  return bioReadByte(sc_file);
}

uint16_t __FASTCALL__ bmReadWord( void )
{
 return bioReadWord(sc_file);
}

uint32_t  __FASTCALL__ bmReadDWord( void )
{
  return bioReadDWord(sc_file);
}

unsigned long  __FASTCALL__ bmGetCurrFilePos( void )
{
  return bioTell(sc_file);
}

unsigned long __FASTCALL__ bmGetFLength( void )
{
  return bioFLength(sc_file);
}

tBool __FASTCALL__ bmReadBuffer(void * buffer,unsigned len)
{
  return bioReadBuffer(sc_file,buffer,len);
}

uint8_t __FASTCALL__ bmReadByteEx(long pos,int RELATION)
{
 bioSeek(sc_file,pos,RELATION);
 return bioReadByte(sc_file);
}

uint16_t __FASTCALL__ bmReadWordEx(long pos,int RELATION)
{
 bioSeek(sc_file,pos,RELATION);
 return bioReadWord(sc_file);
}

uint32_t __FASTCALL__ bmReadDWordEx(long pos,int RELATION)
{
 bioSeek(sc_file,pos,RELATION);
 return bioReadDWord(sc_file);
}

tBool __FASTCALL__ bmReadBufferEx(void * buffer,unsigned len,long pos,int RELATION)
{
 bioSeek(sc_file,pos,RELATION);
 return bioReadBuffer(sc_file,buffer,len);
}

tBool __FASTCALL__ bmWriteByte(uint8_t byte)
{
  return bioWriteByte(sc_file,byte);
}

tBool __FASTCALL__ bmWriteWord(uint16_t word)
{
  return bioWriteWord(sc_file,word);
}

tBool __FASTCALL__ bmWriteDWord(uint32_t dword)
{
  return bioWriteDWord(sc_file,dword);
}

tBool __FASTCALL__ bmWriteBuff(void * buff,unsigned len)
{
  tBool ret;
  ret = bioWriteBuffer(sc_file,buff,len);
  bioFlush(sc_file);
  return ret;
}

tBool __FASTCALL__ bmWriteByteEx(long pos,int RELATION,uint8_t byte)
{
  bioSeek(sc_file,pos,RELATION);
  return bioWriteByte(sc_file,byte);
}

tBool __FASTCALL__ bmWriteWordEx(long pos,int RELATION,uint16_t word)
{
  bioSeek(sc_file,pos,RELATION);
  return bioWriteWord(sc_file,word);
}

tBool __FASTCALL__ bmWriteDWordEx(long pos,int RELATION,uint32_t dword)
{
  bioSeek(sc_file,pos,RELATION);
  return bioWriteDWord(sc_file,dword);
}

tBool  __FASTCALL__ bmWriteBuffEx(long pos,int RELATION,void * buff,unsigned len)
{
  bioSeek(sc_file,pos,RELATION);
  return bioWriteBuffer(sc_file,buff,len);
}

char * __FASTCALL__ bmName( void )
{
  return bioFileName(sc_file);
}

tBool __FASTCALL__ bmEOF( void )
{
  return bioEOF(sc_file);
}
