/*
 * Copyright (c) 2006 Jeremy Erickson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#include "filterquery.h"
#include "verse.h"
#include <qregexp.h>
#include <qstring.h>
#include <vector>
#include <iostream>
using namespace std;

namespace bmemcore
{

FilterQuery::FilterQuery(const QString& newQuery)
{
    mQuery = newQuery;
}

FilterQuery::~FilterQuery()
{}


bool FilterQuery::allows(const Verse& theVerse)
{
    return str2Bool(matches(mQuery, theVerse));
}

bool FilterQuery::patternValid()
{
    return patternValid(mQuery);
}

bool FilterQuery::patternValid(const QString& pattern)
{
    Verse emptyVerse;
    QString result = matches(pattern, emptyVerse);
    return ((result == "true") || (result == "false"));
}

QString FilterQuery::matches(QString query, const Verse& theVerse)
{
    //First we will simplify all whitespace to spaces, or add it where needed
    query.replace("(", " (");
    query.replace(")", ") ");
    QRegExp whiteExp("\\s+");
    query.replace(whiteExp, " ");
    //Don't segfault when there are dollar signs.
    QRegExp dollarExp("(\\$\\d+)");
    query.replace(dollarExp, "\"\\1\"");
    query = query.lower();
    //Then, we replace quoted material with references, to implement escapes,
    //etc.
    vector<QString> replacements;
    int firstQuote = query.find('"');
    int currReplacement = 0;
    while (firstQuote != -1){
        bool stop = false;
        int currIndex = firstQuote;
        bool lastBack = false;
        QString newString = "";
        while (!stop && currIndex < query.length()){
            currIndex++;
            const QChar& currChar = query[currIndex];
            if (lastBack){
                newString.append(currChar);
                lastBack = false;
            }
            else{
                if (currChar == '\\'){
                    lastBack = true;
                }
                else if (currChar == '"'){
                    stop = true;
                }
                else{
                    newString.append(currChar);
                }
            }
        }
        query.replace(firstQuote, currIndex - firstQuote + 1, "$" +
                        QString::number(currReplacement++));
        replacements.push_back(newString);
        firstQuote = query.find('"');
    }
    //Next we parse the actual queries
    QRegExp standardQuery("\\b"
        "(reference|book|chapter|verse|verses|translation|text)"
        " (contains|matches|equals|startswith|starts with|endswith|ends with) " 
        "(\\$\\d+|\\w+)", false);
    while (standardQuery.search(query) != -1)
    {
        QString value;
        QString sought = standardQuery.cap(1).lower();
        if (sought == "reference")
        {
            value = theVerse.getReference();
        }
        else if (sought == "book")
        {
            value = theVerse.getBook();
        }
        else if (sought == "chapter")
        {
            value = theVerse.getChapter();
        }
        else if (sought == "verse" || sought == "verses")
        {
            value = theVerse.getVerses();
        }
        else if (sought == "translation")
        {
            value = theVerse.getTranslation();
        }
        else if (sought == "text")
        {
            value = theVerse.getText();
        }
        value = value.simplifyWhiteSpace().lower();
        bool result;
        sought = standardQuery.cap(2).lower();
        QString desiredVal = standardQuery.cap(3);
        //This situation will be reached if we had a quoted string.
        if (desiredVal[0] == '$'){
            desiredVal.remove(0, 1);
            desiredVal = replacements[desiredVal.toInt()];
        }
        //Old code for when we had regexps with different quoting schemes.
        //int currentCap = 3;
        //while (desiredVal.isEmpty() && currentCap <= 
        //        standardQuery.numCaptures())
        //{
        //    desiredVal = standardQuery.cap(currentCap).lower();
        //    currentCap++;
        //}
        if (sought == "contains")
        {
            result = value.contains(desiredVal, false);
        }
        else if (sought == "matches" || sought == "equals")
        {
            result = (value == desiredVal);
        }
        else if (sought == "startswith" || sought == "starts with")
        {
            result = value.startsWith(desiredVal);
        }
        else if (sought == "endswith" || sought == "ends with")
        {
            result = value.endsWith(desiredVal);
        }
        query.replace(standardQuery.cap(0), bool2Str(result));
    }
    QRegExp catQuery("\\bcategories contains "
        "(\\$\\d+|\\w+)", false);
    while (catQuery.search(query) != -1)
    {
        QString desiredVal = catQuery.cap(1);
        if (desiredVal[0] == '$'){
            desiredVal.remove(0, 1);
            desiredVal = replacements[desiredVal.toInt()];
        }
        //int currentCap = 1;
        //while (desiredVal.isEmpty() && currentCap <= catQuery.numCaptures())
        //{
        //    desiredVal = catQuery.cap(currentCap).lower();
        //    currentCap++;
        //}
        QStringList categories = theVerse.getCategories();
        //QStringList::const_iterator cIt =
        //    std::find(categories.begin(),
        //              categories.end(), desiredVal);
        bool success = false;
        QStringList::const_iterator cIt = categories.begin();
        while (!success && cIt != categories.end())
        {
            if ((*cIt).lower().simplifyWhiteSpace() == desiredVal)
            {
                success = true;
            }
            cIt++;
        }
        query.replace(catQuery.cap(0), bool2Str(success));
    }
    //Then, we do everything in parentheses recursively.
    QRegExp parensExp("\\(([^)]*)\\)");
    int parenLoc = parensExp.search(query);
    while (parenLoc != -1)
    {
        int parenDepth = 1;
        int stop = parenLoc + 1;
        while (stop < query.length() && parenDepth > 0)
        {
            if (query[stop] == '(')
            {
                parenDepth++;
            }
            else if (query[stop] == ')')
            {
                parenDepth--;
            }
            stop++;
        }
        int subLen = stop - parenLoc;
        QString subQuery = query.mid(parenLoc + 1, subLen - 2);
        QString parsedVal = matches(subQuery, theVerse);
        query.replace(parenLoc, subLen, parsedVal);
        parenLoc = parensExp.search(query);
    }
    //Then, perform "not" operations.
    QRegExp notTrue("\\bnot true\\b", false);
    query.replace(notTrue, "false");
    QRegExp notFalse("\\bnot false\\b", false);
    query.replace(notFalse, "true");
    //Afterwards, "and" operations.
    QRegExp andExp("\\b(true|false) and (true|false)\\b", false);
    while (andExp.search(query) != -1)
    {
        query.replace(andExp.cap(0),
            bool2Str(str2Bool(andExp.cap(1)) && str2Bool(andExp.cap(2))));
    }
    //Finally, "or" operations.
    QRegExp orExp("\\b(true|false) or (true|false)\\b", false);
    while (orExp.search(query) != -1)
    {
        query.replace(orExp.cap(0),
            bool2Str(str2Bool(orExp.cap(1)) || str2Bool(orExp.cap(2))));
    }
    return query.stripWhiteSpace();
}

bool FilterQuery::str2Bool(const QString& val)
{
    if (val.lower() == "true"){
        return true;
    }
    else{
        return false;
    }
}

QString FilterQuery::bool2Str(bool val)
{
    if (val){
        return "true";
    }
    else{
        return "false";
    }
}

}

