/*
 ** Copyright (C) 2003-2006 Teus Benschop.
 **  
 ** This program is free software; you can redistribute it and/or modify
 ** it under the terms of the GNU General Public License as published by
 ** the Free Software Foundation; either version 2 of the License, or
 ** (at your option) any later version.
 **  
 ** This program is distributed in the hope that it will be useful,
 ** but WITHOUT ANY WARRANTY; without even the implied warranty of
 ** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 ** GNU General Public License for more details.
 **  
 ** You should have received a copy of the GNU General Public License
 ** along with this program; if not, write to the Free Software
 ** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 **  
 */


#include "libraries.h"
#include "directories.h"
#include <libgen.h>
#include <glib.h>
#include <config.h>
#include "constants.h"
#include "utilities.h"
#include "gwrappers.h"
#include "shell.h"


void directories_check_structure ()
{
  ustring command;
  // Check and/or create the root directory of all data.
  create_directory (directories_get_root ());
  // Check and/or create the projects directory, and so on.
  create_directory (directories_get_projects ());
  create_directory (directories_get_notes ());
  create_directory (directories_get_stylesheets ());
  create_directory (directories_get_configuration ());
  ustring temp;
  temp = directories_get_temp();
  create_directory (temp);
  // There has been a problem that a user ran bibledit as root, straight after
  // installation. This resulted in the creation of the temp bibledit directory
  // with limited access rights. Next time this user ran bibledit under his own
  // name with the result that bibledit could no longer access this directory.
  // Here the problem is solved: make this directory world readable/writable.
  command = "chmod ugoa+rw -R" + shell_quote_space (temp);
  system (command.c_str());
  // Create temporal directory.
  create_directory (directories_get_temp ());
}


ustring directories_get_root ()
/*
Returns the root directory of all data.
If a file ~/.bibledit is found, it is supposed to contain the 
path of the directory that is the root.
If this file is not found, which is the common situation, then ~/.bibledit
is considered to be the directory which is the root.
*/
{
  // Static variables to use.
  static bool root_directory_initialized = false;
  static ustring root_directory;  

  if (!root_directory_initialized) {
    // Default root folder.
    ustring default_root;
    default_root = gw_build_filename (g_get_home_dir(), ".bibledit");
    ustring new_root_pointer_file;
    new_root_pointer_file = gw_build_filename (g_get_home_dir(), ".bibledit-datafolder-pointer");
    if (g_file_test (new_root_pointer_file.c_str(), G_FILE_TEST_IS_REGULAR)) {
      // File exists: read the root directory it contains.
      gchar *contents;
      g_file_get_contents (new_root_pointer_file.c_str(), &contents, NULL, NULL);
      root_directory = contents;
      g_free (contents);
      root_directory = trim (root_directory);
      cout << "Using non-standard datafolder " << root_directory << endl;
      // If it contains nothing, remove the file and proceed with defaults.
      if (root_directory.empty()) {
        root_directory = default_root;
        cout << "This data folder has no name. Resetting to " << default_root << endl;
      }
      // If the new directory is not accessible, remove the file, and get defaults.
      int result;
      result = access (root_directory.c_str(), W_OK);
      if (result != 0) {
        root_directory = default_root;
        cout << "There are not enough access permissions for this data folder. Resetting to "<< default_root << endl;
      }
    } else {
      // Ok, default situation.
      root_directory = default_root;
    }
    root_directory_initialized = true;
  }

  // Return the root of all data, usually <home>/.bibledit
  return root_directory;
}


ustring directories_get_projects ()
{
  // This returns the directory with all the projects.
  return gw_build_filename (directories_get_root (), "projects");
}


ustring directories_get_notes ()
{
  // This returns the directory with the notes
  return gw_build_filename (directories_get_root (), "notes");
}


ustring directories_get_stylesheets ()
{
  // This returns the directory with the stylesheets
  return gw_build_filename (directories_get_root (), "stylesheets");
}


ustring directories_get_configuration ()
{
  // This returns the directory with the configuration
  return gw_build_filename (directories_get_root (), "configuration");
}


ustring directories_get_temp ()
{
  // Returns the temporal directory bibledit uses
  return gw_build_filename (g_get_home_dir(), ".bibledit_temp");
}
