/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef REACTOR_H_
#define REACTOR_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include "FlagOps.h"
#include "IntrusivePtr.h"
#include <ace/config-lite.h>
#include <sigc++/sigc++.h>
#include <exception>

class TimeDelta;
class EventHandlerBase;
namespace ReactorHelpers {
	class HandlerRepository;
	class TimerQueue;
};


class ReactorHandlerId
{
	friend class ReactorHelpers::HandlerRepository;
public:
	ReactorHandlerId();
	
	ReactorHandlerId(ReactorHandlerId const& other);
	
	~ReactorHandlerId();
	
	ReactorHandlerId& operator=(ReactorHandlerId const& other);
	
	bool operator==(ReactorHandlerId const& other) const;
	
	bool operator!=(ReactorHandlerId const& other) const { return !(*this == other); }
	
	operator bool() const { return ReactorHandlerId() != *this; }
private:
	template<typename Rep>
	explicit ReactorHandlerId(Rep const& rep) { new(&opaque) Rep(rep); }
	
	template<typename Rep>
	Rep& rep() { return *reinterpret_cast<Rep*>(&opaque); }
	
	template<typename Rep>
	Rep const& rep() const { return *reinterpret_cast<Rep const*>(&opaque); }
	
	union {
		void* storage[3];
		char  opaque;
	};
};


class ReactorTimerId
{
	friend class ReactorHelpers::TimerQueue;
public:
	ReactorTimerId();
	
	ReactorTimerId(ReactorTimerId const& other);
	
	~ReactorTimerId();
	
	ReactorTimerId& operator=(ReactorTimerId const& other);
	
	bool operator==(ReactorTimerId const& other) const;
	
	bool operator!=(ReactorTimerId const& other) const { return !(*this == other); }
	
	operator bool() const { return ReactorTimerId() != *this; }
private:
	template<typename Rep>
	explicit ReactorTimerId(Rep const& rep) { new(&opaque) Rep(rep); }
	
	template<typename Rep>
	Rep& rep() { return *reinterpret_cast<Rep*>(&opaque); }
	
	template<typename Rep>
	Rep const& rep() const { return *reinterpret_cast<Rep const*>(&opaque); }
	
	union {
		void* storage[3];
		char  opaque;
	};
};


class Reactor
{
	DECLARE_NON_COPYABLE(Reactor)
public:
	enum Status { SUCCESS, FAILURE, INTERRUPTED, STOPPED };
	enum IOEvents {
		NO_EVENTS = 0, READ = 1, WRITE = 2, EXCEPT = 4,
		ALL_EVENTS = READ | WRITE | EXCEPT
	};
	typedef IntrusivePtr<EventHandlerBase> EventHandlerPtr;
	
	
	class Exception : public std::exception
	{
	public:
		Exception(char const* str) : m_str(str) {}
		
		virtual char const* what() const throw() { return m_str; }
	private:
		char const* m_str;
	};
	
	
	class CapacityException : public Exception
	{
	public:
		CapacityException() : Exception("reactor capacity exceeded") {}
	};

	
	class DuplicateHandleException : public Exception
	{
	public:
		DuplicateHandleException() : Exception("duplicate handle in reactor") {}
	};
	
	
	Reactor() {}
	
	virtual ~Reactor();
	
	virtual ReactorHandlerId findHandler(ACE_HANDLE handle) const = 0;
	
	virtual ReactorHandlerId registerHandler(
	    ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events) = 0;
	
	virtual void unregisterHandler(ReactorHandlerId const& id) = 0;
	
	virtual void unregisterAllHandlers() = 0;
	
	virtual void enableEvents(ReactorHandlerId const& id, IOEvents events) = 0;
	
	virtual void disableEvents(ReactorHandlerId const& id, IOEvents events) = 0;
	
	virtual void setEvents(ReactorHandlerId const& id, IOEvents events) = 0;
	
	virtual IOEvents getEvents(ReactorHandlerId const& id) const = 0;
	
	virtual ReactorTimerId registerTimer(
		EventHandlerPtr const& handler, TimeDelta const* timeout = 0) = 0;
	
	virtual void rescheduleTimer(
		ReactorTimerId const& id, TimeDelta const* timeout = 0) = 0;
	
	virtual void unregisterTimer(ReactorTimerId const& id) = 0;
	
	virtual void unregisterAllTimers() = 0;
	
	virtual TimeDelta getRemainingTime(ReactorTimerId const& timer_id) const = 0;
	
	virtual Status handleEvents() = 0;
	
	virtual Status runEventLoop() = 0;
	
	virtual void wakeup() = 0;
	
	virtual void stop() = 0;
	
	virtual void restart() = 0;
	
	virtual sigc::signal<void>& beforeSleepSignal() = 0;
};


DEFINE_FLAG_OPS(Reactor::IOEvents)

#endif
