/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DebugWindow.h"
#include "DebugLayout.h"
#include "DebugMessage.h"
#include "DebugConnection.h"
#include "AutoScrollingWindow.h"
#include "ClientRequestDescriptor.h"
#include "IntrusivePtr.h"
#include "CompiledImages.h"
#include "ColorRectangle.h"
#include <gtkmm/label.h>
#include <gtkmm/alignment.h>
#include <gtkmm/frame.h>
#include <gtkmm/fixed.h>
#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/treeiter.h>
#include <gtkmm/treepath.h>
#include <gtkmm/liststore.h>
#include <gtkmm/treeview.h>
#include <glibmm/ustring.h>
#include <glibmm/convert.h>
#include <cassert>

using namespace std;

class DebugWindow::RequestListModel : public Gtk::ListStore
{
	struct RequestColumns;
public:
	typedef ClientRequestDescriptor::SelectionChangeFunctor SelectionChangeFunctor;
	
	static Glib::RefPtr<RequestListModel> create();
	
	RequestColumns const& getCols() const { return m_columns; }
	
	Gtk::TreeIter appendRequest(Glib::ustring const& url,
		SelectionChangeFunctor const& sel_change_functor);
	
	void onSelectionChange(Gtk::TreeIter const& iter, bool selected);
protected:
	RequestListModel();
	
	virtual ~RequestListModel();
private:
	struct RequestColumns : public Gtk::TreeModelColumnRecord
	{
		Gtk::TreeModelColumn<Glib::ustring> URL;
		Gtk::TreeModelColumn<SelectionChangeFunctor> SEL_CHANGE_FUNCTOR;
		RequestColumns() { add(URL); add(SEL_CHANGE_FUNCTOR); }
	};
	
	RequestColumns m_columns;
};


class DebugWindow::Legend : public Gtk::Frame
{
public:
	Legend();
	
	virtual ~Legend();
private:
	class ConnectionSymbol;
	class HttpMessageSymbol;
	enum { SYMBOL_WIDTH = 12, SYMBOL_HEIGHT = 16 };
	
	void addClientConnectionItem();
	
	void addHttpMessageItem(
		Glib::ustring const& text, HttpMessageType type);
	
	void addDoubleclickNote();
	
	Gtk::Table m_table;
	int m_currentRow;
};


class DebugWindow::Legend::ConnectionSymbol : public Gtk::Fixed
{
public:
	ConnectionSymbol();
	
	virtual ~ConnectionSymbol();
private:
	ColorRectangle m_borders[4];
};


class DebugWindow::Legend::HttpMessageSymbol : public ColorRectangle
{
public:
	HttpMessageSymbol(HttpMessageType type);
	
	virtual ~HttpMessageSymbol();
};


/*============================== DebugWindow =============================*/

DebugWindow::DebugWindow()
:	m_connectionsAdj(20, 1, 99, 1, 10),
	m_trafficLimitAdj(200, 100, 9999, 100, 1000),
	m_connectionsSB(m_connectionsAdj),
	m_trafficLimitSB(m_trafficLimitAdj),
	m_ptrRequestListModel(RequestListModel::create()),
	m_requestList(m_ptrRequestListModel),
	m_lastSelection(),
	m_table1(2, 2),
	m_table1Alignment(0.5, 0.5, 0.0, 0.0),
	m_legendAlignment(0.5, 0.0, 1.0, 0.0)
{
	set_title("Debug");
	set_icon(CompiledImages::window_icon_png.getPixbuf());
	add(m_vbox1);
	
	m_vbox1.pack_start(m_hbox1);
	m_hbox1.pack_start(m_vbox2, Gtk::PACK_SHRINK);
	m_vbox2.pack_start(m_table1Alignment, Gtk::PACK_SHRINK);
	
	m_table1Alignment.add(m_table1);
	m_table1.attach(
		*manage(new Gtk::Label("Connections: ", Gtk::ALIGN_RIGHT)),
		0, 1, 0, 1 // left, right, top, bottom
	);
	m_table1.attach(m_connectionsSB, 1, 2, 0, 1);
	m_table1.attach(
		*manage(new Gtk::Label("Traffic, KB: ", Gtk::ALIGN_RIGHT)),
		0, 1, 1, 2
	);
	m_table1.attach(m_trafficLimitSB, 1, 2, 1, 2);
	
	m_vbox2.pack_start(m_legendAlignment, Gtk::PACK_SHRINK);
	m_legendAlignment.set_border_width(2);
	m_legendAlignment.add(*manage(new Legend));
	
	m_hbox1.pack_start(m_layoutScrollWnd);
	m_layoutScrollWnd.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	m_layoutScrollWnd.set_size_request(400, 350);
	m_layoutScrollWnd.add(m_layout);
	
	m_vbox1.pack_start(m_requestListScrollWnd, Gtk::PACK_SHRINK);
	m_requestListScrollWnd.set_policy(Gtk::POLICY_AUTOMATIC, Gtk::POLICY_AUTOMATIC);
	m_requestListScrollWnd.set_shadow_type(Gtk::SHADOW_IN);
	m_requestListScrollWnd.set_size_request(550, 120);
	m_requestListScrollWnd.add(m_requestList);
	
	m_requestList.append_column(Glib::ustring(), m_ptrRequestListModel->getCols().URL);
	m_requestList.set_headers_visible(false);
	m_requestList.get_selection()->signal_changed().connect(
		sigc::mem_fun(*this, &DebugWindow::onRequestSelectionChange)
	);
	
	m_connectionsAdj.signal_value_changed().connect(
		sigc::mem_fun(*this, &DebugWindow::onTrackConnectionsChange)
	);
	m_trafficLimitAdj.signal_value_changed().connect(
		sigc::mem_fun(*this, &DebugWindow::onTrafficLimitChange)
	);
	onTrackConnectionsChange();
	onTrafficLimitChange();
	
	m_requestList.grab_focus();
	show_all_children();
}

DebugWindow::~DebugWindow()
{
	// we don't want slots to be called during this object's destruction
	sigc::trackable::notify_callbacks();
}

void
DebugWindow::clientConnectionBegin(ACE_thread_t thread_id)
{
	m_layout.clientConnectionBegin(thread_id);
}

void
DebugWindow::clientConnectionEnd(ACE_thread_t thread_id)
{
	m_layout.clientConnectionEnd(thread_id);
}

void
DebugWindow::registerClientRequest(ACE_thread_t thread_id,
	int request_id, std::string request_url)
{
	DebugConnection* conn = m_layout.findOpenConnection(thread_id);
	if (!conn) {
		return;
	}
	
	Glib::ustring url;
	try {
		url = Glib::convert(request_url, "utf-8", "iso-8859-1");
	} catch (Glib::ConvertError&) {}
	
	IntrusivePtr<ClientRequestDescriptor> desc(
		new ClientRequestDescriptor(
			url,
			sigc::mem_fun(
				*m_ptrRequestListModel.operator->(),
				&RequestListModel::appendRequest
			),
			sigc::mem_fun(
				*this,
				&DebugWindow::removeRequestFromList
			),
			sigc::mem_fun(
				*this,
				&DebugWindow::selectRequestFromList
			)
		)
	);
	conn->addClientRequestDescriptor(request_id, desc);
}

void
DebugWindow::httpMessageBegin(
	ACE_thread_t thread_id, int request_id,
	HttpMessageType type, std::string const& headers)
{
	m_layout.httpMessageBegin(thread_id, request_id, type, headers);
}

void
DebugWindow::httpMessageEnd(ACE_thread_t thread_id,
	HttpMessageType type, bool error)
{
	m_layout.httpMessageEnd(thread_id, type, error);
}

void
DebugWindow::logMessage(ACE_thread_t thread_id, std::string const& msg)
{
	m_layout.logMessage(thread_id, msg);
}

void
DebugWindow::logTraffic(ACE_thread_t thread_id,
	SplittableBuffer& traf, TrafficDirection dir)
{
	m_layout.logTraffic(thread_id, traf, dir);
}

void
DebugWindow::onTrackConnectionsChange()
{
	m_layout.setConnectionsToTrack(int(m_connectionsAdj.get_value()));
}

void
DebugWindow::onTrafficLimitChange()
{
	m_layout.setTrafficLimit(int(m_trafficLimitAdj.get_value())*1024);
}

void
DebugWindow::onRequestSelectionChange()
{
	Gtk::TreeIter new_selection = m_requestList.get_selection()->get_selected();
	if (listIteratorsEqual(new_selection, m_lastSelection)) {
		return;
	}
	
	if (m_lastSelection) {
		//assert(m_ptrRequestListModel->iter_is_valid(m_lastSelection));
		m_ptrRequestListModel->onSelectionChange(m_lastSelection, false);
	}
	if (new_selection) {
		//assert(m_ptrRequestListModel->iter_is_valid(new_selection));
		m_ptrRequestListModel->onSelectionChange(new_selection, true);
	}
	
	m_lastSelection = new_selection;
}

void
DebugWindow::removeRequestFromList(Gtk::TreeIter const& iter)
{
	if (listIteratorsEqual(iter, m_lastSelection)) {
		m_lastSelection = Gtk::TreeIter();
	}
	//assert(m_ptrRequestListModel->iter_is_valid(iter));
	m_ptrRequestListModel->erase(iter);
}

void
DebugWindow::selectRequestFromList(Gtk::TreeIter const& iter)
{
	//assert(m_ptrRequestListModel->iter_is_valid(iter));
	m_requestList.set_cursor(Gtk::TreePath(iter));
	m_requestList.grab_focus();
}

bool
DebugWindow::listIteratorsEqual(
	Gtk::TreeIter const& one, Gtk::TreeIter const& other)
{
	// Comparing a default constructed iterator with a valid
	// one results in abort(). Here we do the safe way.
	if (!one) {
		return !other;
	} else if (!other) {
		return false;
	} else {
		return one == other;
	}
}

/*================== DebugWindow::RequestListModel ======================*/

DebugWindow::RequestListModel::RequestListModel()
{
	set_column_types(m_columns);
}

DebugWindow::RequestListModel::~RequestListModel()
{
}

Glib::RefPtr<DebugWindow::RequestListModel>
DebugWindow::RequestListModel::create()
{
	return Glib::RefPtr<DebugWindow::RequestListModel>(new RequestListModel);
}

Gtk::TreeIter
DebugWindow::RequestListModel::appendRequest(
	Glib::ustring const& url,
	SelectionChangeFunctor const& sel_change_functor)
{
	Gtk::TreeIter iter = append();
	Gtk::TreeRow row = *iter;
	row[m_columns.URL] = url;
	row[m_columns.SEL_CHANGE_FUNCTOR] = sel_change_functor;
	return iter;
}

void
DebugWindow::RequestListModel::onSelectionChange(
	Gtk::TreeIter const& iter, bool selected)
{
	SelectionChangeFunctor ftor = (*iter)[m_columns.SEL_CHANGE_FUNCTOR];
	ftor(selected);
}


/*===============-======= DebugWindow::Legend ============================*/

DebugWindow::Legend::Legend()
:	Gtk::Frame("Legend"),
	m_table(AbstractDebugAgent::NUM_HTTP_MESSAGE_TYPES + 2, 2),
	m_currentRow(0)
{
	typedef AbstractDebugAgent ADA;
	
	set_label_align(Gtk::ALIGN_CENTER);
	add(m_table);
	m_table.set_border_width(3);
	m_table.set_col_spacings(3);
	m_table.set_row_spacings(4);
	
	addClientConnectionItem();
	addHttpMessageItem("Incoming request", ADA::INCOMING_REQUEST);
	addHttpMessageItem("Outgoing request", ADA::OUTGOING_REQUEST);
	addHttpMessageItem("Incoming response", ADA::INCOMING_RESPONSE);
	addHttpMessageItem("Script request", ADA::SCRIPT_REQUEST);
	addHttpMessageItem("Script response", ADA::SCRIPT_RESPONSE);
	addHttpMessageItem("Unmodified outgoing\nresponse", ADA::UNMODIFIED_OUTGOING_RESPONSE);
	addHttpMessageItem("Filtered outgoing\nresponse", ADA::FILTERED_OUTGOING_RESPONSE);
	addHttpMessageItem("Ad replacement", ADA::CRAFTED_OUTGOING_RESPONSE);
	addHttpMessageItem("Error response", ADA::ERROR_OUTGOING_RESPONSE);
	addDoubleclickNote();
}

DebugWindow::Legend::~Legend()
{
}

void
DebugWindow::Legend::addClientConnectionItem()
{
	m_table.attach(
		*manage(new ConnectionSymbol),
		0, 1, m_currentRow, m_currentRow + 1,
		Gtk::SHRINK
	);
	m_table.attach(
		*manage(new Gtk::Label("Client connection", Gtk::ALIGN_LEFT)),
		1, 2, m_currentRow, m_currentRow + 1
	);
	++m_currentRow;
}

void
DebugWindow::Legend::addHttpMessageItem(
	Glib::ustring const& text, HttpMessageType type)
{
	m_table.attach(
		*manage(new HttpMessageSymbol(type)),
		0, 1, m_currentRow, m_currentRow + 1,
		Gtk::SHRINK
	);
	m_table.attach(
		*manage(new Gtk::Label(text, Gtk::ALIGN_LEFT)),
		1, 2, m_currentRow, m_currentRow + 1
	);
	++m_currentRow;
}

void
DebugWindow::Legend::addDoubleclickNote()
{
	Glib::ustring text("Doubleclick any object\nfor more info");
	m_table.attach(
		*manage(new Gtk::Label(text, Gtk::ALIGN_LEFT)),
		0, 2, m_currentRow, m_currentRow + 1
	);
	++m_currentRow;
}


/*================ DebugWindow::Legend::ConnectionSymbol ==================*/

DebugWindow::Legend::ConnectionSymbol::ConnectionSymbol()
{
	typedef DebugLayout DL;
	
	set_size_request(SYMBOL_WIDTH, SYMBOL_HEIGHT);
	
	Gdk::Color color("#ac40ef");
	for (int i = 0; i < 4; ++i) {
		m_borders[i].setColor(color);
	}
	for (int i = 0; i < 2; ++i) {
		m_borders[i].set_size_request(DL::CONN_BORDER, SYMBOL_HEIGHT);
	}
	for (int i = 2; i < 4; ++i) {
		m_borders[i].set_size_request(
			SYMBOL_WIDTH - DL::CONN_BORDER*2, DL::CONN_BORDER
		);
	}

	put(m_borders[0], 0, 0);
	put(m_borders[1], SYMBOL_WIDTH - DL::CONN_BORDER, 0);
	put(m_borders[2], DL::CONN_BORDER, 0);
	put(m_borders[3], DL::CONN_BORDER, SYMBOL_HEIGHT - DL::CONN_BORDER); 
}

DebugWindow::Legend::ConnectionSymbol::~ConnectionSymbol()
{
}


/*=============== DebugWindow::Legend::HttpMessageSymbol ==================*/

DebugWindow::Legend::HttpMessageSymbol::HttpMessageSymbol(
	HttpMessageType type)
{
	setColor(DebugMessage::getColorFor(type));
	set_size_request(SYMBOL_WIDTH, SYMBOL_HEIGHT);
}

DebugWindow::Legend::HttpMessageSymbol::~HttpMessageSymbol()
{
}
