/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "ServerConnectionPool.h"
#include "ServerConnection.h"
#include "IntrusivePtr.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include "SymbolicInetAddr.h"
#include "TimeStamp.h"
#include "TimeDelta.h"
#include "Reactor.h"
#include "EventHandler.h"
#include <ace/config-lite.h>
#include <ace/Synch.h>
#include <ace/Singleton.h>
#include <ace/OS_NS_sys_time.h>
#include <ace/OS_NS_sys_socket.h>
#include <boost/multi_index_container.hpp>
#include <boost/multi_index/member.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/multi_index/composite_key.hpp>
#include <iterator>
#include <utility>
#include <cassert>

using namespace std;
using namespace boost::multi_index;

class ServerConnectionPool::ConnRC :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >,
	public std::auto_ptr<ServerConnection>
{
public:
	ConnRC(std::auto_ptr<ServerConnection> conn)
	: std::auto_ptr<ServerConnection>(conn) {}
};


struct ServerConnectionPool::Entry
{	
	Entry(SymbolicInetAddr const& addr,
		TimeStamp const& timeout,
		IntrusivePtr<ConnRC> const& connection)
	:	addr(addr),
		timeout(timeout),
		connection(connection)
	{
	}
	
	SymbolicInetAddr addr;
	TimeStamp timeout; // absolute time value
	IntrusivePtr<ConnRC> connection;
};


class ServerConnectionPool::Impl :
	public ServerConnectionPool,
	private EventHandlerBase
{
public:
	Impl();
	
	virtual ~Impl();
	
	virtual void storeConnection(std::auto_ptr<ServerConnection> conn);
	
	virtual std::auto_ptr<ServerConnection>
		retrieveConnection(SymbolicInetAddr const& addr);
	
	virtual void connectToReactor(Reactor& reactor);
	
	virtual void disconnectFromReactor();
private:
	class AddrTag {};
	class SequenceTag {};
	
	typedef ACE_Recursive_Thread_Mutex Mutex;
	typedef multi_index_container<
		Entry,
		indexed_by<
			ordered_non_unique<
				tag<AddrTag>,
				composite_key<
					Entry,
					member<Entry, SymbolicInetAddr, &Entry::addr>,
					member<Entry, TimeStamp, &Entry::timeout>
				>
			>,
			sequenced<
				tag<SequenceTag>
			>
		>
	> Container;
	typedef Container::index<AddrTag>::type AddrIdx;
	typedef Container::index<SequenceTag>::type SequenceIdx;
	
	virtual void handleTimeout(ReactorTimerId const&);
	
	virtual void ref();
	
	virtual void unref();
	
	void removeTimedOut();
	
	void unregisterCleanupTimer();
	
	void refreshCleanupTimer();
	
	bool isConnAlive(ServerConnection& conn);
	
	Mutex m_mutex;
	Container m_container;
	Reactor* m_pReactor;
	ReactorTimerId m_cleanupTimerId;
	int m_refCount;
	bool m_isCleanupScheduled;
};



ServerConnectionPool*
ServerConnectionPool::instance()
{
	return ACE_Singleton<Impl, ACE_Recursive_Thread_Mutex>::instance();
}


/*======================== ServerConnectionPool::Impl =====================*/

ServerConnectionPool::Impl::Impl()
:	m_pReactor(0),
	m_refCount(0),
	m_isCleanupScheduled(false)
{
}

ServerConnectionPool::Impl::~Impl()
{
}

void
ServerConnectionPool::Impl::storeConnection(auto_ptr<ServerConnection> conn)
{
	assert(conn.get());
	
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	removeTimedOut();
	
	TimeStamp timeout = TimeStamp::fromTimeval(ACE_OS::gettimeofday());
	timeout += TimeDelta::fromSec(TIMEOUT);
	
	IntrusivePtr<ConnRC> conn_ptr(new ConnRC(conn)); // conn is now null
	Entry entry((*conn_ptr)->getAddress(), timeout, conn_ptr);
	
	if (m_container.size() < CAPACITY) {
		// inserting a new entry
		AddrIdx& addr_idx = m_container.get<AddrTag>();
		addr_idx.insert(entry);
	} else {
		// replacing the first entry in sequence (the oldest one)
		SequenceIdx& seq_idx = m_container.get<SequenceTag>();
		SequenceIdx::iterator seq_pos(seq_idx.begin());
		seq_idx.replace(seq_pos, entry);
		seq_idx.relocate(seq_idx.end(), seq_pos);
	}
	
	refreshCleanupTimer();
}

std::auto_ptr<ServerConnection>
ServerConnectionPool::Impl::retrieveConnection(SymbolicInetAddr const& addr)
{
	auto_ptr<ServerConnection> res;
	
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, res);
	
	removeTimedOut();
	
	AddrIdx& idx = m_container.get<AddrTag>();
	while (true) {
		AddrIdx::iterator it = idx.lower_bound(boost::make_tuple(addr));
		if (it != idx.end() && it->addr == addr) {
			IntrusivePtr<ConnRC> conn_ptr(it->connection);
			idx.erase(it);
			if (isConnAlive(**conn_ptr)) {
				res = *conn_ptr;
			} else {
				continue;
			}
		}
		break;
	}
	
	refreshCleanupTimer();
	
	return res;
}

void
ServerConnectionPool::Impl::connectToReactor(Reactor& reactor)
{
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	unregisterCleanupTimer();
	
	TimeDelta timeout(TimeDelta::fromSec(CLEANUP_TIMEOUT));
	m_pReactor = &reactor;
	m_cleanupTimerId = m_pReactor->registerTimer(
		Reactor::EventHandlerPtr(this)
	);
	refreshCleanupTimer();
}

void
ServerConnectionPool::Impl::disconnectFromReactor()
{
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	unregisterCleanupTimer();
}

void
ServerConnectionPool::Impl::handleTimeout(ReactorTimerId const&)
{
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	m_isCleanupScheduled = false;
	removeTimedOut();
	refreshCleanupTimer();
}

void
ServerConnectionPool::Impl::ref()
{
	// we are protected by the caller's guard
	
	++m_refCount;
}

// This one may be called both from inside and outside of this class.
void
ServerConnectionPool::Impl::unref()
{
	ACE_GUARD_RETURN(Mutex, guard, m_mutex, );
	
	if (!--m_refCount) {
		m_cleanupTimerId = ReactorTimerId();
		m_pReactor = 0;
		m_isCleanupScheduled = false;
	}
}

void
ServerConnectionPool::Impl::removeTimedOut()
{
	// we are protected by the caller's guard
	
	if (m_container.empty()) {
		return;
	}
	
	TimeStamp now = TimeStamp::fromTimeval(ACE_OS::gettimeofday());
	SequenceIdx& idx = m_container.get<SequenceTag>();
	
	if (now < idx.front().timeout - TimeDelta::fromSec(TIMEOUT)) {
		// Time went back?
		m_container.clear();
		return;
	}
	
	while (!idx.empty()) {
		SequenceIdx::iterator it(idx.begin());
		if (it->timeout > now) {
			break;
		}
		idx.erase(it);
	}
}

void
ServerConnectionPool::Impl::unregisterCleanupTimer()
{
	// we are protected by the caller's guard
	
	if (m_pReactor) {
		assert(m_cleanupTimerId);
		m_pReactor->unregisterTimer(m_cleanupTimerId);
		// unregisterTimer() will cause unref() to be called
	}
	assert(m_refCount == 0);
	assert(!m_pReactor);
	assert(!m_isCleanupScheduled);
}

void
ServerConnectionPool::Impl::refreshCleanupTimer()
{
	// we are protected by the caller's guard
	
	if (!m_pReactor) {
		return;
	}
	
	assert(m_cleanupTimerId);
	
	if (m_container.empty()) {
		if (m_isCleanupScheduled) {
			m_pReactor->rescheduleTimer(m_cleanupTimerId);
			m_isCleanupScheduled = false;
		}
	} else {
		if (!m_isCleanupScheduled) {
			TimeDelta timeout(TimeDelta::fromSec(CLEANUP_TIMEOUT));
			m_pReactor->rescheduleTimer(m_cleanupTimerId, &timeout);
			m_isCleanupScheduled = true;
		}
	}
}

bool
ServerConnectionPool::Impl::isConnAlive(ServerConnection& conn)
{
	// I don't see a problem with changing handle mode to nonblocking.
	// After all, AsyncConnector also returns non-blocking sockets.
	if (conn.peer().enable(ACE_NONBLOCK) == -1) {
		return false;
	}
	
	char buf[1];
	ssize_t r = ACE_OS::recv(conn.peer().get_handle(), buf, 0);
	// recv with zero length is legal and faster than MSG_PEEK
	
	return (r == -1 && errno == EWOULDBLOCK);
}
