/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPSTREAMWRITER_H_
#define HTTPSTREAMWRITER_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "types.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "NonCopyable.h"
#include <stddef.h>
#include <memory>

class AbstractDataConsumer;
class HttpRequestMetadata;
class HttpResponseMetadata;
class HttpMessageMetadata;
class HttpRequestLine;
class HttpStatusLine;
class HttpHeadersCollection;
class HttpVersion;
class URI;

class HttpStreamWriter
{
	DECLARE_NON_COPYABLE(HttpStreamWriter)
public:
	enum Connection { CONN_KEEP_ALIVE, CONN_CLOSE };
	enum HttpProxy { NO_HTTP_PROXY, THROUGH_HTTP_PROXY };
	enum Chunked { CHUNKED_DONT_USE, CHUNKED_OK };
	
	HttpStreamWriter(AbstractDataConsumer& consumer);
	
	void startRequest(HttpRequestMetadata& metadata,
		Connection conn, HttpProxy http_proxy, Chunked chunked);
	
	void startResponse(HttpResponseMetadata& metadata,
		HttpRequestLine const& request_line, Connection conn);
	
	void appendBodyData(SplittableBuffer& data, bool eof);
	
	void reset();
	
	Connection getConnPersistence() const { return m_connection; }
	
	size_t getBufferedBodyDataSize() const { return m_bufferedBodyPart.size(); }
private:
	enum State {
		WAITING_FOR_METADATA,
		FLAT_BODY,
		BUFFERED_FLAT_BODY,
		CHUNKED_BODY
	};
	enum { BUFFER_CHUNKS_LESS_THAN = 256 }; // can't be 0
	
	void output(bool eof=false);
	
	void processRequestLine(HttpRequestLine& request_line,
		HttpVersion const& http_ver, HttpProxy http_proxy);
	
	void processStatusLine(HttpStatusLine& status_line, HttpVersion const& http_ver);
	
	void prepareCommonHeaders(HttpHeadersCollection& headers);
	
	void prepareRequestHeaders(HttpHeadersCollection& headers, URI const& request_uri);
	
	void prepareResponseHeaders(HttpHeadersCollection& headers) {
		prepareCommonHeaders(headers);
	}
	
	void prepareForbiddenBodyMode(HttpHeadersCollection& headers,
		HttpVersion const& http_ver, Connection conn);
	
	void prepareSizedFlatMode(HttpHeadersCollection& headers,
		HttpVersion const& http_ver, Connection conn, uintmax_t body_size);
	
	void prepareUnsizedFlatMode(HttpHeadersCollection& headers,
		HttpVersion const& http_ver);
	
	void prepareBufferedFlatMode(HttpHeadersCollection& headers,
		HttpVersion const& http_ver, Connection conn);
	
	void prepareChunkedMode(HttpHeadersCollection& headers,
		HttpVersion const& http_ver, Connection conn);
	
	void handleConnection(HttpHeadersCollection& headers,
		HttpVersion const& http_ver, Connection conn);
	
	AbstractDataConsumer& m_rConsumer;
	SBOutStream m_outStream;
	SplittableBuffer m_bufferedBodyPart;
	State m_state;
	Connection m_connection;
	bool m_isEOF;
};

#endif

