/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <config.h>
 
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <fcntl.h>
#include <pwd.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>

#include <glib.h>

#include "irc.h"
#include "debug.h"

#ifndef HAVE_VSNPRINTF
extern char* g_vsprintf (const gchar *fmt, va_list *args, va_list *args2);
#endif

struct sockaddr_in server_addr;
int irc_errno;
messageData irc_message;
messageInfo irc_info;

/* messageList *message_list = NULL; */
/* messageList *ml_tail = NULL; */

int irc_init()
{
  uid_t uid;
  struct passwd *pwdent;
  char hostname[BUFFLEN];

  bs_function_enter();

  /* Get the hostname */
  if (gethostname(hostname, BUFFLEN) != 0) {
    BS_DEBUG(1, "gethostname failed");
    strcpy(hostname, "foo.com");
  }

  /* Get the UID of the current user*/
  uid = getuid();

 /* Get the password entry using the UID */
  if ((pwdent = getpwuid(uid)) == NULL) {
    BS_DEBUG(1, "getpwuid failed");
    bs_function_leave();
    return(ERR_FATAL);
  }

  bs_function_leave();
  return(0);
}

int irc_connect(char *hostname, unsigned port)
{
  struct hostent *hostptr;
  int server_sd;

  bs_function_enter();

  memset((char *) &server_addr, 0, sizeof(server_addr));

  /*  Get host address information. */
  /*  First try using hostname as a dotted-decimal notation.  */
  /*  If that fails treat it as a host name. */

  /* if ( inet_aton(hostname, &server_addr.sin_addr) == 0) { */
    if ( (hostptr = gethostbyname(hostname)) == NULL) {
      perror("irc.c irc_connect gethostbyname");
      bs_function_leave();
      return(-1);
    
    } else {
      memcpy((char *) &server_addr.sin_addr, hostptr->h_addr, hostptr->h_length);
    }
    /* } */

  server_addr.sin_family = AF_INET;
  server_addr.sin_port = htons(port);

  if ( (server_sd = socket(AF_INET, SOCK_STREAM, 0)) < 0 ) {
    perror("irc.c irc_connect socket");
    bs_function_leave();
    return(-1);
  }

  if (connect(server_sd, (struct sockaddr *) &server_addr, sizeof(server_addr)) < 0) {
    perror("irc.c irc_connect connect");
    bs_function_leave();
    return(-1);
  } 

  /* Register with the server */
  /* NOTE: In client to server connections the <hostname> & <servername> arguments */
  /*       to the USER command are ignored, so are set to foo.com                  */

/*   if (pass == NULL) { */
/*     irc_send_format("USER %s foo.com foo.com :%s\r\nNICK %s", */
/* 		   user.login, name, nick); */
/*   } else { */
/*     irc_send_format("PASS %s\r\nUSER %s foo.com foo.com :%s\r\nNICK %s",  */
/* 		   pass, user.login, name, nick); */
/*   } */
  
  fcntl(server_sd, F_SETFL, O_NDELAY);

  bs_function_leave();
  return(server_sd);
}

void irc_disconnect(int server_sd)
{

  bs_function_enter();

  if (!server_sd) {
    return;
  }

  close(server_sd);

  bs_function_leave();
  return;
}

void irc_clear_message()
{
  char *argptr;
  int argn = 0;

  bs_function_enter();

  argptr = irc_message.args[argn];
  while (argptr != NULL) {
    bs_printf(3, "  free'ing irc_message.args[%d]<%s>", argn, argptr);
    g_free(argptr);
    argptr = irc_message.args[++argn];
  }
/*   g_free(irc_message.params); */
  bs_printf(3, "  free'ing irc_message.command<%s>", irc_message.command);
  g_free(irc_message.command);
  if (irc_message.nick != NULL) {
    bs_printf(3, "  free'ing irc_message.nick<%s>", irc_message.nick);
    g_free(irc_message.nick);
  }
  if (irc_message.address != NULL) {
    bs_printf(3, "  free'ing irc_message.address<%s>", irc_message.address);
    g_free(irc_message.address);
  }

  bs_function_leave();
  return;
}

int irc_parse_command(char *command)
{
  char *commptr, *prefix, *params, *nick, *address;
  int nargs=0;
  char argbuff[BUFFLEN];

  bs_function_enter();
  bs_printf(2, "  command<%s>", command);

  if ( command[0] == ':' ) {
    prefix = strtok(command+1, " ");
    commptr = strtok(NULL, " ");
  } else { /* else prefix is missing - probably a PING from the server */
    irc_message.nick = NULL;
    bs_printf(3, "  irc_message.nick = NULL");
    irc_message.address = NULL;
    bs_printf(3, "  irc_message.address = NULL");
    prefix = NULL;
    commptr = strtok(command, " ");
  }

  irc_message.command = g_strdup(commptr);
  bs_printf(3, "  irc_message.command<%s>", irc_message.command);
  if ( (irc_message.number=atoi(commptr)) > 0) {
    irc_message.type = MSG_NUMERIC;
  } else {
    irc_message.type = MSG_COMMAND;
  }
  while ( (params = strtok(NULL, " ")) != NULL) {
    if (strlen(params) == 0) {
      continue;
    }
    if (params[0] == ':') {
      strncpy(argbuff, params, BUFFLEN);
      if ( (params = strtok(NULL, "\0")) != NULL) {
	strncat(argbuff, " ", BUFFLEN);
	strncat(argbuff, params, BUFFLEN);
      }
      /* Leave out the ':' prefix */
      irc_message.args[nargs] = g_strdup(argbuff+1);
      bs_printf(3, "  irc_message.args[%d]<%s>", nargs, irc_message.args[nargs]);
      nargs++;
      break;
    }

    irc_message.args[nargs] = (char *) g_strdup(params);
    bs_printf(3, "  irc_message.args[%d]<%s>", nargs, irc_message.args[nargs]);
    nargs++;
    if (nargs == (MAXARGS-1)) {
      break;
    }
  }
  irc_message.args[nargs] = NULL;
  bs_printf(3, "  irc_message.args[%d] = NULL", nargs);

  if (prefix != NULL) {
    nick = strtok(prefix, "!");
    irc_message.nick = g_strdup(nick);
    bs_printf(3, "  irc_message.nick<%s>", irc_message.nick);
    address = strtok(NULL, "\0");
    if (address) {
      irc_message.address = g_strdup(address);
      bs_printf(3, "  irc_message.address<%s>", irc_message.address);
    } else {
      irc_message.address = NULL;
      bs_printf(3, "  irc_message.address = NULL");
    }
  } else {
    irc_message.nick = NULL;
    bs_printf(3, "  irc_message.nick = NULL");
    irc_message.address = NULL;
    bs_printf(3, "  irc_message.address = NULL");
  }

  bs_function_leave();
  return(nargs);
} 

void irc_parse_buffer(char *buff, unsigned blen)
{
  static char commbuff[BUFFLEN];
/*   static int commlen=0; */
  static int havebuff=0, haveCR = 0;
  char *buffptr=0;
  char *commend;
 
  bs_function_enter();

  if (buff == NULL) {
    /* Clearing buffer */
    havebuff = 0;
    bs_function_leave();
    return;
  }

  buffptr = buff;

  if (haveCR == 1) {
    if ( buff[0] == '\n' ) {
      /* We have a full command - dispatch it */
      commbuff[strlen(commbuff)-1] = '\0';
      irc_parse_command(commbuff);
      havebuff = 0;
      buffptr+=1;
    } else {
      haveCR = 0;
      havebuff = 0;
      commbuff[0] = '\0';
    }
  }

  /* Construct a full command string from the end of the last buffer, and */
  /* the beginning of this one */
  if (havebuff != 0) {
    if ( (commend = strchr(buffptr, '\r')) == NULL) {
      /* CR hasn't been found - append the new string to the existing one */
      if ( (strlen(commbuff)+strlen(buffptr)) >= BUFFLEN) {
	/* We seem to have a serious problem. The string is longer than BUFFLEN */
	/* Throw them both away, and we'll recover later */
	/* TODO: report this back to the user */
	BS_DEBUG(1,"command buffer exceeded\n");
	havebuff = 0;
	commbuff[0] = '\0';
	bs_function_leave();
	return;
      }
      strcat(commbuff, buffptr);
      havebuff = 1;
    } else if (commend[0] == '\0') {
      /* The last character in the buffer was CR */
      /* Raise a flag for next time and copy into local buffer */
      haveCR = 1;
      strcat(commbuff, buffptr);
      havebuff = 1;
      bs_function_leave();
      return;
    } else if (commend[1] == '\n') {
      commend[0] = '\0';
      strcat(commbuff, buffptr);
      irc_parse_command(commbuff);
      buffptr = commend+2;
      havebuff = 0;
      commbuff[0] = '\0';
    }
  }

  while( (commend = strchr(buffptr, '\r')) != NULL) {
    if (commend[1] == '\n') {
      commend[0] = '\0';
      irc_parse_command(buffptr);
      buffptr = commend+2;
      if (buffptr[0] == '\0') {
	havebuff = 0;
	bs_function_leave();
	return;
      }
    } else if (commend[0] == '\0') {
      /* The last character in the buffer was CR */
      /* Raise a flag for next time and copy into local buffer */
      haveCR = 1;
      strncpy(commbuff, buffptr, BUFFLEN);
      havebuff = 1;
      bs_function_leave();
      return;
    } else {
      /* I don't think this should happen, so report it */
      BS_DEBUG(1, "command contains rubbish\n");
      haveCR = 0;
      havebuff = 0;
      commbuff[0] = '\0';
      bs_function_leave();
      return;
    }
  }
  strncpy(commbuff, buffptr, BUFFLEN);
  havebuff = 1;

  bs_function_leave();
  return;
}

/* int irc_cmd_pong(char *message) */
/* { */

/*   bs_function_enter(); */
  
/*   if (message) { */
/*     irc_send_format("PONG %s", message); */
/*   } else { */
/*     irc_send_message("PONG"); */
/*   } */

/*   bs_function_leave(); */
/*   return(0); */
/* } */

void irc_send_format(int server_sd, char *format, ...)
{
  char buff[BUFFLEN];

#ifdef HAVE_VSNPRINTF
  va_list args;

  bs_function_enter();

  va_start (args, format);
  vsnprintf (buff, BUFFLEN, format, args);
  va_end (args);

#else
  gchar *printed;
  va_list args, args2;

  bs_function_enter();

  va_start (args, format);
  va_start (args2, format);

  printed = g_vsprintf (format, &args, &args2);
  strncpy (buff, printed, BUFFLEN);
  buff[BUFFLEN-1] = '\0';

  va_end (args2);
  va_end (args);

#endif             

#if 0
  va_start(args, format);
  vsnprintf(buff, BUFFLEN, format, args);
  va_end(args);
#endif

  irc_send_message(server_sd, buff);

  bs_function_leave();
  return;
}

void irc_send_message(int server_sd, char *message)
{

  bs_function_enter();

  if (server_sd <= 0) { /* Not connected */
    return;
  }

  /* TODO: the socket has been set non-blocking, so the return 
   * values of these need checking */
  if ( (write(server_sd, message, strlen(message)) < strlen(message)) ||
       (write(server_sd, "\r\n", 2) < 2) ) {
    fprintf(stderr, "Error on write in irc_send_message!\n");
  }

  bs_function_leave();
  return;
}

/* void irc_cmd_pass(char *password) */
/* { */
/*   bs_function_enter(); */

/*   if (password) { */
/*     irc_send_format("PASS %s", password); */
/*   } */
/*   bs_function_leave(); */
/*   return;   */
/* } */

/* void irc_cmd_nick(char *nick) */
/* { */
/*   bs_function_enter(); */

/*   if (nick) { */
/*     irc_send_format("NICK %s", nick); */
/*   } */
/*   bs_function_leave(); */
/*   return;   */
/* } */

void irc_cmd_user(int server_sd, char *nick, char *name)
{
  bs_function_enter();

  if (name) {
    irc_send_format(server_sd, "USER %s foo.com foo.com :%s", nick, name);
  }
  bs_function_leave();
  return;  
}

/* void irc_cmd_quit(char *message) */
/* { */

/*   bs_function_enter(); */

/*   if (message) { */
/*     irc_send_format("QUIT :%s", message); */
/*   } else { */
/*     irc_send_message("QUIT"); */
/*   } */

/*   irc_parse_buffer(NULL, 0); */
/*   if (user.server != NULL) { */
/*     g_free(user.server); */
/*     user.server = NULL; */
/*   } */
/*   bs_function_leave(); */
/*   return; */
/* } */
