/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
 
#include <glib.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include "bezerk.h"
#include "ctcp.h"
#include "reply.h"
#include "irc.h"
#include "ch_utils.h"
#include "message.h"
#include "send.h"
#include "util.h"
#include "dcc.h"
#include "debug.h"
#include "version.h"

extern messageData irc_message;

extern GdkColor colour_red;
extern GdkColor colour_yellow;
extern GdkColor colour_blue;
extern GdkColor colour_grey;
extern GdkColor colour_green;
extern GdkColor colour_white;

/* ctcp_recv_* are received in PRIVMSG commands from other people       */
/* ctcp_send_* are used to send requests to other people using PRIVMSG  */
/* ctcp_response_* are responses to our requests                        */

void ctcp_recv_action(Connection *connection, char *action)
{
  Message *message;
  ChannelInfo *target_channel;

  bs_function_enter();

  if ( !irc_message.nick || !irc_message.args || !irc_message.args[0] ) {
    bs_function_leave();
    return;
  }

  if ( (target_channel = find_channel(connection->channels, irc_message.args[0])) == NULL ) {
    bs_function_leave();
    return;
  }
 
  message = message_new(MT_CHANNEL, connection, irc_message.args[0]);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, "> "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, irc_message.nick));
  if ( target_channel != target_channel->window->current_channel ) {
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, " ["));
    message->parts = g_slist_append(message->parts, message_part_new(&colour_yellow, NULL, NULL, 
								     irc_message.args[0]));
    message->parts = g_slist_append(message->parts, message_part_new(&colour_white, NULL, NULL, "]"));
  }
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_white, NULL, NULL, " "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					  action);

   bs_function_leave();
   return;
}

void ctcp_recv_ping(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!args || !irc_message.nick) {
    bs_function_leave();
    return;
  }

  irc_send_format(connection->sd, "NOTICE %s :\001PING %s\001", irc_message.nick, args);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_green, NULL, NULL, "CTCP PING received from "));
  message->parts = g_slist_append(message->parts, 
				  message_part_new(&colour_green, NULL, NULL, irc_message.nick));

  bs_function_leave();
  return;
}
 
void ctcp_recv_version(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!irc_message.nick) {
    bs_function_leave();
    return;
  }

  irc_send_format(connection->sd, "NOTICE %s :\001VERSION Bezerk %s by Tony Gale [Trog]. Go Bezerk!\001", 
	   irc_message.nick, BEZERK_VERSION);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, "CTCP VERSION received from "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, irc_message.nick));

  bs_function_leave();
  return;
}

void ctcp_recv_echo(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!irc_message.nick || !args) {
    bs_function_leave();
    return;
  }

  irc_send_format(connection->sd, "NOTICE %s :\001ECHO %s\001", 
	   irc_message.nick, args);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, "CTCP ECHO received from "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, irc_message.nick));

  bs_function_leave();
  return;
}

void ctcp_recv_clientinfo(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!irc_message.nick) {
    bs_function_leave();
    return;
  }

  irc_send_format(connection->sd, "NOTICE %s :\001CLIENTINFO ACTION CLIENTINFO ECHO PING VERSION\001", 
	   irc_message.nick);

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL,
						  "CTCP CLIENTINFO received from "));
  message->parts = g_slist_append(message->parts, 
				 message_part_new(&colour_green, NULL, NULL, irc_message.nick));

  bs_function_leave();
  return;
}

USER_COMMAND(ctcp_send_action)
{
  Message *message=NULL;

  bs_function_enter();

  if (!args || !window) {
    bs_function_leave();
    return(0);
  }

  if (IS_BEZ_CHANNEL_WINDOW(window)) {
    BezChannelWindow *channel_window = (BezChannelWindow *) window;
    if (channel_window->current_channel) {
      irc_send_format(window->connection->sd, "PRIVMSG %s :\001ACTION %s\001", channel_window->current_channel->name, args);

      message = message_new(MT_CHANNEL, window->connection, channel_window->current_channel->name);
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_green, NULL, NULL, "> "));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_green, NULL, NULL,
						       BEZ_CHANNEL_WINDOW(window)->connection->nick));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white, NULL, NULL, " "));
      message->parts = message_part_parse_new(message->parts, &colour_white, 
					      args);
    }
  } else if (IS_BEZ_MESSAGE_WINDOW(window)) {
    BezMessageWindow *message_window = (BezMessageWindow *) window;
    irc_send_format(window->connection->sd, "PRIVMSG %s :\001ACTION %s\001", message_window->nick, args);

    message = message_new(MT_NICK, window->connection, message_window->nick);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL, "> "));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL,
						     BEZ_MESSAGE_WINDOW(window)->connection->nick));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, " "));
    message->parts = message_part_parse_new(message->parts, &colour_white, 
					    args);
  }

  bs_function_leave();
  return(message ? message->type : 0);
}

USER_COMMAND(ctcp_send_ping)
{
  Message *message=NULL;
  char *current_time;

  bs_function_enter();

  if (!window) {
    bs_function_leave();
    return(0);
  }

  current_time = get_time_string();

  if (window->type == BEZ_CHANNEL) {
    BezChannelWindow *channel_window = (BezChannelWindow *) window;
    if (channel_window->current_channel) {
      if (!args) {
	irc_send_format(window->connection->sd, "PRIVMSG %s :\001PING %s\001", channel_window->current_channel->name, current_time);
      } else {
	irc_send_format(window->connection->sd, "PRIVMSG %s :\001PING %s\001", args, current_time);
      }
      message = message_new(MT_CHANNEL, window->connection, channel_window->current_channel->name);
      message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL, "CTCP PING request sent to "));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_green, NULL, NULL, 
						       args ? args : channel_window->current_channel->name));
    }
  } else if (window->type == BEZ_MESSAGE) {
    BezMessageWindow *message_window = (BezMessageWindow *) window;
    if (!args) {
      irc_send_format(window->connection->sd, "PRIVMSG %s :\001PING %s\001", message_window->nick, current_time);
    } else {
      irc_send_format(window->connection->sd, "PRIVMSG %s :\001PING %s\001", args, current_time);
    }
    message = message_new(MT_NICK, window->connection, message_window->nick);
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL, "CTCP PING request sent to "));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL, 
						     args ? args : message_window->nick));
  }    
  g_free(current_time);

  bs_function_leave();
  return(message ? message->type : 0);
}

USER_COMMAND(ctcp_send_version)
{
  Message *message=NULL;

  bs_function_enter();

   if (!window) {
    bs_function_leave();
    return(0);
  }

  if (window->type == BEZ_CHANNEL) {
    BezChannelWindow *channel_window = (BezChannelWindow *) window;
    if (channel_window->current_channel) {
      if (!args) {
	irc_send_format(window->connection->sd, "PRIVMSG %s :\001VERSION\001", channel_window->current_channel->name);
      } else {
	irc_send_format(window->connection->sd, "PRIVMSG %s :\001VERSION\001", args);
      }

      message = message_new(MT_CHANNEL, window->connection, channel_window->current_channel->name);
      message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP VERSION request sent to "));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_green, NULL, NULL,
						       (args ? args : channel_window->current_channel->name) ));
    }
  } else if (window->type == BEZ_MESSAGE) {
    BezMessageWindow *message_window = (BezMessageWindow *) window;
    if (!args) {
      irc_send_format(window->connection->sd, "PRIVMSG %s :\001VERSION\001", message_window->nick);
    } else {
      irc_send_format(window->connection->sd, "PRIVMSG %s :\001VERSION\001", args);
    }
    
    message = message_new(MT_NICK, window->connection, message_window->nick);
    message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP VERSION request sent to "));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_green, NULL, NULL,
						     (args ? args : message_window->nick) ));
  }

  bs_function_leave();
  return(message ? message->type : 0);
}

/* Echo is a special: command == destination, args == text to send */
USER_COMMAND(ctcp_send_echo)
{
  Message *message;

  bs_function_enter();

  if (!command || !window) {
    bs_function_leave();
    return(0);
  }

  if (!args) {
    irc_send_format(window->connection->sd, "PRIVMSG %s :\001ECHO Bezerk\001", command);
  } else {
    irc_send_format(window->connection->sd, "PRIVMSG %s :\001ECHO %s\001", command, args);
  }

  message = message_new(MT_CHANNEL, window->connection, NULL);
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_green, NULL, NULL, "CTCP ECHO request sent to "));
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_green, NULL, NULL, command));

  bs_function_leave();
  return(message->type);
}

USER_COMMAND(ctcp_send_clientinfo)
{
  Message *message;

  bs_function_enter();

  if (!args || !window) {
    bs_function_leave();
    return(0);
  }

  irc_send_format(window->connection->sd, "PRIVMSG %s :\001CLIENTINFO\001", args);

  message = message_new(MT_CHANNEL, window->connection, NULL);
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_green, NULL, NULL,
						   "CTCP CLIENTINFO request sent to "));
  message->parts = g_slist_append(message->parts,
				  message_part_new(&colour_green, NULL, NULL, args));
  bs_function_leave();
  return(message->type);
}

int ctcp_response_ping(Connection *connection, char *args)
{
  Message *message;
  char *time_ptr;

  bs_function_enter();

  if (!args || !irc_message.nick) {
    bs_function_leave();
    return(0);
  }

  time_ptr = get_time_diff(args);
 
  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP PING "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "("));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.nick));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "): "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, time_ptr));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, " seconds"));

  g_free(time_ptr);

  bs_function_leave();
  return(message->type);
}

int ctcp_response_version(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!args || !irc_message.nick) {
    bs_function_leave();
    return(0);
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP VERSION "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "("));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.nick));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "): "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					  args);

  bs_function_leave();
  return(message->type);
}

int ctcp_response_echo(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!args || !irc_message.nick) {
    bs_function_leave();
    return(0);
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP ECHO "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "("));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.nick));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "): "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					  args);

  bs_function_leave();
  return(message->type);
}

int ctcp_response_clientinfo(Connection *connection, char *args)
{
  Message *message;

  bs_function_enter();

  if (!args || !irc_message.nick) {
    bs_function_leave();
    return(0);
  }

  message = message_new(MT_CHANNEL, connection, NULL);
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_green, NULL, NULL, "CTCP CLIENTINFO "));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "("));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, irc_message.nick));
  message->parts = g_slist_append(message->parts,
				 message_part_new(&colour_white, NULL, NULL, "): "));
  message->parts = message_part_parse_new(message->parts, &colour_white, 
					  args);

  bs_function_leave();
  return(message->type);
}

USER_COMMAND(process_ctcp_command)
{
  char *dest, *comm, *rest;
  int retval=0;

  bs_function_enter();

  if ( !args || !args[0] ) {
      bs_function_leave();
      return(0);
  }

  dest = bs_strtok(args, " ");
  if ( (comm = bs_strtok(NULL, " ")) == NULL ) {
    bs_function_leave();
    return(0);
  }
  rest = bs_strtok(NULL, "\0");

  if (!strcasecmp("PING", comm))             retval = ctcp_send_ping("PING", dest, window);
  else if (!strcasecmp("VERSION", comm))     retval = ctcp_send_version("VERSION", dest, window);
  else if (!strcasecmp("ECHO", comm))        retval = ctcp_send_echo(rest, dest, window);
  else if (!strcasecmp("CLIENTINFO", comm))  retval = ctcp_send_clientinfo("CLIENTINFO", dest, window);

  bs_function_leave();
  return(retval);
}

void process_ctcp_request(Connection *connection)
{
  char *command, *args;

  bs_function_enter();

  if ( !connection || !irc_message.args || !irc_message.args[1]) {
    bs_function_leave();
    return;
  }

  command = strtok(&irc_message.args[1][1], " ");
  if (command[strlen(command)-1] == '\001') {
    command[strlen(command)-1] = '\0';
  }
  args = strtok(NULL, "\001");

  if (!strcmp("ACTION", command))           ctcp_recv_action(connection, args);
  else if (!strcmp("PING", command))        ctcp_recv_ping(connection, args);
  else if (!strcmp("VERSION", command))     ctcp_recv_version(connection, args);
  else if (!strcmp("ECHO", command))        ctcp_recv_echo(connection, args);
  else if (!strcmp("DCC", command))         process_dcc_request(args);

  bs_function_leave();
  return;
}

void process_ctcp_response(Connection *connection)
{
  char *command, *args;

  bs_function_enter();

  if (!irc_message.args || !irc_message.args[1]) {
    bs_function_leave();
    return;
  }

  command = strtok(&irc_message.args[1][1], " ");
  if (command[strlen(command)-1] == '\001') {
    command[strlen(command)-1] = '\0';
  }
  args = strtok(NULL, "\001");

  if (!strcasecmp("PING", command))                ctcp_response_ping(connection, args);
  else if (!strcasecmp("VERSION", command))        ctcp_response_version(connection, args);
  else if (!strcasecmp("ECHO", command))           ctcp_response_echo(connection, args);
  else if (!strcasecmp("CLIENTINFO", command))     ctcp_response_clientinfo(connection, args);

  bs_function_leave();
  return;
}
