/* Bezerk
 * Copyright (C) 1998 Tony Gale.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */
 
#include <unistd.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <errno.h>
#include <string.h>

#include "irc.h"
#include "menu.h"
#include "message.h"
#include "callback.h"
#include "dcc.h"
#include "servers.h"
#include "reply.h"
#include "send.h"
#include "ch_utils.h"
#include "msg_utils.h"
#include "debug.h"
#include "bezerk.h"
#include "popup.h"
#include "util.h"
#include "version.h"
#include "dialogs.h"

extern messageData irc_message;
extern messageInfo irc_info;

extern GSList *connections;

extern GdkColor colour_red;
extern GdkColor colour_yellow;
extern GdkColor colour_blue;
extern GdkColor colour_grey;
extern GdkColor colour_green;
extern GdkColor colour_white;

/* ------------------------------------------------------------
 * destroy_main
 *
 *  Callback for the programs main window. Clean up and exit the
 *  program.
 *
 * Arguments:
 *	"widget" - the object being destroyed
 *	"window" - the BezChannelWindow object of the window being destroyed
 *
 * Results:
 *	
 * Side effects:
 *	the program terminates
 *
 * ----------------------------------------------------------- */

void destroy_main (GtkWidget *widget, BezChannelWindow *window)
{
  bs_function_enter();

  if (window->connection && (window->connection->sd >=0 ) ) {
    send_quit("QUIT", NULL, (BezBaseWindow *) window);
    irc_disconnect(window->connection->sd);
    BEZ_BASE_WINDOW(window)->connection->sd = -1;
    gdk_input_remove(window->connection->tag);
  }
  if (!connection_destroy( (window->connection ? window->connection : NULL) )) {
    gtk_main_quit ();
  }

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * destroy_channel_window
 *
 *  Callback for all but the main window. Clean up and destroy
 *  the window
 *
 * Arguments:
 *	"widget" - the object being destroyed
 *	"window" - the BezChannelWindow object of the window being destroyed
 *
 * Results:
 *
 * Side effects:
 *	window is removed from the list of windows	
 *
 * ----------------------------------------------------------- */

void destroy_channel_window (GtkWidget *widget, BezChannelWindow *window)
{
  GList *glist_entry;

  bs_function_enter();

  window->connection->current_window = NULL;

  if (window->current_channel != NULL) {
    /*    remove_channel(window->current_channel->name);*/
    send_part("PART", NULL, (BezBaseWindow *) window);
  }
  window->connection->channels = g_slist_remove(window->connection->channels, window);

  glist_entry = window->recall_commands;
  while (glist_entry) {
    g_free(glist_entry->data);
    glist_entry = g_list_next(glist_entry);
  }
  g_list_free(window->recall_commands);

  g_free(window);

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * destroy_message_window
 *
 *  Callback for all but the main window. Clean up and destroy
 *  the window
 *
 * Arguments:
 *	"widget" - the object being destroyed
 *	"window" - the BezMessageWindow object of the window being destroyed
 *
 * Results:
 *
 * Side effects:
 *	"window" is removed from the list of message windows	
 *
 * ----------------------------------------------------------- */

void destroy_message_window (GtkWidget *widget, BezMessageWindow *window)
{
  bs_function_enter();

  if ( (window->message_type == DCC_CHAT) && (window->connection) && (window->connection->sd >= 0) ) {
    close(window->connection->sd);
  }

  window->connection->messages = remove_message(window->connection->messages, window->nick);

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * lag_check_callback
 *
 *  Timer callback to check lag to the server
 *
 * Arguments:
 *	"data" - user data (unused)
 *
 * Results:
 *	TRUE, to reset timer
 *
 * Side effects:
 *		
 * ----------------------------------------------------------- */

gint lag_check_callback (gpointer data)
{
  char *time_string;
  Connection *connection;
  /*   bs_function_enter(); */

  if (!connections) {
    return(TRUE);
  }

  /* TODO: Iterate over all server connections */
  connection = (Connection *) connections->data;
  if (connection->status == NOTREG) {
    return(TRUE);
  }
  time_string = get_time_string();
  irc_send_format(connection->sd, "NOTICE %s :LAG %s", connection->nick, time_string);
  g_free(time_string);

  /*   bs_function_leave(); */
  return(TRUE);
}

/* ------------------------------------------------------------
 * socket_input_callback
 *
 *  Callback when the server socket has data readyfor reading
 *
 * Arguments:
 *	"data" - callback user data (unused)
 *
 * Results:
 *	0
 *
 * Side effects:
 *
 * ----------------------------------------------------------- */

void socket_input_callback (gpointer data, gint source, GdkInputCondition condition)
{
  int n;
  static char buff[BUFFSIZE];
  static char *buff_ptr = buff;
  Message *message;
  Connection *connection;
  BezChannelWindow *window;

  bs_function_enter();

  if ( !(connection = connection_find_by_sd(source)) ) {
    /* Somethings gone wrong - shouldn't happen */
    bs_printf(1, "socket_input_callback: unknown source socket: %d", source);
    close(source);
  }

  while (1) {
    if ( (n = recv(source, buff_ptr, 1, 0)) != -1) {
      if ( *buff_ptr == '\n' ) {
	buff_ptr[0] = '\0';
	break;
      } else if ( *buff_ptr == '\r') {
	continue;
      }
    } else {
      switch (errno)
	{
	case EWOULDBLOCK:
	  return;
	case ENOTCONN:
	  /* TODO: need to reconnect to the server */
	  /*       but just clean up for now */	
	  message = message_new(MT_CONSOLE, connection, NULL);
	  message->parts = g_slist_append(message->parts, 
					 message_part_new(&colour_white, NULL, NULL, "Disconnected"));
	  messages_write();
	  gdk_input_remove(connection->tag);
	  window = BEZ_CHANNEL_WINDOW(connection->console);
	  connection_destroy(connection);
	  window->connection = NULL;
/* 	  clean_up(window); */
	  return;
	default:
	  { 
	    char *err_str;
	    err_str = strerror(errno);
	    bs_printf(1, "Unrecognised recv error %d: %s", errno, err_str);
	    message = message_new(MT_CONSOLE, connection, NULL);
	    message->parts = g_slist_append(message->parts, 
					    message_part_new(&colour_white, NULL, NULL, "Disconnected"));
	    messages_write();
	    gdk_input_remove(connection->tag);
	    connection_destroy(connection);
	    return;
	  }
	}
    }
    if ( ++buff_ptr == (buff+BUFFLEN) ) {
      buff_ptr[-1] = '\0';
      break;
    }
  }

  if ( strlen(buff) > 2 ) {
    irc_parse_command(buff);
    channel_issue_message(connection);
    irc_clear_message();
  }

  buff_ptr = buff;
  buff[0] = '\0';

  bs_function_leave();
  return;
}

void toolbar_copy_callback (GtkWidget *widget, BezWindow *window)
{
  bs_function_enter();

  gtk_editable_copy_clipboard(GTK_EDITABLE(BEZ_BASE_WINDOW(window)->text), GDK_CURRENT_TIME);

  bs_function_leave();
  return;
}

void toolbar_paste_callback (GtkWidget *widget, BezWindow *window)
{
  static GdkAtom targets_atom = GDK_NONE;
  static GdkAtom clipboard_atom = GDK_NONE;
  bs_function_enter();

  /* Get the atom corresonding to the string "STRING" */
  if (targets_atom == GDK_NONE) {
    targets_atom = gdk_atom_intern ("STRING", FALSE);
    clipboard_atom = gdk_atom_intern ("CLIPBOARD", FALSE);
  }

  gtk_selection_convert (widget, clipboard_atom,
			 targets_atom, GDK_CURRENT_TIME);

  bs_function_leave();
  return;
}

void toolbar_selection_received (GtkWidget *widget, GtkSelectionData *selection_data, 
				 BezWindow *window)
{
  Message *message;
  char *cmd_tmp, *command;

  if ( (selection_data->length <= 0) || !window ||
       !BEZ_BASE_WINDOW(window)->connection || !BEZ_CHANNEL_WINDOW(window)->current_channel) {
    return;
  }

  if (selection_data->type != GDK_SELECTION_TYPE_STRING) {
    g_print("selection_data->type != GDK_SELECTION_TYPE_STRING\n");
    return;
  }

  cmd_tmp = g_strdup(selection_data->data);
  command = strtok(cmd_tmp, "\n");
  while(command) {
    irc_send_format(BEZ_MESSAGE_WINDOW(window)->connection->sd,
		    "PRIVMSG %s :%s", BEZ_CHANNEL_WINDOW(window)->current_channel->name, command);
    message = message_new(MT_CHANNEL, BEZ_BASE_WINDOW(window)->connection,
			  NULL);
    message->parts = g_slist_append(message->parts, 
				   message_part_new(&colour_blue, NULL, NULL, "<"));
    message->parts = g_slist_append(message->parts,
				   message_part_new(&colour_white,NULL, NULL,
						    BEZ_CHANNEL_WINDOW(window)->connection->nick));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_blue, NULL, NULL, "> "));
    message->parts = g_slist_append(message->parts,
				    message_part_new(&colour_white, NULL, NULL, command));
    command = strtok(NULL, "\n");
  }
  g_free(cmd_tmp);

  messages_write();
  return;
}

/* ------------------------------------------------------------
 * toolbar_connect_callback
 *
 *  Callback for the File->Connect menu option. Connects to server
 *
 * Arguments:
 *	"widget" - the item selected to invoke this function
 *	"window" - BezChannelWindow callback data
 *
 * Results:
 *
 * Side effects:
 *	none on failed connection
 *		or
 *	"connection" is filled in
 *	menu item sensitivities are changed
 *	socket i/o callback to socket_input_callback established
 *
 * ----------------------------------------------------------- */

void toolbar_connect_callback (GtkWidget *widget, BezWindow *window)
{
  Message *message;
  Connection *connection;
  Server *server;
  int server_sd;
  char *selection;

  bs_function_enter();

  /* Pop-down the menu, so we don't leave the pointer grab on */
  while (gtk_events_pending())
        gtk_main_iteration();

  selection = gtk_entry_get_text( GTK_ENTRY( GTK_COMBO(window->channel.toolbar->combo)->entry) );

  if (BEZ_CHANNEL_WINDOW(window)->connection) {
    if ( !connection_find_by_name(selection) ) {
      gchar title[BUFFLEN];
      g_snprintf(title, BUFFLEN, "Bezerk %s by Tony Gale [Trog]", BEZERK_VERSION);
      window = create_channel_window(title, TRUE);
    } else {
      return;
    }
  }


  if ( !(server = servers_find(selection)) ) {
    gdk_beep();
    return;
  }

/*   server = bs_strtok(selection, ":"); */
/*   port = bs_strtok(NULL, "\0"); */

  /* TODO: pick a random server port */
  if ( (server_sd = irc_connect(server->name, server->low_port)) < 0) {
    /* TODO: report this error back to the user */
    BS_DEBUG(1,"file_connect_callback - connect failed");
    return;
  } else {
    connection = connection_create(server, server_sd, server->name, server->low_port, window);
    BEZ_CHANNEL_WINDOW(window)->connection = connection;

    irc_cmd_user(server_sd, getlogin(), connection->name);
    irc_send_format(server_sd, "NICK %s", connection->nick);

/*     menus_set_sensitive("<Main>/File/Disconnect", TRUE); */
/*     menus_set_sensitive("<Main>/File/Connect", FALSE); */
    set_user_status(connection);

    /* Have GTK monitor this socket for us */
    connection->tag = gdk_input_add(server_sd, GDK_INPUT_READ, 
			(GdkInputFunction) socket_input_callback, NULL);

    message = message_new(MT_CONSOLE, connection, NULL);
    message->parts = g_slist_append(message->parts, 
				    message_part_new(&colour_white, NULL, NULL, "Connected"));
    messages_write();
  }

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * file_disconnect_callback
 *
 *  Callback for the File->Disconnect menu option. Disconnects from
 *  the server
 *
 * Arguments:
 *	"widget" - the menu item object
 *	"data" - callback user data (unused)
 *
 * Results:
 *
 * Side effects:
 *	server connection is closed
 *	menu sensitivites changed
 *	
 * ----------------------------------------------------------- */

void file_disconnect_callback (GtkWidget *widget, gpointer data)
{
  bs_function_enter();

/*   send_quit("QUIT", NULL, channel_windows->data); */
/*   messages_write(); */

/*   menus_set_sensitive("<Main>/File/Disconnect", FALSE); */
/*   menus_set_sensitive("<Main>/File/Connect", TRUE); */

  bs_function_leave();
  return;
}

/* ------------------------------------------------------------
 * file_quit_callback
 *
 *  Callback for the File->Quit menu option. Quits the 
 *  program
 *
 * Arguments:
 *	"widget" - the menu item object
 *	"data" - callback user data (unused)
 *
 * Results:
 *
 * Side effects:
 *	the program exits
 *	
 * ----------------------------------------------------------- */

void file_quit_callback (GtkWidget *widget, gpointer data)
{
  bs_function_enter();

/*   current_window = channel_windows->data; */

/*   if (widget == channel_windows->data) { */
     /* TODO: ask about closing main window */
/*   } */
  
/*   send_quit("QUIT", NULL, channel_windows->data); */
  gtk_main_quit();

  bs_function_leave();
  return;
}

void settings_options_callback(GtkWidget *widget, gpointer data)
{

  bs_function_enter();

  preferences_dialog(PREFS_DEFAULT);

  bs_function_leave();
  return;
} 

void settings_aliases_callback(GtkWidget *widget, gpointer data)
{

  bs_function_enter();

  script_dialog();

  bs_function_leave();
  return;
} 

void help_about_callback(GtkWidget *widget, gpointer data)
{

  bs_function_enter();

  about_dialog(widget);

  bs_function_leave();
  return;
}

void settings_user_callback(GtkWidget *widget, gpointer data)
{

  bs_function_enter();

  preferences_dialog(PREFS_PERSONA_PAGE);

  bs_function_leave();
  return;
}

void settings_fonts_callback(GtkWidget *widget, gpointer data)
{

  bs_function_enter();

  preferences_dialog(PREFS_FONTS_PAGE);

  bs_function_leave();
  return;
}

void console_entry_callback(GtkWidget *widget, BezWindow *window)
{
  char *full_text, *entry_text, *spacepos, *complete, *new_text, tmp_char;
  int text_len, count;
  Message *message;

  bs_function_enter();

  if (BEZ_BASE_WINDOW(window)->connection) {
    BEZ_BASE_WINDOW(window)->connection->current_window = window;
  }

  full_text = g_strdup(gtk_entry_get_text(GTK_ENTRY(widget)));
  gtk_entry_set_text(GTK_ENTRY(widget),"");
  if ( (text_len = strlen(full_text)) == 0) {
    bs_function_leave();
    return;
  }

  if ( window->base.incomplete_command ) {
    g_free(window->base.incomplete_command);
    window->base.incomplete_command = NULL;
  }

#ifdef DEBUG_ECHO  
  if (full_text[0] == ':') {
    irc_send_message(BEZ_BASE_WINDOW(window)->connection->sd, full_text);
    /*     gtk_entry_set_text(GTK_ENTRY(widget),""); */
    g_free(full_text);
    bs_function_leave();
    return;
  }
#endif

  /* Check for nick completion */
  if ( (window->type == BEZ_CHANNEL) ) {
    if ( window->channel.current_channel && (full_text[0] != ' ') && (spacepos = strchr(full_text, ' ')) ) {
      if ( strchr( ":,", spacepos[-1]) ) {
	tmp_char = spacepos[-1];
	spacepos[-1]='\0';
	if ( (complete = complete_nick(window->channel.current_channel, full_text)) ) {
	  new_text = g_malloc(strlen(complete) + strlen(spacepos)+2);
	  sprintf(new_text, "%s%c%s", complete, tmp_char, spacepos);
	  g_free(full_text);
	  full_text = new_text;
	} else {
	  spacepos[-1] = tmp_char;
	}
      }
    }
  }

  entry_text = strtok(full_text, "\n");
  while(entry_text) {
    
    for (count = 0 ; count < text_len ; count++) {
      if (entry_text[count] == '\244')
	entry_text[count] = '\002';
      else if (entry_text[count] == '\253')
	entry_text[count] = '\026';
      else if (entry_text[count] == '\273')
	entry_text[count] = '\037';
      else if (entry_text[count] == '\336')
	entry_text[count] = '\003';
    }

    if ( !(BEZ_BASE_WINDOW(window)->connection) ) {
      /*     message = message_new(MT_CONSOLE, BEZ_BASE_WINDOW(window)->connection, NULL); */
      /*     message->parts = g_slist_append(message->parts, */
      /* 				   message_part_new(&colour_white, NULL, NULL, "Not connected")); */
      /*     messages_write(); */
      bs_function_leave();
      return;
    } else if (strlen(entry_text) > 470) {
      bs_function_leave();
      /* TODO: return this as an error */
      return;
  }

    cmd_recall_add(entry_text, (BezBaseWindow *) window);

    /* This is allow me to test things without an irc server */

    if ( (entry_text[0] == '/') && BEZ_BASE_WINDOW(window)->connection->sd) {
      if (IS_BEZ_MESSAGE_WINDOW(window) && (BEZ_MESSAGE_WINDOW(window)->message_type == DCC_CHAT)) {
	dcc_chat_command(entry_text+1, window);
      } else {
	send_issue_command(entry_text+1, window);
      }
    } else if (IS_BEZ_MESSAGE_WINDOW(window) && BEZ_MESSAGE_WINDOW(window)->connection ) {
      if (BEZ_MESSAGE_WINDOW(window)->message_type == DCC_CHAT) {
	irc_send_message(BEZ_MESSAGE_WINDOW(window)->connection->sd, 
			 entry_text);
      } else {
	irc_send_format(BEZ_MESSAGE_WINDOW(window)->connection->sd, 
			"PRIVMSG %s :%s", window->message.nick, entry_text);
      }
      message = message_new(MT_NICK, BEZ_BASE_WINDOW(window)->connection, window->message.nick);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_blue, NULL, NULL, "<"));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white,NULL, NULL, 
						       BEZ_MESSAGE_WINDOW(window)->connection->nick));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_blue, NULL, NULL, "> "));
      message->parts = message_part_parse_new(message->parts, &colour_white, entry_text);    
    } else if (IS_BEZ_CHANNEL_WINDOW(window) && BEZ_CHANNEL_WINDOW(window)->connection &&
	       BEZ_CHANNEL_WINDOW(window)->current_channel) {
      irc_send_format(BEZ_MESSAGE_WINDOW(window)->connection->sd,
		      "PRIVMSG %s :%s", BEZ_CHANNEL_WINDOW(window)->current_channel->name, entry_text);
      message = message_new(MT_CHANNEL, BEZ_BASE_WINDOW(window)->connection,
			    BEZ_CHANNEL_WINDOW(window)->current_channel->name);
      message->parts = g_slist_append(message->parts, 
				      message_part_new(&colour_blue, NULL, NULL, "<"));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_white,NULL, NULL,
						       BEZ_CHANNEL_WINDOW(window)->connection->nick));
      message->parts = g_slist_append(message->parts,
				      message_part_new(&colour_blue, NULL, NULL, "> "));
      message->parts = message_part_parse_new(message->parts, &colour_white, entry_text);
    }

    entry_text = strtok(NULL, "\n");
  }
  messages_write();

  g_free(full_text);

  bs_function_leave();
  return;
}

gint entry_key_press (GtkWidget *widget, GdkEventKey *event, BezBaseWindow *window)
{
  BezChannelWindow *channel_window;
  char *command, *complete, *newcommand, *spacepos, tmp_char;
  int position, newlength;

  if (!window->connection) {
    bs_function_leave();
    return(FALSE);
  }
  window->connection->current_window = (BezWindow *) window;

  switch (event->keyval) {
  case GDK_Up:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    if ( window->recall_current == NULL ) {
      command = gtk_entry_get_text( GTK_ENTRY(widget) );
      if ( strlen(command) > 0) {
	if ( window->incomplete_command != NULL ) {
	  g_free(window->incomplete_command);
	}
	window->incomplete_command = g_strdup(command);
	if ( (command = cmd_recall_previous(window)) ) {
	  gtk_entry_set_text( GTK_ENTRY(widget), command);
	}
      } else if ( window->incomplete_command != NULL ) {
	gtk_entry_set_text( GTK_ENTRY(widget), window->incomplete_command);
	g_free(window->incomplete_command);
	window->incomplete_command = NULL;
      }	else if ( (command = cmd_recall_previous(window)) ) {
	gtk_entry_set_text( GTK_ENTRY(widget), command);
      }
    } else if ( (command = cmd_recall_previous(window)) ) {
      gtk_entry_set_text( GTK_ENTRY(widget), command);
    }
    return TRUE;
  case GDK_Down:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    command = gtk_entry_get_text( GTK_ENTRY(widget) );    
    if ( window->recall_current == NULL && (strlen(command) > 0) ) {
      if ( window->incomplete_command != NULL ) {
	g_free(window->incomplete_command);
      }
      window->incomplete_command = g_strdup(command);
      gtk_entry_set_text( GTK_ENTRY(widget), "");
    } else if ( (command = cmd_recall_next(window)) ) {
      gtk_entry_set_text( GTK_ENTRY(widget), command);
    } else if ( window->incomplete_command != NULL ) {
      command = gtk_entry_get_text( GTK_ENTRY(widget) );
      if ( (strlen(command) > 0) && (strcmp(window->incomplete_command, command) != 0) ) {
	gtk_entry_set_text( GTK_ENTRY(widget), window->incomplete_command);
      } else {
	gtk_entry_set_text( GTK_ENTRY(widget), "");
      }
    } else {
      gtk_entry_set_text( GTK_ENTRY(widget), "");
    } 
    return TRUE;
  case GDK_F1:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    gtk_editable_insert_text( GTK_EDITABLE(widget), "\244", 1, &GTK_EDITABLE(widget)->current_pos);
    return TRUE;
  case GDK_F2:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    gtk_editable_insert_text( GTK_EDITABLE(widget), "\253", 1, &GTK_EDITABLE(widget)->current_pos);
    return TRUE;
  case GDK_F3:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    gtk_editable_insert_text( GTK_EDITABLE(widget), "\273", 1, &GTK_EDITABLE(widget)->current_pos);
    return TRUE;
  case GDK_F4:
    gtk_signal_emit_stop_by_name (GTK_OBJECT (widget), "key_press_event");
    gtk_editable_insert_text( GTK_EDITABLE(widget), "\336", 1, &GTK_EDITABLE(widget)->current_pos);
    return TRUE;
  case GDK_Escape:
    if ( window->type == BEZ_CHANNEL ) {
      channel_window = (BezChannelWindow *) window;
      if (!channel_window->current_channel) {
	break;
      }
      command = g_strdup(gtk_entry_get_text( GTK_ENTRY(widget) ));
      if (strlen(command)) {
	position = GTK_EDITABLE(widget)->current_pos;
	tmp_char = command[position];
	command[position]='\0';
	if ( (spacepos = strrchr(command, ' ')) ) {
	  spacepos[0]='\0';
	  if ( (complete = complete_nick(channel_window->current_channel, spacepos+1)) ) {
	    newlength = strlen(command)+strlen(complete)+1+strlen(command+position+1)+2;
	    newcommand = g_malloc(newlength);
	    g_snprintf(newcommand, newlength, "%s %s%c%s", 
		       command, complete, tmp_char, command+position+1);
	    gtk_entry_set_text(GTK_ENTRY(widget), newcommand);
	    gtk_entry_set_position(GTK_ENTRY(widget), position + (strlen(complete) - strlen(spacepos+1)));
	    /* Need a gtk_widget_draw to update the cursor position */
	    gtk_widget_draw(widget, NULL);
	    g_free(newcommand);
	  }
	} else {
	  if ( (complete = complete_nick(channel_window->current_channel, command)) ) {
	    newlength = strlen(complete) + 1 + strlen(command+position+1) + 2;
	    newcommand = g_malloc(newlength);
	    g_snprintf(newcommand, newlength, "%s%c%s", complete, tmp_char, command+position+1);
	    gtk_entry_set_text(GTK_ENTRY(widget), newcommand);
	  }
	}
      }
      g_free(command);
      return TRUE;
    }
  }

  return FALSE;
}

void channel_clist_callback (GtkWidget *widget, gint row, gint column, 
			     GdkEventButton *bevent, BezChannelWindow *window)
{
  char buff[BUFFLEN+20];
  ChannelInfo *target_channel;
  gchar *channel_name;

  bs_function_enter();

  window->connection->current_window = (BezWindow *) window;

  gtk_clist_get_text ( GTK_CLIST(widget), row, 0, &channel_name);

  window->selected_row=row;

  if ( (target_channel = find_channel(window->connection->channels, channel_name)) == NULL ) {
    printf("channel not found\n");
    bs_function_leave();
    return;
  }

  if (target_channel != window->current_channel) {
    window->current_channel = target_channel;
    set_members_list(target_channel);
    if (target_channel->topic) {
      g_snprintf(buff, BUFFLEN, "Bezerk: %s - %s", target_channel->name, target_channel->topic);
      gtk_window_set_title(GTK_WINDOW(window->window), buff);
    } else {
      g_snprintf(buff, BUFFLEN, "Bezerk: %s", target_channel->name);
      gtk_window_set_title(GTK_WINDOW(window->window), buff);
    }

    set_channel_status(target_channel);
  }

  if (bevent) {
    if (bevent->button == 3) {
      channel_popup(target_channel, bevent);
    }
  }

  bs_function_leave();
  return;
}

void nick_clist_callback (GtkWidget *widget, gint row, gint column, 
			  GdkEventButton *bevent, BezChannelWindow *window)
{

  bs_function_enter();

  window->connection->current_window = (BezWindow *) window;

  if (window->current_channel)
    window->current_channel->selected_member = row;

  if (bevent) {
    if (bevent->button == 3) {
      nick_popup(row, bevent, window);
    }
  }

  bs_function_leave();
  return;
}
