/* cmd-valid-package-name.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "commands/valid-package-name.h"
#include "commands/version.h"
#include "libarch/archive.h"



static t_uchar * usage = N_("[options] name");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_errname, "e", "errname", 1, \
      N_("specify program name for errors")) \
  OP (opt_archive, 0, "archive", 0, \
      N_("require and explicit archive")) \
  OP (opt_no_archive, 0, "no-archive", 0, \
      N_("prohibit and explicit archive")) \
  OP (opt_category, "c", "category", 0, \
      N_("require a category")) \
  OP (opt_package, "p", "package", 0, \
      N_("require category, permit branch")) \
  OP (opt_vsn, "v", "vsn", 0, \
      N_("require a version number")) \
  OP (opt_level, "l", "patch-level", 0, \
      N_("require a patch level")) \
  OP (opt_lvl, 0, "lvl", 0, \
      N_("synonym for --patch-level")) \
  OP (opt_tolerant, "t", "tolerant", 0, \
      N_("tolerate more specific names"))

t_uchar arch_cmd_valid_package_name_help[] = N_("test a package name for validity\n"
                                              "Exit with status 0 if PACKAGE is a valid package name,\n"
                                              "status 1 otherwise.\n"
                                              "\n"
                                              "By default, require a basename or basename plus branch label.\n"
                                              "\n"
                                              "Options -v and -l cancel -b and vice versa.\n"
                                              "\n"
                                              "Option -l implies -v.\n"
                                              "\n"
                                              "If an error name is specified (-e or --errname), then invalid\n"
                                              "names cause an error message on stdout.  Otherwise, the exit\n"
                                              "status is the only output.\n"
                                              "\n"
                                              "By default, validation is strict.  For example, -b checks\n"
                                              "for a valid basename and does not permit a branch label\n"
                                              "or version number.\n"
                                              "\n"
                                              "With -t, more specific names are permitted.  For example, -b\n"
                                              "will permit a branch name, version number, and patch level.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_valid_package_name (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * errname;
  enum arch_valid_package_name_archive archive_disposition;
  enum arch_valid_package_name_archive package_name_type;
  char * type_name;
  int tolerant;
  int result;
  t_uchar * package = NULL;
  struct arch_archive * arch = 0;

  errname = 0;
  archive_disposition = arch_maybe_archive;
  package_name_type = arch_req_package;
  type_name = "package";
  tolerant = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_valid_package_name_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;


        case opt_errname:
          {
            errname = str_save (0, option->arg_string);
            break;
          }

        case opt_archive:
          {
            archive_disposition = arch_req_archive;
            break;
          }

        case opt_no_archive:
          {
            archive_disposition = arch_no_archive;
            break;
          }

        case opt_category:
          {
            package_name_type = arch_req_category;
            type_name = "category";
            break;
          }

        case opt_package:
          {
            package_name_type = arch_req_package;
            type_name = "package";
            break;
          }

        case opt_vsn:
          {
            package_name_type = arch_req_version;
            type_name = "version";
            break;
          }

        case opt_level:
        case opt_lvl:
          {
            package_name_type = arch_req_patch_level;
            type_name = "patch-level";
            break;
          }

        case opt_tolerant:
          {
            tolerant = 1;
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;
  
  if (!arch_valid_package_name (argv[1], arch_maybe_archive, arch_req_category, 1))
    {
      arch = arch_archive_connect_branch (argv[1], &package);
      if(!arch)
	{
	  safe_printfmt (2, "%s is not a valid package or url.\n", argv[1]);
	  exit (2);
	}
      arch_archive_close (arch);
    }
  else
      package = str_save (0, argv[1]);
  
  result = arch_valid_package_name (package, archive_disposition, package_name_type, tolerant);

  if (!result)
    {
      if (errname)
        {
          safe_printfmt (2, "%s: invalid %s id (%s)\n", errname, type_name, package);
        }
      exit (1);
    }

  lim_free (0, package);
  return 0;
}




/* tag: Tom Lord Mon May 12 11:13:10 2003 (valid-package-name.c)
 */
