/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef CONTAINER_H
#define CONTAINER_H

#include <kmainwindow.h>
#include <qtabbar.h>
#include <qtabwidget.h>
#include <qlabel.h>
#include <ksystemtray.h>
#include <qptrlist.h>
#include <qpoint.h>
#include <qclipboard.h>
#include <kaction.h>
#include <qpixmap.h>
#include <qdesktopwidget.h>

class QWidget;
class QPoint;
class QTimer;
class KAction;
class KToggleAction;
class QPopupMenu;
class QSignalMapper;
class QStringList;
class QToolTipGroup;
class KPassivePopup;

class DecoratedBasket;
class Basket;
class Container;

/** Class to pick a color on the screen
  * @author Sbastien Laot
  */
class DesktopColorPicker : public QDesktopWidget
{
  Q_OBJECT
  public:
	/** Construtor, initializer and destructor */
	DesktopColorPicker();
	~DesktopColorPicker();
  public slots:
	/** Begin color picking.
	  * This function returns immediatly, and pickedColor() is emitted if user has
	  * choosen a color, and not canceled the process (by pressing Escape).
	  */
	void pickColor();
  signals:
	/** When user picked a color, this signal is emitted.
	  */
	void pickedColor(const QColor &color);
	/** When user cancel a picking (by pressing Escape), this signal is emitted.
	  */
	void canceledPick();
  protected slots:
	void slotDelayedPick();
  protected:
	void mouseReleaseEvent(QMouseEvent *event);
	void keyPressEvent(QKeyEvent *event);
	bool m_gettingColorFromScreen;
};

/** This class is a QLabel that can emit a clicked() signal when clicked !
  * @author Sbastien Laot
  */
class ClickableLabel : public QLabel
{
  Q_OBJECT
  public:
	/** Construtor, initializer and destructor */
	ClickableLabel(QWidget *parent = 0, const char *name = 0)
	 : QLabel(parent, name) {}
	~ClickableLabel()       {}
  signals:
	void clicked();
  protected:
	virtual void mousePressEvent(QMouseEvent *event)
	{
		if (event->button() & Qt::LeftButton)
			emit clicked();
	}
};

/** Convenient class to develop the displayCloseMessage() dialog
  * hopefuly integrated in KDE 3.4
  * @author Sbastien Laot
  */
class KSystemTray2 : public KSystemTray
{
  Q_OBJECT
  public:
	KSystemTray2(QWidget *parent = 0, const char *name = 0);
	~KSystemTray2();
	/**
	  * Call this method when the user clicked the close button of the window
	  * (the [x]) to inform him that the application sit in the system tray
	  * and willn't be closed (as he is used to).
	  *
	  * You usualy call it from reimplemented KMainWindow::queryClose()
	  *
	  * @since 3.4
	  */
	void displayCloseMessage(QString fileMenu = "");
};

/** This class provide a personalized system tray icon.
  * @author Sbastien Laot
  */
class ContainerSystemTray : public KSystemTray2
{
  Q_OBJECT
  public:
	ContainerSystemTray(QWidget *parent = 0, const char *name = 0);
	~ContainerSystemTray();
  protected:
	void mousePressEvent(QMouseEvent *event);
	void mouseMoveEvent(QMouseEvent *event);
	void mouseReleaseEvent(QMouseEvent *event);
	virtual void dragEnterEvent(QDragEnterEvent *event);
	virtual void dragMoveEvent(QDragMoveEvent* event);
	virtual void dragLeaveEvent(QDragLeaveEvent*);
	virtual void dropEvent(QDropEvent *event);
	void wheelEvent(QWheelEvent *event);
	void enterEvent(QEvent*);
	void leaveEvent(QEvent*);
  public slots:
	void updateToolTip();
  protected slots:
	void updateToolTipDelayed();
	void showBasket(int index);
  private:
	QTimer    *m_showTimer;
	QTimer    *m_autoShowTimer;
	bool       m_canDrag;
	QPoint     m_pressPos;
	Container *m_parentContainer;
	QPixmap    m_iconPixmap;
	QPixmap    m_lockedIconPixmap;
};

/** This class provide a QTabBar with a popup menu for the baskets
  * and drag and drop possibilities.
  * @author Sbastien Laot
  */
class ContainerTabBar : public QTabBar
{
  Q_OBJECT
  public:
	ContainerTabBar(QWidget *parent = 0, const char *name = 0);
	~ContainerTabBar();
	int tabIndexForCursorPosition(const QPoint &curPos);
	void updateToolTips(const QStringList &tips);
  protected:
	virtual void dragEnterEvent(QDragEnterEvent*);
	virtual void dragMoveEvent(QDragMoveEvent *event);
	virtual void dragLeaveEvent(QDragLeaveEvent*);
	virtual void dropEvent(QDropEvent *event);
	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseMoveEvent(QMouseEvent *event);
	virtual void mouseReleaseEvent(QMouseEvent*);
	virtual void mouseDoubleClickEvent(QMouseEvent *event);
	virtual void contextMenuEvent(QContextMenuEvent *event);
  signals:
	void doubleClicked(int tabIndex);
	void wantContextMenu(int tabIndex, const QPoint &globalPos);
	void wantContextMenu(int tabIndex, const QRect  &globalRect);
	void wantShowPage(int tabIndex);
	void wantPaste(int tabIndex, QClipboard::Mode mode);
  private slots:
	void timeout();

  private:
	QTimer        *m_showTimer;
	int            m_basketIndexToMove;
	Basket        *m_pointedBasket;
	int            m_pointedBasketIndex;
	QToolTipGroup *m_ttGroup;
};

/** This class provide a QTabWidget with a popup menu for the tabs.
  * @author Sbastien Laot
  */
class ContainerTabWidget : public QTabWidget
{
  Q_OBJECT
  public:
	ContainerTabWidget(QWidget *parent = 0, const char *name = 0);
	~ContainerTabWidget();
	void updateToolTips(const QStringList &tips);
  signals:
	void doubleClicked(int tabIndex);
	void contextMenu(int tabIndex, const QPoint &globalPos);
	void contextMenu(int tabIndex, const QRect  &globalRect);
	void wantPaste(int tabIndex, QClipboard::Mode mode);
  public slots:
	void setCurrentPage(int index); // We overload it to not try to show a page that is already shown
  private slots:
	void slotWantPaste(int tabIndex, QClipboard::Mode mode);
	void slotDoubleClicked(int tabIndex);
	void slotContextMenu(int tabIndex, const QPoint &globalPos);
	void slotContextMenu(int tabIndex, const QRect  &globalRect);
  protected:
	virtual void contextMenuEvent(QContextMenuEvent *event);
};

/** The window that contain baskets, organized by tabs.
  * @author Sbastien Laot
  */
class Container : public KMainWindow
{
  Q_OBJECT
  public:
	/** Construtor, initializer and destructor */
	Container(QWidget *parent = 0, const char *name = 0);
	~Container();
  private:
	void setupActions();
	void setupStatusBar();
  public slots:
	/** GUI Main Window actions **/
	void changeStatusbar(const QString &text);
	void postStatusbarMessage(const QString &text);
	void setStatusBarHint(const QString &hint); // It's better to call setStatusBarHint(...) OR resetStatusBar()
	void resetStatusBarHint();                  //
	void setStatusBarDrag();                    //
	void setStatusBarNormal();                  //  setStatusBarNormal() AND setStatusBarEditing()
	void setStatusBarEditing();                 //  should be used only internaly
	void feedbackShiftCursor(bool shiftPressed);
	void stopHoverFeedbackBH(); // Used in OnClickAction class
	/** Baskets manipulations */
	Basket* loadBasket(const QString &folderName, bool save = true, bool showErrors = true); // Load a basket from folder and save the list, or not
	void loadBaskets();
	void save(); // To save baskets order and memorize the mirrored folders
	void changeBasketPlace(int index, int newIndex);
	/** Display updates */
	void basketNameChanged(Basket *basket, const QString &name);
	void basketIconChanged(Basket *basket, const QString &icon);
	void updateTabs();
	void updateToolTips();
	void linkLookChanged();
	void showItemsToolTipChanged();
	void tabPlacementChanged(bool onTop);
	void searchPlacementChanged(bool onTop);
	/** User actions */
	void setActive(bool active = true);
	void changeActive();
	void showAppPurpose();
	void slotDontShowEmptyHelp(bool dont);
	/** Basket */
	void askNewBasket();
	void propBasket();
	void propBasket(int index);
	void delBasket();
	void lockBasket();
	void clearCurrentStack();
	void openCurrentMirror();
	void reloadCurrentMirror();
	void exportToHTML();
	void importKNotes();
	void importKJots();
	void importStickyNotes();
	void hideOnEscape();
	bool askForQuit();
	/** Edit */
	void undo();
	void redo();
	void cutItem();
	void copyItem();
	void pasteInCurrentBasket();
	void pasteSelInCurrentBasket();
	void pasteToBasket(int index, QClipboard::Mode mode = QClipboard::Clipboard);
	void delItem();
	void slotSelectAll();
	void slotUnselectAll();
	void slotInvertSelection();
	void slotSelectCheckedItems();
	void showHideSearchBar(bool show, bool switchFocus = true);
	void slotResetSearch();
	/** Go **/
	void goToPreviousBasket();
	void goToNextBasket();
	void goToPreviousCheckedItem();
	void goToNextCheckedItem();
	void goToPreviousUncheckedItem();
	void goToNextUncheckedItem();
	/** Item */
	void editItem();
	void editItemMetaData();
	void openItem();
	void openItemWith();
	void saveItemAs();
	void checkItem();
	void moveOnTop();
	void moveOnBottom();
	void moveItemUp();
	void moveItemDown();
	/** Insert **/
	void insertEmpty(int type);
	void insertWizard(int type);
	void slotColorFromScreen(bool global = false);
	void slotColorFromScreenGlobal();
	void colorPicked(const QColor &color);
	void colorPickingCanceled();
	/** Settings **/
	void toggleMenuBar();
	void toggleToolBar();
	void toggleStatusBar();
	void showShortcutsSettingsDialog();
	void showGlobalShortcutsSettingsDialog();
	void configureToolbars();
	void configureNotifications();
	void showSettingsDialog();
  public:
	/** Necessary to the System tray popup menu, and others... */
	QPtrList<Basket> listBaskets();
	uint currentBasketIndex();
	Basket* currentBasket();
	Basket* basketAt(int index);
	DecoratedBasket* decoratedBasketAt(int index);
	DecoratedBasket* currentDecoratedBasket();
	Basket* basketForFolderName(const QString &folderName);
	QPopupMenu* popupMenu(const QString &menuName);
	QString fromICS(const QString &ics);
  private:
	QString m_passiveDroppedTitle;
	bool    m_passiveDroppedShortTypeName;
	KPassivePopup *m_passivePopup;
  public:
	static const int c_delayTooltipTime;
  public slots:
	void setCurrentBasket(int index);
	/** Global shortcuts */
	void showPassiveDropped(const QString &title, bool showTypeName = true);
		// %1 will be replaced by typeName(), unless showTypeName is false (it will then remove %1)
		// %2 with currentBasket()->name()
	void showPassiveDroppedDelayed(); // Do showPassiveDropped(), but delayed
	void showPassiveContent();
	void showPassiveImpossible(const QString &message);
	// For GUI :
	void rotateCurrentStack();
	void insertStackItemInCurrentWindow();
	void setSearching(bool searching);
  protected:
	bool queryExit();
	bool queryClose();
	void enterEvent(QEvent*);
	void leaveEvent(QEvent*);
	virtual void resizeEvent(QResizeEvent*);
	virtual void moveEvent(QMoveEvent*);
  public:
	void wheelEvent(QWheelEvent *event); // To redirect SystemTray wheelEvent
	void show();
  private slots:
	/** User actions */
	void tabChanged(QWidget*);
	void contextMenu(int tabIndex, const QPoint &globalPos);
	void contextMenu(int tabIndex, const QRect  &globalRect);
	void changedSelectedItems();
	void areSelectedItemsCheckedChanged(bool checked);
	void timeoutTryHide();
	void timeoutHide();
	void basketNumberChanged();
	void currentBasketChanged();
	void isLockedChanged();
	void countSelectedsChanged();
	void updateItemsActions();
	void rebuildBasketsMenu();
	void slotNewToolbarConfig();
  public: // TODO: Migrate to private
	KAction *actNewBasket;
	KAction *actConfigGlobalShortcuts;
	KAction *actAppConfig;
	KAction *m_actHideWindow;
	KAction *actQuit;
	QPtrList<KAction> actBasketsList;
  private:
	// Basket actions :
//	KAction       *m_actNewBasket;
	KAction       *m_actPropBasket;
	KAction       *m_actDelBasket;
	KToggleAction *m_actLockBasket;
	KAction       *m_actClearStack;
	KAction       *m_actOpenMirror;
	//KAction       *m_actReloadMirror;
//	KAction       *m_actHideWindow;
//	KAction       *m_actQuit;
	// Edit actions :
	KAction       *m_actUndo;
	KAction       *m_actRedo;
	KAction       *m_actCutItem;
	KAction       *m_actCopyItem;
	KAction       *m_actPaste;
	KAction       *m_actDelItem;
	KAction       *m_actSelectAll;
	KAction       *m_actUnselectAll;
	KAction       *m_actInvertSelection;
	KAction       *m_actSelectCheckedItems;
	KToggleAction *m_actShowSearch;
	KAction       *m_actResetSearch;
	// Go actions :
	KAction       *m_actPreviousBasket;
	KAction       *m_actNextBasket;
	QSignalMapper *m_goBasketsMapper;
	KAction       *m_actPrevChecked;
	KAction       *m_actNextChecked;
	KAction       *m_actPrevUnchecked;
	KAction       *m_actNextUnchecked;
	// Items actions :
	KAction       *m_actEditItem;
	KAction       *m_actEditItemMetaData;
	KAction       *m_actOpenItem;
	KAction       *m_actOpenItemWith;
	KAction       *m_actSaveItemAs;
	KToggleAction *m_actCheckItems;
	KAction       *m_actMoveOnTop;
	KAction       *m_actMoveItemUp;
	KAction       *m_actMoveItemDown;
	KAction       *m_actMoveOnBottom;
	// Insert actions :
	KAction       *m_actInsertText;
	KAction       *m_actInsertHtml;
	KAction       *m_actInsertLink;
	KAction       *m_actInsertImage;
	KAction       *m_actInsertColor;
	KAction       *m_actImportKMenu;
	KAction       *m_actInsertLauncher;
	KAction       *m_actImportIcon;
	KAction       *m_actLoadFile;
	KAction       *m_actMirrorFile;
	KAction       *m_actColorPicker;
	// Settings actions :
	KToggleAction *m_actShowMenubar;
	KToggleAction *m_actShowToolbar;
	KToggleAction *m_actShowStatusbar;
	//
	QPtrList<KAction> m_insertActions;
  private:
	QVBoxLayout        *m_layout;
	ContainerTabWidget *m_tab;
	int                 m_contextIndex;
	QLabel             *m_basketStatus;
	QLabel             *m_selectionStatus;
	ClickableLabel     *m_lockStatus;
	QTimer             *m_tryHideTimer;
	QTimer             *m_hideTimer;
	bool                m_startDocked;
	DesktopColorPicker *m_colorPicker;
	bool                m_colorPickWasShown;
	bool                m_colorPickWasGlobal;
};

#endif // CONTAINER_H
