/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastien.laout@tuxfamily.org                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlayout.h>
#include <qvbox.h>
#include <qwidget.h>
#include <qcheckbox.h>
#include <qsizepolicy.h>
#include <qlabel.h>
#include <qbuttongroup.h>
#include <qradiobutton.h>
#include <klineedit.h>
#include <qtextbrowser.h>
#include <kurlrequester.h>
#include <kurlcompletion.h>
#include <klocale.h>
#include <kapplication.h>
#include <kurl.h>
#include <qdir.h>
#include <qfileinfo.h>

#include "addbasketwizard.h"
#include "global.h"
#include "container.h"
#include "basketfactory.h"

/** AddBasketWizard */

AddBasketWizard::AddBasketWizard(QWidget *parent, const char *name)
 : KWizard(parent, name, true)
{
	setCaption( kapp->makeStdCaption(i18n("New Basket Wizard")) );

	setupTypePage();
	setupMirrorPage();
	setupNamePage();

	typeChanged(0);
	m_types->find(0)->setFocus();

/*	nextButton()->setDefault(false);
	nextButton()->setAutoDefault(true);
	finishButton()->setAutoDefault(true);*/
}

AddBasketWizard::~AddBasketWizard()
{
}

// Type page :

void AddBasketWizard::setupTypePage()
{
	m_typePage = new QVBox(0);
	m_typePage->setSpacing(5);

	new QLabel(i18n("Choose a type for the new basket:"), m_typePage);

	m_types = new QButtonGroup(0L);
	QHBox *hbox = new QHBox(m_typePage); // Indent the radioButtons below
	(new QWidget(hbox))->setFixedSize(20, 20);
	QWidget *options = new QVBox(hbox);

	m_types->insert( new QRadioButton(i18n("N&otes && clips"), options) );
	new QLabel(i18n("Collect various data into one place or take various notes."),           options); // 0
	m_types->insert( new QRadioButton(i18n("Chec&k list"), options) );
	new QLabel(i18n("Items can be checked to make a shop list or a TODO list."),             options); // 1
	m_types->insert( new QRadioButton(i18n("&Stack"), options) );
	new QLabel(i18n("Receive data you dropped to and that could be dragged outside later."), options); // 2
	m_types->insert( new QRadioButton(i18n("&Mirror a folder"), options) );
	new QLabel(i18n("Every file in the mirrored folder have an item representing it."),      options); // 3
	m_types->insert( new QRadioButton(i18n("&Application launchers"), options) );
	(new QLabel(i18n("Store links to applications you often use."),                          options))->setShown(false); // 4
	m_types->insert( new QRadioButton(i18n("Clip&board manager"), options) );
	(new QLabel(i18n("Manage the clipboard like with Klipper, but with stack features."),    options)) // 5
		->setEnabled( ! BasketFactory::isAlreadyAClipboard() );
	// "Folder (and quick saver) basket"
	// "Bookmarks watcher"
	// "KMenu watcher"...
	// "Live query (items from all baskets)" (display all important items, all related to..., etc...)
	// "Demo (tutorial) basket"
	addStretch(m_typePage);

	QFont font = m_types->find(0)->font();
	font.setBold(true);
	for (int i = 0; i < m_types->count(); i++)
		m_types->find(i)->setFont(font);

	m_types->find(0)->toggle();

	m_types->find(4)->setShown(false); // I perhapse willn't do it
	m_types->find(5)->setEnabled( ! BasketFactory::isAlreadyAClipboard() ); // Only one clipboard manager is allowed

	connect( m_types, SIGNAL(clicked(int)), this, SLOT(typeChanged(int)) );

	addPage(m_typePage, i18n("Basket Type"));
	setHelpEnabled(m_typePage, false);
}

void AddBasketWizard::typeChanged(int type)
{
	m_shouldChooseAnotherName = true;

//	setNextEnabled(   m_typePage, !(type == 5 && BasketFactory::isAlreadyAClipboard()) /*&& type != 4*/ ); // Alpha version
	setAppropriate(   m_mirrorPage, type == 3 );
}

int AddBasketWizard::choosenType()
{
	return m_types->selectedId();
}

// Mirror page :

void AddBasketWizard::addStretch(QWidget *parent)
{
	(new QWidget(parent))->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
}

void AddBasketWizard::setupMirrorPage()
{
	m_mirrorPage = new QVBox(0);
	m_mirrorPage->setSpacing(5);

	new QLabel(i18n("Choose a folder to mirror:"), m_mirrorPage);

	m_url = new KURLRequester(m_mirrorPage);
	m_url->completionObject()->setMode(KURLCompletion::DirCompletion);
	m_url->setMode(KFile::Directory);
	connect( m_url->lineEdit(), SIGNAL(textChanged(const QString&)), this, SLOT(mirrorChanged(const QString&)) );

	m_mirrorState = new QLabel(m_mirrorPage);

	QWidget     *wid  = new QWidget(m_mirrorPage);
	QHBoxLayout *hLay = new QHBoxLayout(wid, 0, 6);
	m_reopenMirror = new QCheckBox(i18n("Open the existing basket"), wid);
	m_reopenMirror->setChecked(true);
	hLay->addSpacing(20);
	hLay->addWidget(m_reopenMirror);

	m_onlyNewFiles = new QCheckBox(i18n("Mirror only new files"), m_mirrorPage);
	connect( m_reopenMirror, SIGNAL(clicked()), this, SLOT(mirrorReopenChanged()) );
	m_mirrorPathError = false; // ?
	//mirrorReopenChanged();

	addStretch(m_mirrorPage);
	new QLabel(i18n("<center><i>If the folder do not exists, it will be created.</i></center>"), m_mirrorPage);
	addStretch(m_mirrorPage);

	m_url->lineEdit()->setText( QDir::homeDirPath() + "/" );

	addPage(m_mirrorPage, i18n("Folder to Mirror"));
	setHelpEnabled(m_mirrorPage, false);
}

void AddBasketWizard::mirrorChanged(const QString&)
{
	KURL url(m_url->url()); // m_url->url() To get ~user or environment variables substituted for local files
	QDir dir(url.path());

	m_shouldChooseAnotherName = true;
	m_reopenMirror->setEnabled(false);
	setNextEnabled(m_mirrorPage, false);

	m_mirrorPathError = true;

	if ( ! url.isValid() ) {
		m_mirrorState->setText(
			i18n("<nobr><i><b>The path you entered is malformed.</b><br>"
			     "Please check its validity.</i></nobr>") );
	} else if ( ! url.isLocalFile() ) {
		m_mirrorState->setText(
			i18n("<nobr><i><b>The folder you entered isn't local.</b><br>"
			     "You cannot mirror folders that are not on your computer.</i></nobr>") );
	} else if ( BasketFactory::isAlreadyMirrored(m_url->url()) ) {
		m_mirrorState->setText(
			i18n("<nobr><i><b>The folder you entered is already mirrored.</b><br>"
			     "You cannot mirror a folder twice.</i></nobr>") );
	} else if ( QFileInfo(dir.path()).exists() ) {
		if ( QFileInfo(dir.path()).isFile() ) {
			m_mirrorState->setText(
				i18n("<nobr><i><b>The name you entered is a file, not a folder.</b><br>"
				     "You cannot create a basket that mirror a file.</i></nobr>") );
		} else if ( QFileInfo(dir, ".basket").exists() ) { // TODO: If .basket is a folder
			m_mirrorState->setText(
				i18n("<nobr><i>The folder exists.<br>"
				     "It contain basket data: you can reload (open) or erase it (create a new one).</i></nobr>") );
			m_reopenMirror->setEnabled(true);
			m_mirrorPathError = false;
//			setNextEnabled(m_mirrorPage, true);
		} else {
			m_mirrorState->setText(
				i18n("<nobr><i>The folder exists.<br>"
				     "The new basket will mirror the folder contents.</i></nobr>") );
			m_mirrorPathError = false;
//			setNextEnabled(m_mirrorPage, true);
		}
	} else {
		m_mirrorState->setText(
			i18n("<nobr><i>The folder doesn't exist.<br>"
			     "A new blank folder will be created and mirrored.</i></nobr>") ); // TODO: Check permissions
		m_mirrorPathError = false;
//		setNextEnabled(m_mirrorPage, true);
	}

	mirrorReopenChanged();
}

void AddBasketWizard::mirrorReopenChanged()
{
	m_onlyNewFiles->setEnabled( !m_reopenMirror->isEnabled() || !m_reopenMirror->isChecked() );
	setFinishEnabled( m_mirrorPage, !m_onlyNewFiles->isEnabled() ); // Do not give a name when reloading baskets
	setNextEnabled(   m_mirrorPage,  m_onlyNewFiles->isEnabled() && !m_mirrorPathError ); // And disable Next if finish is shown

	if (!m_onlyNewFiles->isEnabled()) // If Finish is enabled, make it the default button
		finishButton()->setDefault(true);
	else // Otherwise, make Next the default one
		nextButton()->setDefault(true);

//	nextButton()->setShown(   !m_onlyNewFiles->isEnabled()/*nextButton()->isEnabled()*/   );
//	finishButton()->setShown( m_onlyNewFiles->isEnabled() && !m_mirrorPathError/*finishButton()->isEnabled()*/ );
}

// Name page :

void AddBasketWizard::setupNamePage()
{
	m_namePage = new QVBox(0);
	m_namePage->setSpacing(5);

	new QLabel(i18n("Choose a name for the new basket:"), m_namePage);
	m_name = new KLineEdit("", m_namePage);
	addStretch(m_namePage);

	connect( m_name, SIGNAL(textChanged(const QString &)), this, SLOT(nameChanged(const QString &)) );

	addPage(m_namePage, i18n("Basket Name"));
	setHelpEnabled(m_namePage, false);
}

void AddBasketWizard::nameChanged(const QString &name)
{
	setFinishEnabled( m_namePage, ! name.isEmpty() );
}

QString AddBasketWizard::choosenName()
{
	return m_name->text();
}

void AddBasketWizard::showPage(QWidget* page)
{
	if (page == m_namePage)
		if (m_shouldChooseAnotherName) {
			m_name->setText( BasketFactory::newBasketName((BasketFactory::BasketType)choosenType(), "", m_url->url()) );
			m_shouldChooseAnotherName = false;
		}

	KWizard::showPage(page);

	// Always show Next AND Finish buttons
	//  bacause in mirror page, sometimes Finish should be enabled and then disabled
	//  but once a button is enabled for a page, it isn't hidden when disable it
	//  => cause flickers and a "dynamic UI", very bad for usability (and also not user friendly)
	// FIXME: It doesn't work !
/*	setFinishEnabled( page, true  ); // Overwize it will be shown/hidden (flicker!) sometimes
	kapp->processEvents();
	setFinishEnabled( page, false ); //  when on mirror page
	setNextEnabled(   page, true  ); // IDEM: Show it even if disabled
*/
	if (page == m_mirrorPage)
		mirrorChanged(m_url->url());
	else if (page == m_namePage) {
		m_name->setFocus();
		m_name->selectAll();
	}

	if (page == m_namePage) {
		finishButton()->setDefault(true);
		setNextEnabled( page, false ); // In case user have passed the Mirror page, canceled and use another type
	} else if (page != m_mirrorPage) // m_mirrorPage will do that by itself
		nextButton()->setDefault(true);
}

void AddBasketWizard::accept()//createBasket()
{
	QString folderName = "";
	switch (choosenType()) {
		case 0:
			folderName = BasketFactory::createBasketClips(m_name->text());
			break;
		case 1:
			folderName = BasketFactory::createBasketChecks(m_name->text());
			break;
		case 2:
			folderName = BasketFactory::createBasketStack(true, 2, m_name->text());
			break;
		case 3:
			folderName = BasketFactory::createBasketMirror(m_onlyNewFiles->isChecked(), m_reopenMirror->isChecked(),
			                                               m_name->text(), m_url->url());
			break;
		case 5:
			folderName = BasketFactory::createBasketClipboard(m_name->text());
			break;
	}
	Global::mainContainer->loadBasket(folderName);
	Global::mainContainer->setActive(true);

	QDialog::accept();
}

#include "addbasketwizard.moc"
