/*================================================================
 * preload samples in a midi file invoking sfxload program
 *
 * Copyright (C) 1996-1998 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/wait.h>
#include "util.h"
#include "midievent.h"
#include "controls.h"
#include "channel.h"

#define LOADER		"sfxload"
#define APPEND_OPT	"-N"
#define SUB_OPT		"-b1"
#define CLEAR_OPT	"-x"

typedef struct _SubRec {
	char *args;
	struct _SubRec *next;
} SubRec;

typedef struct _SFPatchRec {
	int preset, bank, keynote;
	struct _SFPatchRec *next;
} SFPatchRec;

static int nsubs;
static SubRec *psub;

static int npreloads;
static SFPatchRec *pload;
static char *fontfile;

static int exec_loader(MidiInfo *mp, char **argv);
static int sfxload_files(MidiInfo *mp, char **arglist);
static char **build_arglist(MidiInfo *mp);
static void sample_clear(MidiInfo *mp);

/* clear variables */
void init_preload(MidiInfo *mp)
{
	nsubs = 0;
	psub = NULL;

	npreloads = 0;
	pload = NULL;

	fontfile = NULL;
}

/* load fonts */
void load_partial_fonts(MidiInfo *mp)
{
	char **argv = NULL;
	SubRec *q, *nextq;
	SFPatchRec *p, *next;
	int rc = 1;

	if (!nsubs && !npreloads) {
		if (fontfile) {
			free(fontfile);
			fontfile = NULL;
		}
		return;
	}

	if (fontfile)
		argv = build_arglist(mp);
	for (p = pload; p; p = next) {
		next = p->next;
		free(p);
	}

	if (mp->incremental)
		rc = sfxload_files(mp, argv);

	if (rc) {
		sample_clear(mp);
		if (sfxload_files(mp, argv)) {
			if (ctl)
				ctl->cmsg(CMSG_INFO, -1, "preload: couldn't load all fonts");
		}
	}

	if (argv) {
		int i;
		for (i = 0; i < npreloads + 2; i++)
			safe_free(argv[i]);
	}
	for (q = psub; q; q = nextq) {
		nextq = q->next;
		free(q->args);
		free(q);
	}
	if (fontfile)
		free(fontfile);
}

/* load sub fonts and partial fonts via sfxload */
static int sfxload_files(MidiInfo *mp, char **arglist)
{
	SubRec *p;

	/* load sub fonts */
	for (p = psub; p; p = p->next) {
		int i;
		char *argv[100];

		i = 0;
		argv[i++] = LOADER;
		argv[i++] = SUB_OPT;
		if (mp->incremental)
			argv[i++] = APPEND_OPT;
		argv[i++] = p->args;
		argv[i++] = NULL;
		if (exec_loader(mp, argv))
			return 1;
	}

	/* load default fonts */
	return exec_loader(mp, arglist);
}

/* build sfxload argument list */
static char **build_arglist(MidiInfo *mp)
{
	char **argv = (char**)safe_malloc(sizeof(char*) * (npreloads + 4));
	SFPatchRec *p;
	int i = 0;

	argv[i++] = safe_strdup(LOADER);
	if (mp->incremental)
		argv[i++] = safe_strdup(APPEND_OPT);
	for (p = pload; p; p = p->next) {
		char tmp[32];
		if (p->keynote != -1)
			sprintf(tmp, "-L%d/%d/%d", p->preset, p->bank,
				p->keynote);
		else if (p->bank != 0)
			sprintf(tmp, "-L%d/%d", p->preset, p->bank);
		else
			sprintf(tmp, "-L%d", p->preset);
		argv[i++] = safe_strdup(tmp);
	}
	argv[i++] = fontfile;
	argv[i++] = NULL;

	return argv;
}

/* parse MIDI file and retrieve the required instruments */
void preload_sample(MidiInfo *mp, char *file)
{
	MidiEvent *ev;
	int i;

	if (fontfile)
		safe_free(fontfile);
	fontfile = safe_strdup(file);

	channel_init(mp);

	ev = mp->list;
	for (i = 0; i < mp->nlists; i++, ev++) {
		do_midi_event(ev, mp, EV_PRELOAD);
	}
}

/* append the instrument to list */
void add_preload(int chn, int preset, int bank, int keynote)
{
	SFPatchRec *rec;

	for (rec = pload; rec; rec = rec->next) {
		if (rec->preset == preset && rec->bank == bank &&
		    rec->keynote == keynote)
			return;
	}

	rec = (SFPatchRec*)safe_malloc(sizeof(SFPatchRec));
	rec->preset = preset;
	rec->bank = bank;
	rec->keynote = keynote;
	rec->next = pload;
	pload = rec;
	npreloads++;
}

/* call sfxload to clear samples */
static void sample_clear(MidiInfo *mp)
{
	static char *argv[] = {
		LOADER, CLEAR_OPT, NULL,
	};

	exec_loader(mp, argv);
}

/* execute sfxload: return 0 if success */
static int exec_loader(MidiInfo *mp, char **argv)
{
	int pid, rc;
	char **p;
	fprintf(stderr, "loader");
	for (p = argv; *p; p++)
		fprintf(stderr, " %s", *p);
	fprintf(stderr, "\n");

	midi_close(mp);
	if ((pid = fork()) == 0) {
		execvp(LOADER, argv);
		exit(1);
	}
	wait(&rc);
	return rc;
}

/* add the file to loading list */
void load_sub_sf(MidiInfo *mp, char *args)
{
	SubRec *p;

	p = (SubRec*)safe_malloc(sizeof(SubRec));
	p->args = safe_strdup(args);
	p->next = psub;
	psub = p;
	nsubs++;
}

