// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CRecord_h__
#define __CRecord_h__

/** \file 
 *  Semantic information about a class or union. */

#include "Puma/CStructure.h"

namespace Puma {


class CTemplateInfo;


/** \class CRecord CRecord.h Puma/CRecord.h
 *  Semantic information about a class or union. */
class CRecord : public CStructure {
  CTemplateInfo *_TemplateInfo;
  bool _isTemplate;
  bool _hiddenFwdDecl;

protected:
  /** Constructor.
   *  \param id The object type. */
  CRecord (ObjectId id);

public:
  /** Destructor. */
  ~CRecord ();

  /** Get the template information if this is a class or union template.
   *  \return The template information or NULL if not a template. */
  CTemplateInfo *TemplateInfo () const;
  /** Set the template information of a class/union template.
   *  \param info The template information. */
  void TemplateInfo (CTemplateInfo *info);

  /** Set the qualification scope of a nested class or union 
   *  or class/union declared in a namespace.
   *  \param scope The qualification scope. */
  void QualifiedScope (CStructure *scope);

  /** Check if the class/union is complete (defined)
   *  at the given source code position. The position
   *  is specified by the unique number of the CT_Token
   *  tree node representing the name of the class/union.
   *  \param pos The source code position. */
  bool isComplete (unsigned long pos = 0) const;
  /** Check if the class/union is defined. */
  bool isDefined () const;
  /** Check if the class/union is a template. */
  bool isTemplate () const;
  /** Check if this is a class/union forward declaration
   *  hidden by another declaration in the same scope. */
  bool isHiddenForwardDecl () const;

  /** Set whether this class/union is a template.
   *  \param v \e true if it is a template. */
  void isTemplate (bool v);
  /** Set whether this is a class/union forward declaration
   *  hidden by another declaration in the same scope. 
   *  \param v \e true if it is a hidden forward declaration. */
  void isHiddenForwardDecl (bool v);
};

inline CRecord::CRecord (CObjectInfo::ObjectId id) :
  CStructure (id),
  _TemplateInfo ((CTemplateInfo*)0),
  _isTemplate (false),
  _hiddenFwdDecl (false)
 {}
 
inline bool CRecord::isTemplate () const 
 { return _isTemplate; }
inline void CRecord::isTemplate (bool v)
 { _isTemplate = v; }

inline bool CRecord::isHiddenForwardDecl () const 
 { return _hiddenFwdDecl; }
inline void CRecord::isHiddenForwardDecl (bool v)
 { _hiddenFwdDecl = v; }
  
inline CTemplateInfo *CRecord::TemplateInfo () const
 { return _TemplateInfo; }
inline void CRecord::TemplateInfo (CTemplateInfo *info) 
 { _TemplateInfo = info; }

inline void CRecord::QualifiedScope (CStructure *s)
 { _QualScope = s; }
 
//inline unsigned CRecord::Referers () const
// { return _Referer.length (); }
//inline unsigned CRecord::Usages () const
// { return _Used.length (); }
//inline unsigned CRecord::Transferers () const
// { return _Transferer.length (); }
//inline unsigned CRecord::Transfers () const
// { return _Transfered.length (); }
//inline CObjectInfo *CRecord::Referer (unsigned n) const
// { return _Referer.lookup (n); }
//inline CObjectInfo *CRecord::Usage (unsigned n) const
// { return _Used.lookup (n); }
//inline CFunctionInfo *CRecord::Transferer (unsigned n) const
// { return _Transferer.lookup (n); }
//inline CObjectInfo *CRecord::Transfer (unsigned n) const
// { return _Transfered.lookup (n); }


} // namespace Puma

#endif /* __CRecord_h__ */
