// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/Unit.h"
#include "Puma/Token.h"
#include "Puma/StrCol.h"
#include <sstream>
#include <string.h>
using namespace std;

namespace Puma {


char *StrCol::dup (const char *str, int len) {
  char *res = new char[len + 1];
  memcpy(res, str, len);
  res[len] = '\0';
  return res;
}


char *StrCol::dup (const char *str) {
  return dup (str, strlen (str));
}


// Return true if character is a white space.
bool StrCol::isSpace (const char c) {
  if (c == ' '  || c == '\n' || c == '\t' || c == '\f' ||
      c == '\v' || c == '\b' || c == '\r')
    return true;
  return false;
}


// Return true if the string consists only of spaces.
bool StrCol::onlySpaces (const char *str) {
  if (! str) 
    return false;
    
  while (*str != '\0') {
    if (! isSpace (*str)) 
      return false;
    str++;
  }
  return true;
}


// Return true if the two strings differ not only in spaces.
bool StrCol::effectivelyDifferent (const char *str1, const char *str2) {
  if (! str1 || ! str2) 
    return true;

  // Skip the leading white spaces.
  while (isSpace (*str1)) str1++;
  while (isSpace (*str2)) str2++;

  while (true) {
    // Skip white spaces if there are some at these position
    // in both strings or if we are at the end of a string.
    if ((isSpace (*str1) && isSpace (*str2)) || 
        (*str1 == '\0' || *str2 == '\0')) {
      while (isSpace (*str1)) str1++;
      while (isSpace (*str2)) str2++;
    } 

    // Found a difference. Strings are different.
    if (*str1 != *str2) 
      return true;

    // Stop comparison at end of string.
    if (*str1 == '\0' || *str2 == '\0')  
      break;
        
    str1++;
    str2++;        
  }

  // Can only be reached if the both strings are not 
  // effectively different.
  return false;
}


// Build a string from an unit.
char *StrCol::buildString (Unit *unit) {
  if (! unit) 
    return (char*) 0;

  Token *token;
  std::ostringstream str;

  // Fill the return string buffer.
  for (token = (Token*) unit->first (); token; 
       token = (Token*) unit->next (token))
    str << token->text ();
    
  // Duplicate and return the string.
  return dup (str.str ().c_str ());
}


} // namespace Puma
