// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "Puma/CSyntax.h"
#include "Puma/CTokens.h"
#include "Puma/CTree.h"

namespace Puma {


void CSyntax::configure (Config &config) {
  // initialize FIRST and FOLLOW sets
  // later (when all rules have an init function) it should be sufficient
  // to call init_translation_unit()
  init_prim_expr ();
  init_cv_qual_seq ();
  init_decl_spec ();
  init_cmpd_literal ();
  init_postfix_expr1 ();
  init_identifier ();
  init_direct_declarator1 ();
}

void CSyntax::init_prim_types () {
  _prim_types.set (TOK_C_BOOL);
  _prim_types.set (TOK_CHAR);
  _prim_types.set (TOK_SHORT);
  _prim_types.set (TOK_INT);
  _prim_types.set (TOK_LONG);
  _prim_types.set (TOK_SIGNED);
  _prim_types.set (TOK_UNSIGNED);
  _prim_types.set (TOK_FLOAT);
  _prim_types.set (TOK_DOUBLE);
  _prim_types.set (TOK_WCHAR_T);
  _prim_types.set (TOK_VOID);
  _prim_types.set (TOK_UNKNOWN_T);
}
 
bool CSyntax::is_fct_def () {
  State s = token_provider->get_state ();
  bool result = false, braces = false;
  Token *current;
  int token;

  while ((current = token_provider->current ())) {
    token = current->type ();
    
    // before function parameter list
    if (! braces) {
      // only declaration
      if (token == TOK_SEMI_COLON || 
          token == TOK_COMMA ||
          token == TOK_ASSIGN ||
          token == TOK_TYPEDEF) {
        break;
      // structure definition in function return type
      } else if (token == TOK_OPEN_CURLY) {
        skip_curly_block ();
      // array delimiter in return type
      } else if (token == TOK_OPEN_SQUARE) {
        skip_block (TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
      // function parameter list
      } else if (token == TOK_OPEN_ROUND) {
        skip_round_block ();
        braces = true;
      // names etc.
      } else {
        token_provider->next ();
        locate_token ();
      }    
    // after function parameter list
    } else {
      // only declaration
      if (token == TOK_SEMI_COLON || 
          token == TOK_COMMA ||
          token == TOK_OPEN_ROUND) {
        break;
      // asm declaration specifier
      } else if (token == TOK_ASM) {
        skip ();
        skip_round_block ();
      // function body..
      } else if (token == TOK_OPEN_CURLY) {
        result = true;
        break;
      // K&R function parameter declaration
      } else {
        static int stop_tokens[] = { TOK_SEMI_COLON, 0 };
        skip (stop_tokens, true);
      }
    }
  }

  token_provider->set_state (s);
  return result;
}

bool CSyntax::is_ass_expr () {
  // result cache!
  static Token *last_token = 0;
  static bool last_result = false;
  
  Token *current = token_provider->current ();
  if (! current)
    return false;

  // return the last result if we check the same token
  if (last_token == current)
    return last_result;
  
  last_token = current;
  last_result = false;
  int token;
  State s = token_provider->get_state ();

  while ((current = token_provider->current ())) {
    token = current->type ();

    if (token == TOK_SEMI_COLON || token == TOK_COMMA ||
        token == TOK_CLOSE_CURLY || token == TOK_CLOSE_ROUND ||
        token == TOK_CLOSE_SQUARE || token == TOK_COLON ||
        token == TOK_QUESTION) {
      break;
    } else if (token == TOK_OPEN_CURLY) {
      skip_curly_block ();
    } else if (token == TOK_OPEN_ROUND) {
      skip_round_block ();
    } else if (token == TOK_OPEN_SQUARE) {
      skip_block (TOK_OPEN_SQUARE, TOK_CLOSE_SQUARE);
    } else if (token == TOK_ASSIGN || token == TOK_MUL_EQ ||
               token == TOK_DIV_EQ || token == TOK_MOD_EQ ||
               token == TOK_ADD_EQ || token == TOK_SUB_EQ ||
               token == TOK_RSH_EQ || token == TOK_LSH_EQ ||
               token == TOK_AND_EQ || token == TOK_XOR_EQ || 
               token == TOK_IOR_EQ) {
      last_result = true;
      break;
    } else {
      skip ();
    }
  }

  token_provider->set_state (s);
  return last_result;
}

/*****************************************************************************/
/*                                                                           */
/*                             C   G r a m m a r                             */
/*                                                                           */
/*---------------------------------------------------------------------------*/
/*                                                                           */
/* This grammar is derived from:                                             */
/*                                                                           */
/*                           International Standard                          */
/*                            ISO/IEC 9899-1999(E)                           */
/*                          Second edition 1999-04-01                        */
/*                          Programming languages - C                        */
/*                                                                           */
/*                                    and                                    */
/*                                                                           */
/*                           International Standard                          */
/*                            ISO/IEC 14882:1998(E)                          */
/*                          First edition 1998-09-01                         */
/*                         Programming languages - C++                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                           A . 1   K e y w o r d s                         */
/*                                                                           */
/*****************************************************************************/

void CSyntax::init_typedef_name () {
  _typedef_name_1.set (TOK_ID);
}

CTree *CSyntax::rule_typedef_name () { 
  // 1: ID
  consume (); // already checked during prediction
  return semantic ().typedef_name ();
}

bool CSyntax::typedef_name () {
  return look_ahead (TOK_ID) && parse (&CSyntax::rule_typedef_name);
}

CTree *CSyntax::private_name () { 
  return semantic ().PrivateName (); 
}

/*****************************************************************************/
/*                                                                           */
/*               A . 2   L e x i c a l  c o n v e n t i o n s                */
/*                                                                           */
/*****************************************************************************/

void CSyntax::init_identifier () {
  _identifier_1.set (TOK_ID);
}

CTree *CSyntax::rule_identifier () {
  // 1: ID
  consume (); // already checked by lookahead
  return builder ().simple_name (); 
}
   
bool CSyntax::identifier () {
  return predict_1 (_identifier_1) && parse (&CSyntax::rule_identifier);
}

void CSyntax::init_literal () {
  init_cmpd_str ();
  _literal_1 = _cmpd_str_1;
  _literal_1.set (TOK_INT_VAL);
  _literal_1.set (TOK_ZERO_VAL);
  _literal_1.set (TOK_CHAR_VAL);
  _literal_1.set (TOK_FLT_VAL);
}
  
CTree *CSyntax::rule_literal () {
  // 1: cmpd_str
  // 1: TRUE
  // 1: FALSE
  // 1: 0
  // 1: ..
  if (!cmpd_str ())
    consume (); // already checked by lookahead
  return builder ().literal (); 
}

bool CSyntax::literal () {
  return predict_1 (_literal_1) && parse (&CSyntax::rule_literal);
}

void CSyntax::init_cmpd_str () {
  init_str_literal ();
  _cmpd_str_1 = _str_literal_1;
}

CTree *CSyntax::rule_cmpd_str () {
  // 1+: str_literal...
  return seq (&CSyntax::str_literal) ? 
    builder ().cmpd_str () : (CTree*)0; 
}

bool CSyntax::cmpd_str () {
  return predict_1 (_cmpd_str_1) && parse (&CSyntax::rule_cmpd_str);
}

void CSyntax::init_str_literal () {
  _str_literal_1.set (TOK_STRING_VAL);
}

CTree *CSyntax::rule_str_literal () {
  // 1: ".."
  consume (); // checked by lookahead ();
  return builder ().str_literal (); 
}

bool CSyntax::str_literal () {
  return predict_1 (_str_literal_1) && parse (&CSyntax::rule_str_literal);
}

/*****************************************************************************/
/*                                                                           */
/*                   A . 3   B a s i c  c o n c e p t s                      */
/*                                                                           */
/*****************************************************************************/

CTree *CSyntax::trans_unit () {
  // 1: decl_seq
  return parse (&CSyntax::decl_seq) ? 
    semantic ().trans_unit () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                      A . 4   E x p r e s s i o n s                        */
/*                                                                           */
/*****************************************************************************/

void CSyntax::init_prim_expr () {
  init_literal ();
  init_id_expr ();
  _prim_expr_1 = _literal_1;
  _prim_expr_1 |= _id_expr_1;
  _prim_expr_1.set (TOK_OPEN_ROUND);
}

CTree *CSyntax::rule_prim_expr () {
  // 1: literal
  // 1: id_expr
  // 3: (  expr  )
  return (literal () || 
          id_expr () ||
          (parse (TOK_OPEN_ROUND) && 
           parse (&CSyntax::expr) && 
           parse (TOK_CLOSE_ROUND))) ? 
    builder ().prim_expr () : (CTree*)0; 
}

bool CSyntax::prim_expr () {
  return predict_1 (_prim_expr_1) && parse (&CSyntax::rule_prim_expr);
}

void CSyntax::init_id_expr () {
  _id_expr_1.set (TOK_ID);
}

CTree *CSyntax::rule_id_expr () {
  // 1: ID
  consume (); // already checked
  return semantic ().id_expr (); 
}

bool CSyntax::id_expr () {
  return look_ahead () == TOK_ID && parse (&CSyntax::rule_id_expr);
}

void CSyntax::init_cmpd_literal () {
  _cmpd_literal_1.set (TOK_OPEN_ROUND);
}

CTree *CSyntax::rule_cmpd_literal () { 
  // 5 : (  type_id  )  {  }
  // 6 : (  type_id  )  {  init_list  }
  consume (); // already checked by lookahead
  return (parse (&CSyntax::type_id) && 
          parse (TOK_CLOSE_ROUND) && 
          parse (TOK_OPEN_CURLY) && opt (parse (&CSyntax::init_list)) && 
          parse (TOK_CLOSE_CURLY)) ?
    builder ().cmpd_literal () : (CTree*)0;
}

bool CSyntax::cmpd_literal () {
  return predict_1 (_cmpd_literal_1) && parse (&CSyntax::rule_cmpd_literal);
}

CTree *CSyntax::postfix_expr () { 
  // 1 : cmpd_literal
  // 1+: cmpd_literal  postfix_expr1...
  // 1 : prim_expr
  // 1+: prim_expr  postfix_expr1...
  return ((cmpd_literal () ||
           prim_expr ()) &&
          opt (seq (&CSyntax::postfix_expr1))) ?
    builder ().postfix_expr () : (CTree*)0;
}

void CSyntax::init_postfix_expr1 () {
  _postfix_expr1_1.set (TOK_DECR);
  _postfix_expr1_1.set (TOK_INCR);
  _postfix_expr1_1.set (TOK_DOT);
  _postfix_expr1_1.set (TOK_PTS);
  _postfix_expr1_1.set (TOK_OPEN_ROUND);
  _postfix_expr1_1.set (TOK_OPEN_SQUARE);
}
  
CTree *CSyntax::rule_postfix_expr1 () { 
  // 1: --
  // 1: ++
  // 2: .  identifier
  // 2: ->  identifier
  // 2: (  )
  // 3: (  expr_list  )
  // 3: [  expr  ]
  return (parse (TOK_OPEN_ROUND) ?
           (parse (&CSyntax::expr_list), parse (TOK_CLOSE_ROUND)) :
          parse (TOK_OPEN_SQUARE) ?
           (parse (&CSyntax::expr) && parse (TOK_CLOSE_SQUARE)) :
          (parse (TOK_DECR) || parse (TOK_INCR) ||
           ((parse (TOK_DOT) || parse (TOK_PTS)) && 
            identifier ()))) ?
    builder ().postfix_expr1 () : (CTree*)0;
}

bool CSyntax::postfix_expr1 () {
  return predict_1 (_postfix_expr1_1) && parse (&CSyntax::rule_postfix_expr1);
}

CTree *CSyntax::expr_list () { 
  // 1 : ass_expr
  // 3+: ass_expr  ,  ass_expr ...
  return list (&CSyntax::ass_expr, TOK_COMMA) ? 
    builder ().expr_list () : (CTree*)0;
}

CTree *CSyntax::unary_expr () { 
  // 1: postfix_expr
  // 2: --  unary_expr
  // 2: ++  unary_expr
  // 2: any_unary_op  cast_expr
  // 2: SIZEOF  unary_expr
  // 4: SIZEOF  unary_expr1
  static int any_unary_op[] = { TOK_AND, TOK_MUL, TOK_PLUS, TOK_MINUS, 
                                TOK_TILDE, TOK_NOT, 0 };
  return (parse (&CSyntax::postfix_expr) ||
          ((parse (TOK_DECR) || parse (TOK_INCR)) ?
            parse (&CSyntax::unary_expr) : 
           parse (any_unary_op) ? 
            parse (&CSyntax::cast_expr) :
           parse (TOK_SIZEOF) && 
            (parse (&CSyntax::unary_expr1) || 
             parse (&CSyntax::unary_expr)))) ?
    builder ().unary_expr () : (CTree*)0;
}

CTree *CSyntax::unary_expr1 () { 
  // 3: (  type_id  )
  return (parse (TOK_OPEN_ROUND) && parse (&CSyntax::type_id) && 
          parse (TOK_CLOSE_ROUND)) ?
    builder ().unary_expr1 () : (CTree*)0;
}

CTree *CSyntax::cast_expr () { 
  // 1: cast_expr1
  // 1: unary_expr
  return (parse (&CSyntax::cast_expr1) || parse (&CSyntax::unary_expr)) ?
    builder ().cast_expr () : (CTree*)0;
}

CTree *CSyntax::cast_expr1 () { 
  // 2+: cast_expr2... unary_expr
  return (seq (&CSyntax::cast_expr2) && parse (&CSyntax::unary_expr)) ?
    builder ().cast_expr1 () : (CTree*)0;
}

CTree *CSyntax::cast_expr2 () { 
  // 3: (  type_id  )
  return (parse (TOK_OPEN_ROUND) && parse (&CSyntax::type_id) && 
          parse (TOK_CLOSE_ROUND)) ?
    builder ().cast_expr2 () : (CTree*)0;
}

CTree *CSyntax::mul_expr () { 
  // 1 : cast_expr
  // 3+: cast_expr  any_mul_op  cast_expr ...
  static int any_mul_op[] = { TOK_MUL, TOK_DIV, TOK_MODULO, 0 };
  return list (&CSyntax::cast_expr, any_mul_op) ? 
    builder ().mul_expr () : (CTree*)0;
}

CTree *CSyntax::add_expr () { 
  // 1 : mul_expr
  // 3+: mul_expr  any_add_op  mul_expr ...
  static int any_add_op[] = { TOK_PLUS, TOK_MINUS, 0 };
  return list (&CSyntax::mul_expr, any_add_op) ? 
    builder ().add_expr () : (CTree*)0;
}

CTree *CSyntax::shift_expr () { 
  // 1 : add_expr
  // 3+: add_expr  any_shift_op  add_expr ...
  static int any_shift_op[] = { TOK_LSH, TOK_RSH, 0 };
  return list (&CSyntax::add_expr, any_shift_op) ? 
    builder ().shift_expr () : (CTree*)0;
}

CTree *CSyntax::rel_expr () { 
  // 1 : shift_expr
  // 3+: shift_expr  any_rel_op  shift_expr ...
  static int any_rel_op[] = { TOK_LESS, TOK_GREATER, TOK_LEQ, TOK_GEQ, 0 };
  return list (&CSyntax::shift_expr, any_rel_op) ? 
    builder ().rel_expr () : (CTree*)0;
}

CTree *CSyntax::equ_expr () { 
  // 1 : rel_expr
  // 3+: rel_expr  any_equ_op  rel_expr ...
  static int any_equ_op[] = { TOK_EQL, TOK_NEQ, 0 };
  return list (&CSyntax::rel_expr, any_equ_op) ? 
    builder ().equ_expr () : (CTree*)0;
}

CTree *CSyntax::and_expr () { 
  // 1 : equ_expr
  // 3+: equ_expr  &  equ_expr ...
  return list (&CSyntax::equ_expr, TOK_AND) ? 
    builder ().and_expr () : (CTree*)0;
}

CTree *CSyntax::excl_or_expr () { 
  // 1 : and_expr
  // 3+: and_expr  ^  and_expr ...
  return list (&CSyntax::and_expr, TOK_ROOF) ? 
    builder ().excl_or_expr () : (CTree*)0;
}

CTree *CSyntax::incl_or_expr () { 
  // 1 : excl_or_expr
  // 3+: excl_or_expr  |  excl_or_expr ...
  return list (&CSyntax::excl_or_expr, TOK_OR) ? 
    builder ().incl_or_expr () : (CTree*)0;
}

CTree *CSyntax::log_and_expr () { 
  // 1 : incl_or_expr
  // 3+: incl_or_expr  &&  incl_or_expr ...
  return list (&CSyntax::incl_or_expr, TOK_AND_AND) ? 
    builder ().log_and_expr () : (CTree*)0;
}

CTree *CSyntax::log_or_expr () { 
  // 1 : log_and_expr
  // 3+: log_and_expr  ||  log_and_expr ...
  return list (&CSyntax::log_and_expr, TOK_OR_OR) ? 
    builder ().log_or_expr () : (CTree*)0;
}

CTree *CSyntax::cond_expr () { 
  // 1: log_or_expr
  // 5: log_or_expr  ?  expr  :  cond_expr
  return (parse (&CSyntax::log_or_expr) && 
          (parse (TOK_QUESTION) ? (parse (&CSyntax::expr) && 
                                   parse (TOK_COLON) && 
                                   parse (&CSyntax::cond_expr)) : true)) ?
    builder ().cond_expr () : (CTree*)0;
}

CTree *CSyntax::ass_expr () { 
  // 1 : cond_expr
  // 2+: ass_expr1...  cond_expr
  return (seq (&CSyntax::ass_expr1), parse (&CSyntax::cond_expr)) ? 
    builder ().ass_expr () : (CTree*)0;
}

CTree *CSyntax::ass_expr1 () { 
  // 2: unary_expr  any_ass_op
  static int any_ass_op[] = { TOK_ASSIGN, TOK_MUL_EQ, TOK_DIV_EQ, TOK_MOD_EQ, 
                              TOK_ADD_EQ, TOK_SUB_EQ, TOK_RSH_EQ, TOK_LSH_EQ,
                              TOK_AND_EQ, TOK_XOR_EQ, TOK_IOR_EQ, 0 };
  return (is_ass_expr () && parse (&CSyntax::unary_expr) && parse (any_ass_op)) ? 
    builder ().ass_expr1 () : (CTree*)0;
}

CTree *CSyntax::expr () { 
  // 1 : ass_expr
  // 3+: ass_expr  ,  ass_expr ...
  return list (&CSyntax::ass_expr, TOK_COMMA) ? 
    builder ().expr () : (CTree*)0;
}

CTree *CSyntax::const_expr () { 
  // 1: cond_expr
  return parse (&CSyntax::cond_expr) ? 
    builder ().const_expr () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                      A . 5   S t a t e m e n t s                          */
/*                                                                           */
/*****************************************************************************/

CTree *CSyntax::stmt () { 
  // 1: label_stmt
  // 1: expr_stmt
  // 1: cmpd_stmt
  // 1: select_stmt
  // 1: iter_stmt
  // 1: jump_stmt
  return (parse (&CSyntax::label_stmt) ||
          parse (&CSyntax::expr_stmt) ||
          parse (&CSyntax::cmpd_stmt) ||
          parse (&CSyntax::select_stmt) ||
          parse (&CSyntax::iter_stmt) ||
          parse (&CSyntax::jump_stmt)) ? 
    builder ().stmt () : (CTree*)0; 
}
           
CTree *CSyntax::label_stmt () {
  // 3: identifier  :  stmt 
  // 3: DEFAULT  :  stmt
  // 4: CASE  const_expr  :  stmt
  return ((identifier () ||
           parse (TOK_DEFAULT) || 
           (parse (TOK_CASE) && parse (&CSyntax::const_expr))) && 
          parse (TOK_COLON) && parse (&CSyntax::stmt)) ?
    semantic ().introduce_label () : (CTree*)0;
}

CTree *CSyntax::expr_stmt () {
  // 1: ;
  // 2: expr  ;
  return (parse (&CSyntax::expr), parse (TOK_SEMI_COLON)) ? 
    builder ().expr_stmt () : (CTree*)0;
}

CTree *CSyntax::cmpd_stmt () {
  // 2: {  }
  // 3: {  stmt_seq  }
  if (! parse (TOK_OPEN_CURLY))
    return (CTree*)0;
  semantic ().enter_local_scope (); // enter local (block) scope
  return (parse (&CSyntax::stmt_seq), parse (TOK_CLOSE_CURLY)) ? 
    semantic ().cmpd_stmt () : (CTree*)0;
}

CTree *CSyntax::stmt_seq () {
  // 1+: stmt..
  int skip[] = { TOK_SEMI_COLON, TOK_CLOSE_CURLY, 0 };
  int finish[] = { TOK_CLOSE_CURLY, 0 };

  if (! (parse (&CSyntax::block_decl) || catch_error (&CSyntax::stmt, 
         "invalid statement or declaration", finish, skip)))
    return (CTree*)0;

  while (parse (&CSyntax::block_decl) || catch_error (&CSyntax::stmt, 
         "invalid statement or declaration", finish, skip));
  return builder ().stmt_seq ();
}

CTree *CSyntax::select_stmt () {
  // 5: SWITCH  (  condition  )  sub_stmt
  // 5: IF  (  condition  )  sub_stmt
  // 7: IF  (  condition  )  sub_stmt  ELSE  sub_stmt
  bool is_if_stmt = parse (TOK_IF);
  if  (! ((is_if_stmt || parse (TOK_SWITCH)) && parse (TOK_OPEN_ROUND)))
    return (CTree*)0;
  semantic ().enter_local_scope (); // enter statement scope
  return (parse (&CSyntax::condition) && parse (TOK_CLOSE_ROUND) && 
          parse (&CSyntax::sub_stmt) &&
          ((is_if_stmt && parse (TOK_ELSE)) ? // 'else' part
           parse (&CSyntax::sub_stmt) : true)) ?
    semantic ().select_stmt () : (CTree*)0;
}

CTree *CSyntax::sub_stmt () {
  // 1: stmt
  return parse (&CSyntax::stmt) ?
    builder ().sub_stmt () : (CTree*)0; 
}

CTree *CSyntax::condition () {
  // 1: expr
  return (parse (&CSyntax::expr)) ? 
    builder ().condition () : (CTree*)0;
}

CTree *CSyntax::iter_stmt () {
  // 5: WHILE  (  condition  )  sub_stmt
  // 6: FOR  (  for_init_stmt  ;  )  sub_stmt
  // 7: DO  sub_stmt  WHILE  (  expr  )  ;
  // 7: FOR  (  for_init_stmt  ;  expr  )  sub_stmt
  // 7: FOR  (  for_init_stmt  condition  ;  )  sub_stmt
  // 8: FOR  (  for_init_stmt  condition  ;  expr  )  sub_stmt
  if (parse (TOK_DO)) 
    return (parse (&CSyntax::sub_stmt) && 
            parse (TOK_WHILE) && parse (TOK_OPEN_ROUND) && 
            parse (&CSyntax::expr) && parse (TOK_CLOSE_ROUND) && 
            parse (TOK_SEMI_COLON)) ? 
      builder ().iter_stmt () : (CTree*)0;
  semantic ().enter_local_scope (); // enter statement scope
  return ((parse (TOK_WHILE) ? 
           (parse (TOK_OPEN_ROUND) && parse (&CSyntax::condition)) :
           (parse (TOK_FOR) && parse (TOK_OPEN_ROUND) && 
            parse (&CSyntax::for_init_stmt) && 
            opt (parse (&CSyntax::condition)) && 
            parse (TOK_SEMI_COLON) && opt (parse (&CSyntax::expr)))) &&
          parse (TOK_CLOSE_ROUND) && parse (&CSyntax::sub_stmt)) ?
    semantic ().iter_stmt () : (CTree*)0;
}

CTree *CSyntax::for_init_stmt () {
  // 1: simple_decl
  // 1: expr_stmt
  return (parse (&CSyntax::simple_decl) ||
          parse (&CSyntax::expr_stmt)) ?
    builder ().for_init_stmt () : (CTree*)0;
}

CTree *CSyntax::jump_stmt () {
  // 2: BREAK  ;
  // 2: CONTINUE  ;
  // 2: RETURN  ;
  // 3: RETURN  expression  ;
  // 3: GOTO  identifier  ;
  return ((parse (TOK_BREAK) || 
           parse (TOK_CONTINUE) ||
           (parse (TOK_RETURN) && opt (parse (&CSyntax::expr))) ||
           (parse (TOK_GOTO) && identifier ())) &&
          parse (TOK_SEMI_COLON)) ?
    builder ().jump_stmt () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                     A . 6   D e c l a r a t i o n s                       */
/*                                                                           */
/*****************************************************************************/

CTree *CSyntax::decl_seq () { 
  // 1+: decl
  int skip[] = { TOK_SEMI_COLON, TOK_CLOSE_CURLY, 0 };
  int finish[] = { TOK_CLOSE_CURLY, 0 };

  if (! catch_error (&CSyntax::decl, "invalid declaration", finish, skip))
    return (CTree*)0;

  while (catch_error (&CSyntax::decl, "invalid declaration", finish, skip));
  return builder ().decl_seq ();
}

CTree *CSyntax::decl () { 
  // 1: block_decl
  // 1: fct_def
  return (is_fct_def () ? parse (&CSyntax::fct_def) :
          (parse (&CSyntax::block_decl) ||
           parse (&CSyntax::fct_def))) ? 
    builder ().decl () : (CTree*)0;
}

CTree *CSyntax::block_decl () { 
  // 1: simple_decl
  // 1: asm_def
  return (parse (&CSyntax::simple_decl) ||
          parse (&CSyntax::asm_def))? 
    builder ().block_decl () : (CTree*)0;
}

CTree *CSyntax::simple_decl () { 
  // 1: ;
  // 2: decl_spec_seq  ;
  // 3: decl_spec_seq  init_declarator_list  ;
  if (parse (TOK_SEMI_COLON) ||
      ((parse (&CSyntax::decl_spec_seq) || semantic ().implicit_int ()) && 
       ((! look_ahead (TOK_SEMI_COLON) && 
         parse (&CSyntax::init_declarator_list)),
        parse (TOK_SEMI_COLON)))) {
    semantic ().finish_decl ();
    return builder ().simple_decl ();
  }
  return (CTree*)0;
}

void CSyntax::init_decl_spec () {
  init_storage_class_spec ();
  init_type_spec ();
  init_fct_spec ();
  init_misc_spec ();
  _decl_spec_1 = _storage_class_spec_1;
  _decl_spec_1 |= _type_spec_1;
  _decl_spec_1 |= _fct_spec_1;
  _decl_spec_1 |= _misc_spec_1;
}

CTree *CSyntax::rule_decl_spec () { 
  // 1: storage_class_spec
  // 1: type_spec
  // 1: fct_spec
  // 1: misc_spec
  return (storage_class_spec () ||
          type_spec () ||
          fct_spec () ||
          misc_spec ()) ? 
    builder ().decl_spec () : (CTree*)0; 
}

bool CSyntax::decl_spec () {
  return predict_1 (_decl_spec_1) && parse (&CSyntax::rule_decl_spec);
}

void CSyntax::init_misc_spec () {
  _misc_spec_1.set (TOK_TYPEDEF);
}

CTree *CSyntax::rule_misc_spec () { 
  // 1: TYPEDEF
  consume (); // already checked by lookahead
  return builder ().misc_spec ();
}

bool CSyntax::misc_spec () {
  return predict_1 (_misc_spec_1) && parse (&CSyntax::rule_misc_spec);
}

CTree *CSyntax::decl_spec_seq () { 
  // 1+: decl_spec_seq1...
  semantic ().decl_spec_seq ();
  return seq (&CSyntax::decl_spec_seq1) ?
    semantic ().begin_decl () : semantic ().decl_spec_seq_err ();
}

CTree *CSyntax::decl_spec_seq1 () { 
  // 1: decl_spec
  return decl_spec () ?
    semantic ().decl_spec_seq1 () : (CTree*)0;
}

void CSyntax::init_storage_class_spec () {
  _storage_class_spec_1.set (TOK_AUTO);
  _storage_class_spec_1.set (TOK_REGISTER);
  _storage_class_spec_1.set (TOK_STATIC);
  _storage_class_spec_1.set (TOK_EXTERN);
}

CTree *CSyntax::rule_storage_class_spec () { 
  // 1: storage_class_specs
  consume (); // already checked by lookahead
  return builder ().storage_class_spec ();
}

bool CSyntax::storage_class_spec () { 
  return predict_1 (_storage_class_spec_1) &&
         parse (&CSyntax::rule_storage_class_spec);
}

void CSyntax::init_fct_spec () {
  _fct_spec_1.set (TOK_INLINE);
}

CTree *CSyntax::rule_fct_spec () { 
  // 1: INLINE
  consume (); // already check token in lookahead
  return builder ().fct_spec ();
}

bool CSyntax::fct_spec () {
  return predict_1 (_fct_spec_1) && parse (&CSyntax::rule_fct_spec);
}

void CSyntax::init_type_spec () {
  init_simple_type_spec ();
  init_class_spec ();
  init_enum_spec ();
  init_elaborated_type_spec ();
  init_cv_qual ();
  _type_spec_1 = _simple_type_spec_1;
  _type_spec_1 |= _class_spec_1;
  _type_spec_1 |= _enum_spec_1;
  _type_spec_1 |= _elaborated_type_spec_1;
  _type_spec_1 |= _cv_qual_1;
}

CTree *CSyntax::rule_type_spec () { 
  // 1: simple_type_spec
  // 1: class_spec
  // 1: enum_spec
  // 1: elaborated_type_spec
  // 1: cv_qual
  return (simple_type_spec () ||
          class_spec () ||
          enum_spec () ||
          elaborated_type_spec() ||
          cv_qual ()) ? 
    builder ().type_spec () : (CTree*)0;
}

bool CSyntax::type_spec () {
  return predict_1 (_type_spec_1) && parse (&CSyntax::rule_type_spec);
}

void CSyntax::init_simple_type_spec () {
  init_type_name ();
  init_prim_types ();
  _simple_type_spec_1 = _type_name_1;
  _simple_type_spec_1 |= _prim_types;
}
  
CTree *CSyntax::rule_simple_type_spec () { 
  // 1: type_name
  // 1: prim_types
  return ((_prim_types[look_ahead ()] && consume ()) || type_name ()) ?
    builder ().simple_type_spec () : (CTree*)0;
}

bool CSyntax::simple_type_spec () {
  return predict_1 (_simple_type_spec_1) && 
         parse (&CSyntax::rule_simple_type_spec);
}

void CSyntax::init_type_name () {
  init_typedef_name ();
  _type_name_1 = _typedef_name_1;
}

CTree *CSyntax::rule_type_name () { 
  // 1: typedef_name
  return typedef_name () ? builder ().type_name () : (CTree*)0;
}   

bool CSyntax::type_name () {
  return predict_1 (_type_name_1) && parse (&CSyntax::rule_type_name);
}

void CSyntax::init_elaborated_type_spec () {
  init_class_key ();
  _elaborated_type_spec_1 = _class_key_1;
  _elaborated_type_spec_1.set (TOK_ENUM);
}

CTree *CSyntax::rule_elaborated_type_spec () {
  // 2: class_key  identifier
  // 2: ENUM  identifier
  return ((class_key () || parse (TOK_ENUM)) && identifier ()) ?
    semantic ().introduce_tag () : (CTree*)0;
}

bool CSyntax::elaborated_type_spec () {
  return predict_1 (_elaborated_type_spec_1) && 
         parse (&CSyntax::rule_elaborated_type_spec);
}

void CSyntax::init_enum_spec () {
  _enum_spec_1.set (TOK_ENUM);
}

CTree *CSyntax::rule_enum_spec () { 
  // 5: ENUM  identifier  {  enumerator_list  }
  // 5: ENUM  private_name  {  enumerator_list  }
  if (! (consume () && 
         (identifier () || parse (&CSyntax::private_name)) &&
         look_ahead (TOK_OPEN_CURLY)))
    return (CTree*)0;
  consume ();
  // create and introduce empty enum...
  CTree *es = semantic ().introduce_enum ();
  if (! es)
    return (CTree*)0;
  return (parse (&CSyntax::enumerator_list) && parse (TOK_CLOSE_CURLY)) ? 
    builder ().enum_spec (es) : builder ().enum_spec_err (es);
}

bool CSyntax::enum_spec () {
  return predict_1 (_enum_spec_1) && parse (&CSyntax::rule_enum_spec);
}

CTree *CSyntax::enumerator_list () { 
  // 1 : enumerator_def
  // 2 : enumerator_def  ,
  // 3+: enumerator_def  ,  enumerator_def ...
  // 4+: enumerator_def  ,  enumerator_def ...  , 
  return list (&CSyntax::enumerator_def, TOK_COMMA, true) ?
    builder ().enumerator_list () : (CTree*)0;
}

CTree *CSyntax::enumerator_def () { 
  // 1: enumerator
  // 3: enumerator  =  const_expr
  return (parse (&CSyntax::enumerator) && 
          (parse (TOK_ASSIGN) ? parse (&CSyntax::const_expr) : true)) ?
    builder ().enumerator_def () : (CTree*)0;
}

CTree *CSyntax::enumerator () {
  // 1: identifier
  return identifier () ?
    semantic ().introduce_enumerator () : (CTree*)0;
}

CTree *CSyntax::asm_def () {
  // 1: ASM  (  str_literal  )  ;
  return (parse (TOK_ASM) && 
          parse (TOK_OPEN_ROUND) && str_literal () && 
          parse (TOK_CLOSE_ROUND) && parse (TOK_SEMI_COLON)) ?
    builder ().asm_def () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                     A . 7   D e c l a r a t o r s                         */
/*                                                                           */
/*****************************************************************************/

CTree *CSyntax::init_declarator_list () { 
  // 1 : init_declarator
  // 3+: init_declarator  ,  init_declarator ...
  return list (&CSyntax::init_declarator, TOK_COMMA) ?
    builder ().init_declarator_list () : (CTree*)0;
}

// this is only a hook for optional extension introduced by aspects
// TODO: better decouple syntax and semantic completely and weave advice
//       without hook
CTree *CSyntax::init_declarator_ext () {
  return 0; 
}

CTree *CSyntax::init_declarator () { 
  // 1: declarator
  // 2: declarator  init
  // 2: declarator  init_declarator_ext_list
  // 3: declarator  init_declarator_ext_list init
  if (! parse (&CSyntax::declarator)) {
    semantic ().finish_declarator ();
    return (CTree*)0;
  }
  
  // parse optional init declarator extensions, such as GNU asm("symbol")
  parse (&CSyntax::init_declarator_ext);

  // first introduce the new name (if any) 
  // before parsing the intializer
  CTree* id;
  if (semantic ().in_arg_decl_seq ())
    id = semantic ().declare_parameter ();
  else
    id = semantic ().introduce_object ();
  semantic ().finish_declarator ();
  if (! id) 
    return (CTree*)0;
  
  parse (&CSyntax::init);
  return semantic ().init_declarator (id);
}

CTree *CSyntax::declarator () { 
  // 1 : direct_declarator
  // 2+: ptr_operator..  direct_declarator
  return (opt (seq (&CSyntax::ptr_operator)) &&
          parse (&CSyntax::direct_declarator)) ?
    builder ().declarator () : (CTree*)0;
}

CTree *CSyntax::direct_declarator () { 
  // 1 : declarator_id
  // 2+: declarator_id  direct_declarator1...
  // 3 : (  declarator  ) 
  // 3+: (  declarator  )  direct_declarator1...
  return ((parse (&CSyntax::declarator_id) ||
           (parse (TOK_OPEN_ROUND) && 
            parse (&CSyntax::declarator) &&
            parse (TOK_CLOSE_ROUND))) &&  
          opt (seq (&CSyntax::direct_declarator1))) ? 
    builder ().direct_declarator () : (CTree*)0;
}

void CSyntax::init_direct_declarator1 () {
  _direct_declarator1_1.set (TOK_OPEN_SQUARE);
  _direct_declarator1_1.set (TOK_OPEN_ROUND);
}

CTree *CSyntax::rule_direct_declarator1 () { 
  // 3: [  array_delim  ]
  // 3: (  identifier_list  )
  // 3: (  param_decl_clause  )
  return ((look_ahead () == TOK_OPEN_ROUND && consume ()) ? 
          ((parse (&CSyntax::identifier_list) ||
            parse (&CSyntax::param_decl_clause)) &&
           parse (TOK_CLOSE_ROUND)) : 
          (parse (TOK_OPEN_SQUARE) && 
           parse (&CSyntax::array_delim) &&
           parse (TOK_CLOSE_SQUARE))) ?
    builder ().direct_declarator1 () : (CTree*)0;
}

bool CSyntax::direct_declarator1 () {
  int next = look_ahead ();
  return (next == TOK_OPEN_ROUND || next == TOK_OPEN_SQUARE) &&
    parse (&CSyntax::rule_direct_declarator1);
}

CTree *CSyntax::identifier_list () {
  // 1 : identifier
  // 3+: identifier  ,  identifier ...
  semantic ().enter_param_decl_clause (); // enter function prototype scope
  bool success = (list (&CSyntax::identifier, TOK_COMMA));
  semantic ().leave_param_decl_clause ();
  return success ?
    semantic ().identifier_list () : (CTree*)0;
}

CTree *CSyntax::array_delim () { 
  // 0:
  // 1: *
  // 1: ass_expr
  // 1: cv_qual_seq
  // 2: cv_qual_seq  *
  // 2: cv_qual_seq  ass_expr
  // 2: STATIC  ass_expr
  // 3: STATIC  cv_qual_seq  ass_expr
  // 3: cv_qual_seq  STATIC  ass_expr
  bool in_param_decl = semantic ().in_param_decl_clause (); 
  bool have_cv_qual = in_param_decl && cv_qual_seq ();
  bool is_static = in_param_decl && parse (TOK_STATIC);
  if (parse (TOK_MUL)) 
    return builder ().array_delim ();
  if (! have_cv_qual && is_static)
    cv_qual_seq ();
  if (! parse (&CSyntax::ass_expr) && is_static)
    return (CTree*)0;
  return semantic ().array_delim ();
}

CTree *CSyntax::ptr_operator () { 
  // 1: *
  // 2: *  cv_qual_seq
  return (parse (TOK_MUL) && opt (cv_qual_seq ())) ? 
    builder ().ptr_operator () : (CTree*)0;
}

void CSyntax::init_cv_qual_seq () {
  init_cv_qual ();
  _cv_qual_seq_1 = _cv_qual_1;
}

CTree *CSyntax::rule_cv_qual_seq () { 
  // 1+: cv_qual...
  return seq (&CSyntax::cv_qual) ? 
    builder ().cv_qual_seq () : (CTree*)0;
}

bool CSyntax::cv_qual_seq () {
  return predict_1 (_cv_qual_seq_1) && parse (&CSyntax::rule_cv_qual_seq);
}

void CSyntax::init_cv_qual () {
  _cv_qual_1.set (TOK_CONST);
  _cv_qual_1.set (TOK_VOLATILE);
  _cv_qual_1.set (TOK_RESTRICT);
}

CTree *CSyntax::rule_cv_qual () { 
  // 1: type_quals
  consume (); // checked during the prediction
  return builder ().cv_qual ();
}

bool CSyntax::cv_qual () {
  return predict_1 (_cv_qual_1) && parse (&CSyntax::rule_cv_qual);
}

CTree *CSyntax::declarator_id () { 
  // 1: identifier
  return identifier () ? 
    builder ().declarator_id () : (CTree*)0;
}

CTree *CSyntax::type_id () { 
  // 2: type_spec_seq  abst_declarator
  // 2: type_spec_seq  private_name
  bool success = parse (&CSyntax::type_spec_seq) && 
                 (parse (&CSyntax::abst_declarator) ||
                  parse (&CSyntax::private_name));
  semantic ().finish_declarator ();
  return success ? 
    semantic ().introduce_named_type () : (CTree*)0;
}

CTree *CSyntax::type_spec_seq () { 
  // 1+: type_spec_seq1...
  semantic ().decl_spec_seq ();
  return seq (&CSyntax::type_spec_seq1) ?
    semantic ().begin_decl () : semantic ().decl_spec_seq_err ();
}

CTree *CSyntax::type_spec_seq1 () { 
  // 1: type_spec
  return type_spec () ?
    semantic ().decl_spec_seq1 () : (CTree*)0;
}

CTree *CSyntax::abst_declarator () { 
  // 1 : direct_abst_declarator
  // 1+: ptr_operator..
  // 2+: ptr_operator..  direct_abst_declarator
  return (parse (&CSyntax::direct_abst_declarator) ||
          (seq (&CSyntax::ptr_operator) &&
           opt (parse (&CSyntax::direct_abst_declarator)))) ?
    semantic ().abst_declarator () : (CTree*)0;
}

CTree *CSyntax::direct_abst_declarator () { 
  // 1 : direct_abst_declarator1
  // 2+: direct_abst_declarator1  direct_abst_declarator1...
  // 3 : ( abst_declarator )
  // 3+: ( abst_declarator )  direct_abst_declarator1...
  return ((parse (&CSyntax::direct_abst_declarator1) ||
           (parse (TOK_OPEN_ROUND) && 
            parse (&CSyntax::abst_declarator) &&
            parse (TOK_CLOSE_ROUND))) &&  
          opt (seq (&CSyntax::direct_abst_declarator1))) ? 
    semantic ().direct_abst_declarator () : (CTree*)0;
}

CTree *CSyntax::direct_abst_declarator1 () { 
  // 2: [  ]
  // 3: [  *  ]
  // 3: [  ass_expr  ]
  // 3: (  param_decl_clause  )
  return (parse (TOK_OPEN_ROUND) ?  
          (parse (&CSyntax::param_decl_clause), parse (TOK_CLOSE_ROUND)) : 
          (parse (TOK_OPEN_SQUARE) && 
           (parse (&CSyntax::ass_expr) || opt (parse (TOK_MUL))) &&
           parse (TOK_CLOSE_SQUARE))) ?
    builder ().direct_abst_declarator1 () : (CTree*)0;
}

CTree *CSyntax::param_decl_clause () { 
  // 0:
  // 1: param_decl_list
  // 3: param_decl_list  ... 
  semantic ().enter_param_decl_clause (); // enter function prototype scope
  if (!look_ahead (TOK_CLOSE_ROUND) && parse (&CSyntax::param_decl_list)) 
    parse (TOK_ELLIPSIS);
  semantic ().leave_param_decl_clause ();
  return semantic ().param_decl_clause ();
}

CTree *CSyntax::param_decl_list () { 
  // 1 : param_decl
  // 2 : param_decl  ,
  // 3+: param_decl  ,  param_decl ...
  // 4+: param_decl  ,  param_decl ...  , 
  return list (&CSyntax::param_decl, TOK_COMMA, true) ? 
    builder ().param_decl_list () : (CTree*)0;
}

CTree *CSyntax::param_decl () { 
  // 1: param_decl1 
  if (! parse (&CSyntax::param_decl1))
    return (CTree*)0;
  semantic ().finish_decl ();
  return builder ().param_decl ();
}
           
CTree *CSyntax::param_decl1 () { 
  // 2: decl_spec_seq  abst_declarator
  // 2: decl_spec_seq  declarator
  // 2: decl_spec_seq  private_name
  return (parse (&CSyntax::decl_spec_seq) && 
          (parse (&CSyntax::param_decl2) || 
           (parse (&CSyntax::abst_declarator) ||
            parse (&CSyntax::private_name)))) ?
    semantic ().introduce_parameter () : (CTree*)0;
}

CTree *CSyntax::param_decl2 () { 
  // 1: declarator
  semantic ().begin_param_check (); // ambiguity check
  parse (&CSyntax::declarator);
  return semantic ().finish_param_check ();
}

CTree *CSyntax::fct_def () { 
  // 2: declarator  fct_body
  // 3: decl_spec_seq  declarator  fct_body
  // 4: decl_spec_seq  declarator  arg_decl_seq  fct_body
  if (! ((parse (&CSyntax::decl_spec_seq) || semantic ().implicit_int ()) && 
         parse (&CSyntax::declarator))) {
    semantic ().finish_declarator ();
    return (CTree*)0;
  }
  // introduce name and check scope; functions shall be 
  // defined only in namespace or class scope
  if (! semantic ().introduce_function ())
    return (CTree*)0;
  return (parse (&CSyntax::arg_decl_seq), parse (&CSyntax::fct_body)) ? 
    semantic ().finish_fct_def () : (CTree*)0;
}

CTree *CSyntax::arg_decl_seq () { 
  // 1+: simple_decl
  semantic ().enter_arg_decl_seq ();
  bool success = (seq (&CSyntax::simple_decl));
  semantic ().leave_arg_decl_seq ();
  return success ?
    semantic ().arg_decl_seq () : (CTree*)0;
}

CTree *CSyntax::fct_body () { 
  // 1: cmpd_stmt
  return parse (&CSyntax::cmpd_stmt) ? 
    builder ().fct_body () : (CTree*)0;
}

CTree *CSyntax::init () { 
  // 2: =  init_clause 
  return (parse (TOK_ASSIGN) && parse (&CSyntax::init_clause)) ?
    builder ().init () : (CTree*)0;
}

CTree *CSyntax::init_clause () { 
  // 1: ass_expr
  // 3: {  init_list  }
  return (parse (&CSyntax::ass_expr) ||
          (parse (TOK_OPEN_CURLY) && 
           parse (&CSyntax::init_list) && 
           parse (TOK_CLOSE_CURLY))) ?
    builder ().init_clause () : (CTree*)0;
}

CTree *CSyntax::init_list () { 
  // 1 : init_list_item
  // 2 : init_list_item  ,
  // 3+: init_list_item  ,  init_list_item ...
  // 4+: init_list_item  ,  init_list_item ...  ,
  return list (&CSyntax::init_list_item, TOK_COMMA, true) ? 
    builder ().init_list () : (CTree*)0;
}

CTree *CSyntax::init_list_item () { 
  // 1: init_clause
  // 2: designation  init_clause
  return (parse (&CSyntax::designation), parse (&CSyntax::init_clause)) ? 
    builder ().init_list_item () : (CTree*)0;
}

CTree *CSyntax::designation () { 
  // 2+: designator...  =
  return (seq (&CSyntax::designator) && parse (TOK_ASSIGN)) ? 
    builder ().designation () : (CTree*)0;
}

CTree *CSyntax::designator () { 
  // 2: .  identifier
  // 3: [  const_expr  ]
  return (parse (TOK_OPEN_SQUARE) ? 
          (parse (&CSyntax::const_expr) && parse (TOK_CLOSE_SQUARE)) :
          (parse (TOK_DOT) && identifier ())) ? 
    builder ().designator () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                          A . 8   C l a s s e s                            */
/*                                                                           */
/*****************************************************************************/

void CSyntax::init_class_spec () {
  init_class_head ();
  _class_spec_1 = _class_head_1;
}

CTree *CSyntax::rule_class_spec () { 
  // 3: class_head  {  }
  // 4: class_head  {  member_spec  }
  if (!(class_head () && look_ahead (TOK_OPEN_CURLY) && consume ()))
    return (CTree*)0; 
  // create empty class|union|...
  CTree *cs = semantic ().introduce_class ();
  if (! cs) 
    return (CTree*)0;
  return ((!look_ahead (TOK_CLOSE_CURLY) && parse (&CSyntax::member_spec)),
          parse (TOK_CLOSE_CURLY)) ? 
    semantic ().class_spec () : semantic ().class_spec_err (cs);
}

bool CSyntax::class_spec () {
  return predict_1 (_class_spec_1) && parse (&CSyntax::rule_class_spec);
}

void CSyntax::init_class_head () {
  init_class_key ();
  _class_head_1 = _class_key_1;
}

CTree *CSyntax::rule_class_head () { 
  // 1: class_key
  // 2: class_key  identifier
  return (class_key () && 
          (identifier () || parse (&CSyntax::private_name))) ?
    builder ().class_head () : (CTree*)0;
}

bool CSyntax::class_head () {
  return predict_1 (_class_head_1) && parse (&CSyntax::rule_class_head);
}

void CSyntax::init_class_key () {
  _class_key_1.set (TOK_STRUCT);
  _class_key_1.set (TOK_UNION);  
}

CTree *CSyntax::rule_class_key () {
  // 1: STRUCT
  // 1: UNION
  consume (); // the token was checked during the prediction
  return builder ().class_key ();
}

bool CSyntax::class_key () { 
  return predict_1 (_class_key_1) && parse (&CSyntax::rule_class_key);
}

CTree *CSyntax::member_spec () { 
  // 1+: member_decl...
  int skip[] = { /*TOK_SEMI_COLON,*/ TOK_CLOSE_CURLY, 0 };
  int finish[] = { TOK_CLOSE_CURLY, 0 };

  if (! catch_error (&CSyntax::member_decl, 
    "invalid member declaration", finish, skip))
    return (CTree*)0;

  while (catch_error (&CSyntax::member_decl, 
    "invalid member declaration", finish, skip));
  return builder ().member_spec ();
}

CTree *CSyntax::member_decl () { 
  // 3: type_spec_seq  member_declarator_list  ;
  if (! (parse (&CSyntax::type_spec_seq) && 
         parse (&CSyntax::member_declarator_list) && 
         parse (TOK_SEMI_COLON)))
    return (CTree*)0;
  semantic ().finish_decl ();
  return builder ().member_decl (); 
}

CTree *CSyntax::member_declarator_list () { 
  // 1 : member_declarator
  // 3+: member_declarator  ,  member_declarator ...
  return list (&CSyntax::member_declarator, TOK_COMMA) ? 
    builder ().member_declarator_list () : (CTree*)0;
}

CTree *CSyntax::member_declarator () { 
  // 1: declarator
  // 3: declarator  :  const_expr
  // 3: private_name  :  const_expr
  bool have_declarator = (parse (&CSyntax::declarator));
  semantic ().finish_declarator ();
  return (look_ahead (TOK_COLON) ?
            ((have_declarator || parse (&CSyntax::private_name)) &&
           parse (TOK_COLON) && parse (&CSyntax::const_expr)) : 
          have_declarator) ? 
    semantic ().introduce_member () : (CTree*)0;
}

/*****************************************************************************/
/*                                                                           */
/*                   A . 9   D e r i v e d  c l a s s e s                    */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*        A . 1 0   S p e c i a l  m e m b e r  f u n c t i o n s            */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                     A . 1 1   O v e r l o a d i n g                       */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                       A . 1 2   T e m p l a t e s                         */
/*                                                                           */
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                A . 1 3  E x c e p t i o n  h a n d l i n g                */
/*                                                                           */
/*****************************************************************************/


} // namespace Puma
