#include <gc.h>

#include "throughput.h"

static void _throughput_free(gpointer p);

Throughput* throughput_new()
{
  Throughput *t;

  g_assert(t = g_new(Throughput, 1));
  t->timer = NULL;
  memset(t->values, 0, sizeof(t->values));
  t->index = 0;
  t->byte_counter = 0;
  t->mutex = g_mutex_new();

  return gc_add(t, _throughput_free, "throughput");
}

static void _throughput_free(gpointer p)
{
  Throughput *t;
  g_assert(t = (Throughput*) p);

  if (t->timer) g_timer_destroy(t->timer);
  g_mutex_free(t->mutex);
  g_free(t);
}

void throughput_measure(Throughput *t, guint bytes)
{
  g_assert(t);

  if (!bytes)
    return;

  g_mutex_lock(t->mutex);
  if (t->timer)
    {
      t->values[t->index++] = (guint)(bytes / g_timer_elapsed(t->timer, NULL));

      while (t->index >= ASD_THROUGHPUT_VALUES_MAX) 
	t->index -= ASD_THROUGHPUT_VALUES_MAX;

      g_timer_reset(t->timer);
    }
  else
    {
      t->timer = g_timer_new();
      g_timer_start(t->timer);
    }

  t->byte_counter += bytes;
 
  g_mutex_unlock(t->mutex);
}

guint throughput_value(Throughput *t)
{
  gulong l;
  guint i, c;

  g_assert(t);
  g_mutex_lock(t->mutex);
  
  l = c = 0;
  for (i = 0; i < ASD_THROUGHPUT_VALUES_MAX; i++)
    if (t->values[i])
      {
        l += t->values[i];
        c++;
      }

  g_mutex_unlock(t->mutex);

  return c ? l/c : 0;
}

void throughput_reset(Throughput *t)
{
  guint i;
  g_assert(t);

  g_mutex_lock(t->mutex);
  for (i = 0; i < ASD_THROUGHPUT_VALUES_MAX; i++)
    t->values[i] = 0;

  t->byte_counter = 0;
  g_mutex_unlock(t->mutex);
}
