#include <recursive-lock.h>

#include <latency.h>

static LatencyBank _bank;  // Global per-source latency
static gboolean _bank_reset = FALSE;

static GStaticMutex _bank_mutex = G_STATIC_MUTEX_INIT;
static GStaticPrivate _bank_private = G_STATIC_PRIVATE_INIT;

#define _bank_lock() recursive_lock(&_bank_mutex, &_bank_private)
#define _bank_unlock() recursive_unlock(&_bank_mutex, &_bank_private)

static gulong _tv_diff(GTimeVal *tva, GTimeVal *tvb) // a-b
{
  gulong r, a, b;
  g_assert(tva && tvb);

  a = (gulong)tva->tv_sec*1000+(gulong)tva->tv_usec/1000;
  b = (gulong)tvb->tv_sec*1000+(gulong)tvb->tv_usec/1000;

  if (a >= b)
    r = a - b;
  else
    r = a + (24*60*60) - b;

  return r;
}

void latency_reset(Latency *l)
{
  g_assert(l);
  memset(l, 0, sizeof(Latency));
}

void latency_set(Latency *l, OriginId id)
{
  g_assert(l && (id.id < ORIGIN_MAX));

  g_get_current_time(&l->tv[id.id]);
}

void latency_merge(Latency *a, Latency *b)
{
  guint i;
  g_assert(a && b);

  for (i = 0; i < ORIGIN_MAX; i++)
    if (b->tv[i].tv_sec)
      {
        a->tv[i].tv_sec = b->tv[i].tv_sec;
        a->tv[i].tv_usec = b->tv[i].tv_usec;
      }
}

void latency_bank_reset(LatencyBank *b)
{
  g_assert(b);
  memset(b, 0, sizeof(LatencyBank));
}

void latency_bank_process(LatencyBank *b, Block *block)
{
  guint i;
  GTimeVal c;
  g_assert(b && block);

  g_get_current_time(&c);
  
  for (i = 0; i < ORIGIN_MAX; i++)
    if (block->origin.enabled[i])
      if (block->origin.instance[i] >= b->instance[i])
        {
          guint msec = _tv_diff(&c, &block->latency.tv[i]);
          if (msec && (msec < 10*1000)) // Ignore zero latency, or > 10sec
            {
              b->msec[i] = msec;
              b->instance[i] = block->origin.instance[i];
            }
        }
}


guint32 latency_bank_get(LatencyBank *b, OriginId id)
{
  guint32 r = 0;

  g_assert(b && (id.id < ORIGIN_MAX));

  if (id.instance == b->instance[id.id])
    r = b->msec[id.id];

  return r;
}


void latency_global_process(Block *b)
{
  g_assert(b);

  _bank_lock();

  if (!_bank_reset)
    {
      latency_bank_reset(&_bank);
      _bank_reset = TRUE;
    }

  latency_bank_process(&_bank, b);

  _bank_unlock();
}


guint32 latency_global_get(OriginId id)
{
  guint32 r;

  g_assert(id.id < ORIGIN_MAX);

  _bank_lock();
  r = latency_bank_get(&_bank, id);
  _bank_unlock();

  return r;
}

guint32 latency_global_average()
{
  gulong result = 0;
  guint count = 0;
  guint i;
  _bank_lock();

  for (i = 0; i < ORIGIN_MAX; i++)
    if (_bank.msec[i])
      {
        result += _bank.msec[i];
        count++;
      }
  
  _bank_unlock();

  return count ? result/count : 0;
}
