/*
 * Copyright (c) 2001 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Various utility classes
 *
 *	by Tony Sideris	(02:49AM Nov 17, 2002)
 *================================================*/
#include "arson.h"

#include <qfile.h>

#include <kio/netaccess.h>
#include <kstddirs.h>
#include <kglobal.h>
#include <klocale.h>
#include <kurl.h>

#include "docwidget.h"
#include "utils.h"

/*========================================================*/
/*	A net-aware filename
 *========================================================*/

ArsonNetFile::ArsonNetFile (const KURL &url)
	: m_local(QString::null)
{
	if ((m_bLocal = url.isLocalFile()))
		m_local = url.path();

	else if (!KIO::NetAccess::download(url, m_local))
		throw ArsonError(
			i18n("Failed to download %1")
			.arg(url.path()));
}

ArsonNetFile::~ArsonNetFile (void)
{
	if (!m_bLocal && m_local != QString::null)
		QFile::remove(QFile::encodeName(m_local));
}

/*========================================================*/
/*	A file filter builder
 *========================================================*/

const ArsonFileFilter::Filter ArsonFileFilter::allFiles(I18N_NOOP("All Files"), "*");

ArsonFileFilter::ArsonFileFilter (const QString &init, uint flags)
	: m_flags(flags)
{
	fromString(init);
}

ArsonFileFilter::ArsonFileFilter (const QString *arr, size_t cnt, uint flags)
	: m_flags(flags)
{
	for (int index = 0; index < cnt; ++index)
		addFilter(arr[index]);
}

ArsonFileFilter::ArsonFileFilter (ArsonFileListDoc *pd, uint flags)
	: m_flags(flags)
{
	pd->buildFileFilter(*this);
}

/*========================================================*/

void ArsonFileFilter::Filter::fromString (const QString &str)
{
	const int at = str.find('|');

	if (at != -1)
	{
		m_filter = str.left(at);
		m_desc = str.right(str.length() - at - 1);
	}
}

void ArsonFileFilter::fromString (const QString &str)
{
	const QStringList sl = QStringList::split(QString("\n"), str);

	for (QStringList::ConstIterator it = sl.begin(), end = sl.end();
		 it != end; ++it)
		addFilter((*it));
}

/*========================================================*/

QString ArsonFileFilter::toString (void) const
{
	QString result, all;

	for (FILTERS::ConstIterator it = m_filters.begin(), end = m_filters.end();
		 it != end; ++it)
	{
		const Filter &cur = (*it);

		if (!result.isEmpty())
			result.append("\n");

		if (m_flags & AllTypes &&
			cur.filter() != "*" && cur.filter() != "*.*")
		{
			if (!all.isEmpty())
				all.append(" ");

			all.append(cur.filter());
		}

		result.append(cur.toString());
	}

	if (m_flags & AllFiles)
		result.append(QString("\n") + allFiles.toString());

	if (m_flags & AllTypes)
		result = Filter(i18n("All Supported Types"), all).toString() + "\n" + result;

	Trace("filter: %s\n", result.latin1());
	return result;
}

QString ArsonFileFilter::Filter::toString (void) const
{
	QString result (m_filter + "|" + i18n(m_desc));

	if (m_filter.length() <= 16)
		result.append(QString(" (") + m_filter + ")");

	return result;
}

/*========================================================*/
/*	Only the filters, no descriptions (like for QDir)
 *========================================================*/

QString ArsonFileFilter::filters (void) const
{
	QString result;

	for (FILTERS::ConstIterator it = m_filters.begin(), end = m_filters.end();
		 it != end; ++it)
	{
		if ((*it).filter() == "*" || (*it).filter() == "*.*")
			continue;

		if (!result.isEmpty())
			result.append(" ");

		result.append((*it).filter());
	}

	return result;
}

/*========================================================*/

void ArsonFileFilter::addFilter (const QString &desc, const QString &filter)
{
	addFilter(Filter(desc, filter));
}

void ArsonFileFilter::addFilter (const Filter &filter)
{
	if (!m_filters.contains(filter))
		m_filters.append(filter);
}

void ArsonFileFilter::addFilter (const QString &str)
{
	addFilter(Filter(str));
}

/*========================================================*/
/*	Read in a user's XML config file
 *========================================================*/

int ArsonUserCfg::readConfig (const char *fname, bool userOnly)
{
	int res = 0;
	
	if (!userOnly)
	{
#ifdef ARSONDBG
		const QString cfg = fname;
#else
		const QString cfg = KGlobal::dirs()->findResource("config", fname);
#endif	//	ARSONDBG

		if (readCfg(cfg))
			res |= global;
	}

	if (readCfg(locateLocal("config", fname)))
		res |= local;

	return res;
}

/*========================================================*/

bool ArsonUserCfg::readCfg (const char *fname)
{
	Trace("ArsonUserCfg::readCfg reading %s\n", fname);

	try { return arsonParseXml(KURL(QString("file:") + fname), this); }
	catch (ArsonError &err) { return false; }
//	catch (...) { return false; }
}

/*========================================================*/
