/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	SCSI and IOCTL devices
 *
 *	by Tony Sideris	(01:40PM Apr 23, 2002)
 *================================================*/
/*================================================*/
/*	IDE devices
 *
 *	by Brian Kreulen	(03:30PM Sep 29, 2003)
 *================================================*/
#include "arson.h"

#include <qcombobox.h>
#include <qdir.h>

#include <klocale.h>
#include <kconfig.h>
#include <kapp.h>

#include "process.h"
#include "device.h"
#include "konfig.h"

#define IOCTLPREF		"ioctl"
#define DEVGRP			"devices"

/*========================================================*/

QString ArsonDevice::description (void) const
{
	return QString();
}

QString ArsonDevice::display (void) const
{
	return QString("IOCTL: ") + QString(m_dev);
}

/*========================================================*/
/*	A SCSI Device
 *========================================================*/

ArsonScsiDevice::ArsonScsiDevice (const char *id, const char *make,
	const char *model, const char *ver, const char *type)
	: ArsonDevice(NULL, DT_SCSIDEV), m_id(id),
	m_make(make),
	m_model(model),
	m_version(ver),
	m_type(type)
{
	if (id)
	{
		KConfig *pk = kapp->config();

		pk->setGroup(DEVGRP);
		setDev(pk->readEntry(id));
	}
}

/*========================================================*/

QString ArsonScsiDevice::description (void) const
{
	QString temp;
	temp.sprintf("%s %s, %s",
		m_make.data(),
		m_model.data(),
		m_version.data());

	return temp;
}

QString ArsonScsiDevice::display (void) const
{
	QString res ("SCSI: ");

	res.append(m_id);

	if (!dev().isEmpty())
		res.append(" (").append(dev()).append(")");

	return res + " " + description();
}

/*========================================================*/
/*	IDE Device
 *========================================================*/

ArsonIdeDevice::ArsonIdeDevice (const char *id, const char *make,
	const char *model, const char *type)
	: ArsonDevice(NULL, DT_IDEDEV), m_id(id),
	m_make(make),
	m_model(model),
	m_type(type)
{
   if (id)
	{
		KConfig *pk = kapp->config();

		pk->setGroup(DEVGRP);
		setDev(pk->readEntry(id));
	}
}

/*========================================================*/

QString ArsonIdeDevice::description (void) const
{
	QString temp;
	temp.sprintf("%s %s %s",
		m_make.data(),
      m_type.data(),
		m_model.data());

	return temp;
}

QString ArsonIdeDevice::display (void) const
{
	QString res ("IDE: ");

	res.append(m_id);

	if (!dev().isEmpty())
		res.append(" (").append(dev()).append(")");

	return res + " " + description();
}
 
/*========================================================*/
/*	Device access
 *========================================================*/

ArsonDeviceList::ArsonDeviceList (void)
{
	//	Nothing...
}

ArsonDeviceList::~ArsonDeviceList (void)
{
	clear();
}

/*========================================================*/

ArsonDeviceList &ArsonDeviceList::operator= (const ArsonDeviceList &obj)
{
	QStringList keys = obj.keys();
	QStringList::ConstIterator it, end;

	clear();

	for (it = keys.begin(), end = keys.end(); it != end; ++it)
		if (const ArsonDevice *pd = obj.device(*it))
		{
			switch (pd->devType())
         {
            case ArsonDevice::DT_SCSIDEV:
               addDev(*it, new ArsonScsiDevice(*((ArsonScsiDevice *) pd)));
               break;

            case ArsonDevice::DT_IDEDEV:
               addDev(*it, new ArsonIdeDevice(*((ArsonIdeDevice *) pd)));
               break;

            default:
               addDev(*it, new ArsonDevice(*pd));
         }
		}

	return *this;
}

ArsonDeviceList::ArsonDeviceList (const ArsonDeviceList &obj)
{
	(*this) = obj;
}

/*========================================================*/

void ArsonDeviceList::addDev (const char *dev, ArsonDevice *ptr)
{
	delDev(dev);
	m_devs[QCString(dev)] = ptr;
}

void ArsonDeviceList::delDev (const char *dev)
{
	const QCString key (dev);
	DEVICES::Iterator it = m_devs.find(key);

	if (it != m_devs.end())
		delete it.data();

	m_devs.remove(key);
}

/*========================================================*/

bool ArsonDeviceList::addIoctlDev (const char *dev)
{
	const QCString key(dev);

	if (m_devs.find(key) != m_devs.end())
	{
		Trace("Attempt to add duplicate %s\n", dev);
		return false;
	}

	addDev(key, new ArsonDevice(dev, ArsonDevice::DT_IOCTLDEV));
	return true;
}

bool ArsonDeviceList::delIoctlDev (const char *dev)
{
	if (m_devs.find(QCString(dev)) == m_devs.end())
	{
		Trace("Failed to remove non-existant device: %s\n", dev);
		return false;
	}

	delDev(dev);
	return true;
}

/*========================================================*/

uint ArsonDeviceList::count (int onlyDevType) const
{
	DEVICES::ConstIterator it, end;
	uint count = 0;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
		if (onlyDevType == it.data()->devType())
			++count;

	return count;
}

/*========================================================*/

void ArsonDeviceList::clear (void)
{
	DEVICES::Iterator it, end;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
		delete it.data();

	m_devs.clear();
}

void ArsonDeviceList::clear (int onlyDevType)
{
	DEVICES::Iterator it, end;
	QStringList keys;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
		if (onlyDevType == it.data()->devType())
			keys.append(it.key());

	for (QStringList::Iterator it = keys.begin(), end = keys.end(); it != end; ++it)
		delDev(*it);
}

/*========================================================*/

const ArsonDevice *ArsonDeviceList::device (const char *dev) const
{
	DEVICES::ConstIterator it = m_devs.find(QCString(dev));

	return (it == m_devs.end()) ? NULL : it.data();
}

ArsonDevice *ArsonDeviceList::device (const char *dev)
{
	DEVICES::Iterator it = m_devs.find(QCString(dev));

	return (it == m_devs.end()) ? NULL : it.data();
}

/*========================================================*/

QStringList ArsonDeviceList::keys (void) const
{
	QStringList sl;
	DEVICES::ConstIterator it, end;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
		sl.append(it.key());

	return sl;
}

/*========================================================*/

ArsonDeviceUiList ArsonDeviceList::uiList () const
{
	DEVICES::ConstIterator it, end;
	QStringList sl;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
         sl.append(it.key());

	return ArsonDeviceUiList(sl, this);
}

/*========================================================*/

ArsonDeviceUiList ArsonDeviceList::uiList (int onlyDevType) const
{
	DEVICES::ConstIterator it, end;
	QStringList sl;

	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
      if (onlyDevType == it.data()->devType())
         sl.append(it.key());

	return ArsonDeviceUiList(sl, this);
}

/*========================================================*/

void ArsonDeviceList::load (KConfig *pk)
{
	pk->setGroup(DEVGRP);
	clear(false);

	for (int index = 0; ; ++index)
	{
		QString key (IOCTLPREF), val;
		key += QString::number(index);

		val = pk->readEntry(key, QString::null);

		if (val == QString::null)
			break;

		addIoctlDev(val.latin1());
	}

	scanbus();
}

void ArsonDeviceList::save (KConfig *pk)
{
	DEVICES::ConstIterator it, end;
	int index = 0;

	pk->setGroup(DEVGRP);
	
	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
	{
		if (it.data()->devType() != ArsonDevice::DT_IOCTLDEV)
			pk->writeEntry(it.data()->id().data(),
				QString(it.data()->dev()));
		else
		{
			QString key (IOCTLPREF);
			key += QString::number(index++);

			pk->writeEntry(key, QString(it.data()->dev()));
		}
	}
}

/*========================================================*/
/*	Runs appropriate command to get a listing of the
 *	available SCSI devices, extracts the usable CD-ROM
 *	drives from the output and store in the drive list
 *========================================================*/

class busScan : public ArsonUtilityProcess
{
	typedef QValueList<ArsonScsiDevice> DEVLIST;
	
public:
	busScan (int program, const char *arg, Communication comm)
		: ArsonUtilityProcess(ACONFIG, comm, QString::null, program)
	{
		(*this) << arg;
	}

	ArsonScsiDevice *device (int index)
	{ return (index < m_devs.count()) ? &(m_devs[index]) : NULL; }
	uint count (void) const { return m_devs.count(); }

	void addDevice (const ArsonScsiDevice &dev)
	{
		Trace("SCSI Device: %s\n",
			dev.display().latin1());

		m_devs.append(dev);
	}

protected:
	DEVLIST m_devs;
};

class cdrecordBusScan : public busScan
{
public:
	cdrecordBusScan (void)
		: busScan(ArsonConfig::PROGRAM_CDRECORD, "-scanbus", Communication(Stderr | Stdout))
	{
	}

	virtual void output (const QString &out, bool error)
	{
		KRegExp re(
			"[\t ]*([0-9]+,[0-9]+,[0-9]+)[\t ]+[0-9]+\\)[\t ]+"
			"'([^']+)' +'([^']+)' +'([^']+)' +(.+CD-ROM).*$");

		Assert(buffering());
		
		/*	Create a new device for each line matching the pattern:
		 *	0,0,0	  0) 'SAF     ' 'CD-RW2224       ' '2.43' Removable CD-ROM
		 */
		if (re.match(out))
		{
			addDevice(
				ArsonScsiDevice(
					re.group(1),
					(const char *) QString(re.group(2)).stripWhiteSpace(),
					(const char *) QString(re.group(3)).stripWhiteSpace(),
					(const char *) QString(re.group(4)).stripWhiteSpace(),
					re.group(5)));
		}
	}
};

class cdrdaoBusScan : public busScan
{
public:
	cdrdaoBusScan (void)
		: busScan(ArsonConfig::PROGRAM_CDRDAO, "scanbus", Communication(Stdout | Stderr))
	{
		//	Nothing...
	}

	virtual void output (const QString &out, bool error)
	{
		/*	Match cdrdao output:
		0,0,0: SONY, CD-RW CRX1611, TYS3
		0,1,0: , 20DY, 1.70
		 */
		KRegExp re(
			" *([0-9]+,[0-9]+,[0-9]+): ([^,]*), ([^,]*), ([^,]*)");

		Assert(buffering());
		
		if (re.match(out))
		{
			addDevice(
				ArsonScsiDevice(
					re.group(1),
					QString(re.group(2)).stripWhiteSpace(),
					QString(re.group(3)).stripWhiteSpace(),
					QString(re.group(4)).stripWhiteSpace(),
					NULL));
		}
	}
};

/*========================================================*/

int ArsonDeviceList::scanbus (void)
{
	int res = -1;

   res += scanScsiBus();

   if (g_CONFIG.isIDEBurningSupported())
      res += scanIdeBus();
   
	return res;
}

/*========================================================*/

int ArsonDeviceList::scanScsiBus(void)
{
	busScan *ptr;
	DEVICES::ConstIterator it, end;
	QMap<QCString,QCString> devs;

   /*	Create the appropriate scanner instance
	 *	based on the user's preference.
	 */
	if (ACONFIG.programPref(PROGGRP_DEVSCAN) == "cdrdao")
		ptr = new cdrdaoBusScan;
	else
		ptr = new cdrecordBusScan;

	//	Save old device values for non IOCTL devices so we can restore
	for (it = m_devs.begin(), end = m_devs.end(); it != end; ++it)
		if (it.data()->devType() != ArsonDevice::DT_IOCTLDEV)
			devs[it.key()] = it.data()->dev();

	clear(true);	//	Reset the list contents

	if (ptr->execute())
	{
		int index;

		//	Copy the devices to from the scanner to the device list.
		for (index = 0; ; ++index)
		{
			if (ArsonScsiDevice *pd = ptr->device(index))
			{
				QMap<QCString,QCString>::ConstIterator it = devs.find(pd->id());

				if (it != devs.end())
					pd->setDev(it.data());

				addDev(pd->id(), new ArsonScsiDevice(*pd));
			}
			else
				break;
		}
	}

   delete ptr;
   return scsiCount();
}

/*========================================================*/

int ArsonDeviceList::scanIdeBus (void)
{
   QDir procIdeDir = QString("/proc/ide");
   QStringList entryList = procIdeDir.entryList(QString("hd?"));
   QFile *hdFile;
   QString media;
   QString newDev;

   for (QStringList::Iterator it = entryList.begin(); it != entryList.end(); ++it)
   {
      hdFile = new QFile("/proc/ide/" + (*it) + "/media");

      if (!hdFile)
         continue;

      hdFile->open(IO_ReadOnly);
      hdFile->readLine(media, 6); // 6 is the size of "cdrom"
      hdFile->close();
      delete hdFile;

      // "cdrom" is returned for cdroms, cdr/w and dvd drives
      if (media != "cdrom")
         continue;

      hdFile = new QFile("/proc/ide/" + (*it) + "/model");

      if (!hdFile)
         continue;

      hdFile->open(IO_ReadOnly);
      hdFile->readLine(media, 1024);
      hdFile->close();
      delete hdFile;

      newDev = "/dev/" + (*it);
		addDev(newDev, new ArsonIdeDevice(newDev, media.section(' ', 0, 0),
         media.section(' ', 2, 2).stripWhiteSpace(), media.section(' ', 1, 1)));
   }
   
   return ideCount();   
}

/*========================================================*/
/*	For use in comboboxes
 *========================================================*/

ArsonDeviceUiList::ArsonDeviceUiList (const QStringList &sl, const ArsonDeviceList *pd)
	: m_keys(sl), m_devs(pd)
{
	//	Nothing...
}

/*========================================================*/

void ArsonDeviceUiList::fillList (QComboBox *pl, const char *sel) const
{
	QStringList::ConstIterator it, end;
	int nsel = -1, index = 0;

	if (!m_devs) return;
	pl->clear();

	for (it = m_keys.begin(), end = m_keys.end(); it != end; ++it, ++index)
	{
		if (nsel == -1 && *it == sel)
			nsel = index;

		if (const ArsonDevice *pd = m_devs->device(*it))
      {
         Trace("inserting (%s)",(const char *)pd->display());
			pl->insertItem(pd->display());
      }
	}

	if (nsel != -1)
		pl->setCurrentItem(nsel);
}

QCString ArsonDeviceUiList::fromList (QComboBox *pl) const
{
	const int sel = pl->currentItem();

	if (sel >= 0 && sel < m_keys.count())
		return m_keys[sel].latin1();

	return QCString();
}

/*========================================================*/
