// DragonLog.cpp: implementation of the CDragonLog class.
//
//////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "DragonLog.h"

#ifndef _WIN32
#include <limits.h>
#define _MAX_PATH PATH_MAX
#define DRAGON_LOG_PATH_SEP "/"
#include <sys/types.h>
#include <sys/stat.h>
  #ifndef HAVE_FTS_OPEN
  #include <ftw.h>
  #endif
#endif

#ifdef _WIN32
#define DRAGON_LOG_PATH_SEP "\\"
#endif

#ifdef HAVE_FTS_OPEN
#include <fts.h>
#endif

#define DRAGON_LOG "dragon.log."

int *CDragonLog::m_gFilesExist;

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CDragonLog::CDragonLog()
{

}

CDragonLog::~CDragonLog()
{

}

void CDragonLog::IDSVersion (char *sIDSProduct, int& nIDSID, int& nIDSMajor, int& nIDSMinor, char *sIDSRev) const
{
	strcpy (sIDSProduct, "Dragon");
	nIDSID = 5;
	nIDSMajor = 4;
	nIDSMinor = 2;
	sIDSRev[0] = '\0';
}

// return false if the field value means SKIP
// true otherwise

bool CDragonLog::ParseField (char cField, const char *sField)
{
	int len = strlen (sField);
	char *sFieldSub = NULL, *p = NULL;
	int nTmp;
	int ip1, ip2, ip3, ip4;
	int yr, mt, dy, hr, mn, sc;

	bool bRet = true;

	switch (cField)
	{
	case 'S':
		// Source IP address	
		if (sscanf (sField, "%d.%d.%d.%d", &ip1, &ip2, &ip3, &ip4) == 4)
		{
			unsigned char *octets = (unsigned char *)&m_nFromIP;
			octets[0] = (unsigned char) (ip1 & 0xff);
			octets[1] = (unsigned char) (ip2 & 0xff);
			octets[2] = (unsigned char) (ip3 & 0xff);
			octets[3] = (unsigned char) (ip4 & 0xff);
		}
		else
			m_nFromIP = 0;
		break;
	case 's':
		// Source IP as an int
		if (sscanf (sField, "%u", &nTmp) == 1)
			m_nFromIP = nTmp;
		else
			m_nFromIP = 0;
		break;
	case 'D':
		// Dest IP address
		if (sscanf (sField, "%d.%d.%d.%d", &ip1, &ip2, &ip3, &ip4) == 4)
		{
			unsigned char *octets = (unsigned char *)&m_nToIP;
			octets[0] = (unsigned char) (ip1 & 0xff);
			octets[1] = (unsigned char) (ip2 & 0xff);
			octets[2] = (unsigned char) (ip3 & 0xff);
			octets[3] = (unsigned char) (ip4 & 0xff);
		}
		else
			m_nToIP = 0;
		break;
	case 'd':
		// Dest IP as an int
		if (sscanf (sField, "%u", &nTmp) == 1)
			m_nToIP = nTmp;
		else
			m_nToIP = 0;
		break;
	case 'G':
		// Source port (can be empty)
		if (sField[0] == '\0')
		{
			m_nFromPort = 0;
			break;
		}
		if (sscanf (sField, "%d", &nTmp) == 1)
			m_nFromPort = nTmp;
		else
			m_nFromPort = 0;
		break;
	case 'H':
		// Dest port (Can be empty)
		if (sField[0] == '\0')
		{
			m_nToPort = 0;
			break;
		}
		if (sscanf (sField, "%d", &nTmp) == 1)
			m_nToPort = nTmp;
		else
			m_nToPort = 0;
		break;
	case 'T':
		// Date/time (YYYY-MM-DD HH:MM:SS)
		if (sscanf (sField, "%d-%d-%d %d:%d:%d", &yr, &mt, &dy, &hr, &mn, &sc) == 6)
		{
			// The 2000-01-01 date is arbitrary.
			m_nDate = GetDate (yr, mt, dy, 0, 0, 0, 0);
			m_nTime = GetDate (2000, 1, 1, hr, mn, sc, 0) - GetDate (2000, 1, 1, 0, 0, 0, 0);
		}
		break;
	case 't':
		// Date (YYYY-MM-DD)
		if (sscanf (sField, "%d-%d-%d", &yr, &mt, &dy) == 6)
		{
			m_nDate = GetDate (yr, mt, dy, 0, 0, 0, 0);
		}
		break;
	case 'h':
		// Time (HH:MM:SS)
		if (sscanf (sField, "%d:%d:%d", &hr, &mn, &sc) == 6)
		{
			// The 2000-01-01 date is arbitrary.
			m_nTime = GetDate (2000, 1, 1, hr, mn, sc, 0) - GetDate (2000, 1, 1, 0, 0, 0, 0);
		}
		break;
	case 'N':
		// Name of sensor	(SKIP)
		break;
	case 'E':
		// Event Name
		strncpy (m_sID, sField, LOG_STRLEN);
		break;
	case 'A':
		// Alarm Data		(portscan, protocol, icmp type information)
		//                   do not parse icmp type information
		sFieldSub = strdup (sField);
		p = sFieldSub;
		while (*p)
		{
		  if (*p == ',')
		    *p = '\0';			// break into subfields
		  p++;
		}
		p = sFieldSub;
		while (p < sFieldSub + len)
		{
			if (strcmp (p, "tcp") == 0)
				m_nProtocol = PROTO_TCP;
			else if (strcmp (p, "udp") == 0)
				m_nProtocol = PROTO_UDP;
			else if (strncmp (p, "min=", strlen ("min=")) == 0)
			{
				int n = -1;
				if (sscanf (p + strlen ("min="), "%d", &n) == 1)
					m_nPortscanLow = n;
			}
			else if (strncmp (p, "max=", strlen ("max=")) == 0)
			{
				int n = -1;
				if (sscanf (p + strlen ("max="), "%d", &n) == 1)
					m_nPortscanHigh = n;
			}

			// jump to the next subfield
			while (*p != '\0')
			  p++;
			p++;
		}
		free (sFieldSub);
		break;
	case 'P':
		// Protocol (0 for N/A; Skip record)
		if (sscanf (sField, "%d", &nTmp) == 1)
			m_nProtocol = nTmp;
		else
		{
			m_nProtocol = 0;				// which would mean SKIP
			bRet = false;
		}
		break;
	case 'B':
		// Direction (I = int, E = ext, F = from, T = to)
		break;
	case 'C':
		// TCP flags		(SKIP for now)
		break;
	case 'F':
		// Filler Field		(SKIP)
		break;
	}

	return bRet;
}


bool CDragonLog::LastIncident ()
{
	return m_nFileIndex == m_nNumFilesExist;
}

bool CDragonLog::ReadIncident ()
{
    char sTryFile[_MAX_PATH + 1];
	char sLineBuffer[LOG_STRLEN + 1], sTmp[LOG_STRLEN + 1];
	char *pSep = NULL, *pFmt = NULL, *pFmtEnd = NULL;
	
	if (!m_pCurrentLogFile)
	{
	  for (; m_nFileNum < 1000; m_nFileNum++)
        if (m_pFilesExist[m_nFileNum] == 1)
          break;

	  sprintf (sTryFile, "%s" DRAGON_LOG_PATH_SEP "%s%03d", m_sPath, DRAGON_LOG, m_nFileNum);
	  m_pCurrentLogFile = fopen (sTryFile, "r");
	  if (!m_pCurrentLogFile)			// if couldn't open the file
	  {
	    m_nFileIndex ++;
		m_nFileNum ++;
		return false;
	  }
    }		

	pSep = m_sExportLogFS;
	int nSepLen = strlen (pSep);
	pFmt = m_sExportLogFormat;
	pFmtEnd = pFmt + strlen (pFmt);

	while (!feof (m_pCurrentLogFile))
	{
		if (fgets (sLineBuffer, LOG_STRLEN, m_pCurrentLogFile) != sLineBuffer)
			break;

		int len = strlen (sLineBuffer);

		// first, set some reasonable defaults
		m_nDate = 0.0;
		m_nTime = 0.0;
		m_nPortscanLow = -1;
		m_nPortscanHigh = -1;
		m_nProtocol = 0;
		m_nFromIP = 0;
		m_nToIP = 0;
		m_nFromPort = 0;
		m_nToPort = 0;
		m_sID[0] = '\0';

		int i = 0;
		while (i < len && pFmt < pFmtEnd)
		{
			int j = 0;
			while (i < len && strncmp (sLineBuffer + i, pSep, nSepLen) != 0)
				sTmp[j++] = sLineBuffer[i++];

			sTmp[j] = '\0';
			i += nSepLen;
			
			// parse next format thing
			if (!ParseField (pFmt[1], sTmp))
				return false;
			pFmt += 2;
		}

		// now the fields are parsed.
		m_nCount = 1;
		m_sPortScan[0] = '\0';
		if (m_nPortscanLow != -1 && m_nPortscanHigh != -1)
		  sprintf (m_sPortScan, "%d-%d", m_nPortscanLow, m_nPortscanHigh);
		if (m_nProtocol == 0)
		{
			if (strncmp (m_sID, "TCP-SCAN", strlen ("TCP-SCAN")) == 0)
				m_nProtocol = PROTO_TCP;
			else if (strncmp (m_sID, "UDP-SCAN", strlen ("UDP-SCAN")) == 0)
				m_nProtocol = PROTO_UDP;
		}
		
		m_nDateTime = m_nDate + m_nTime;

		// compare date
		if (m_nDateTime <= m_nStartDate)
		{
			m_bSkippedRecords = true;
			continue;
		}
		else if (m_nDateTime > m_nLastDate)
			m_nLastDate = m_nDateTime;

		return true;
	}

	// EOF reached
	fclose (m_pCurrentLogFile);
	m_pCurrentLogFile = NULL;
	m_nFileIndex ++;
	m_nFileNum ++;

	return false;
}

void CDragonLog::CloseLog ()
{
  delete[] m_pFilesExist;
  free (m_sPath);
}

void CDragonLog::addStuff (const char *fileName, int *pFilesExist)
{
	int nFile = -1;

	/* the log filename format is DRAGON_LOGxxx where xxx is a zero-padded 3-digit number */

	if (sscanf (fileName + strlen (DRAGON_LOG), "%03d", &nFile) != 1)
		return;
	
	pFilesExist[nFile] = 1;
}

#if !(defined HAVE_FTS_OPEN) && !(defined _WIN32)
int CDragonLog::nftwCallback (const char *file, const struct stat *sb, int flag, struct FTW *s)
{
  char *sFile = NULL;

  if (!(sb->st_mode & S_IFREG) || s->level != 1)
    return 0;

  sFile = strrchr (file, '/');
  if (sFile)
    sFile++;            /* skip the / */
  else
    sFile = (char *) file;

  if (strncmp (sFile, DRAGON_LOG, strlen(DRAGON_LOG)) == 0)
    addStuff (sFile, m_gFilesExist);

  return 0;
}
#endif

// usual values:
// DRAGONLOG_FORMAT "%T%N%E%S%D%G%H%B%C%P%A"
// DRAGONLOG_FS "|"


// format of "filename":
// [directory],[cfg file]

bool CDragonLog::OpenLog (const char *filename)
{
  m_bSkippedRecords = false;

  char *sCfgFile = strchr (filename, ',');
  if (!sCfgFile)
	return false;
  
  *sCfgFile = '\0';
  m_sPath = strdup (filename);
  *sCfgFile = ',';  
  sCfgFile++;						// skip ,

  // open cfg file to determine log fmt
  char sLineBuffer[LOG_STRLEN + 1];
  m_sExportLogFormat[0] = '\0';
  m_sExportLogFS[0] = '\0';

  FILE *pCfgFile = fopen (sCfgFile, "r");
  if (!pCfgFile)
	return false;

  while (!feof (pCfgFile))
  {
	if (fgets (sLineBuffer, LOG_STRLEN, pCfgFile) != sLineBuffer)
      break;
	if (sLineBuffer[0] == '#')
		continue;
    else if (strncmp (sLineBuffer, "EXPORTLOG_FORMAT", strlen ("EXPORTLOG_FORMAT")) == 0)
      strncpy (m_sExportLogFormat, sLineBuffer + strlen ("EXPORTLOG_FORMAT") + 1, LOG_STRLEN);
	else if (strncmp (sLineBuffer, "EXPORTLOG_FS", strlen ("EXPORTLOG_FS")) == 0)
      strncpy (m_sExportLogFS, sLineBuffer + strlen ("EXPORTLOG_FS") + 1, LOG_STRLEN);
  }
  fclose (pCfgFile);

  if (strlen (m_sExportLogFormat) == 0 || strlen (m_sExportLogFS) == 0)
	return false;

  // right-trim the fmt strings
  int n = strlen (m_sExportLogFormat) - 1;
  while (m_sExportLogFormat[n] == ' ' || m_sExportLogFormat[n] == '\n' || m_sExportLogFormat[n] == '\t' || m_sExportLogFormat[n] == '\r')
  {
	m_sExportLogFormat[n] = '\0';
	n--;
  }
  n = strlen (m_sExportLogFS) - 1;
  while (m_sExportLogFS[n] == ' ' || m_sExportLogFS[n] == '\n' || m_sExportLogFS[n] == '\t' || m_sExportLogFS[n] == '\r')
  {
	m_sExportLogFS[n] = '\0';
	n--;
  }


#ifdef _WIN32
  char sFiles[_MAX_PATH+1];
  WIN32_FIND_DATA findData;
  HANDLE hFind = INVALID_HANDLE_VALUE;
#endif

#ifdef HAVE_FTS_OPEN
  char *sPaths[2];
  FTS *pFts = NULL;
  FTSENT *pFtsEnt = NULL;
#endif

  m_pFilesExist = new int[1000];			/* 000..999 possible files */
  for (int i = 0; i < 1000; i++)
    m_pFilesExist[i] = 0;
  m_nFileIndex = 0;
  m_nFileNum = 0;
  m_nNumFilesExist = 0;
  m_pCurrentLogFile = NULL;

#ifdef _WIN32
  strcpy (sFiles, m_sPath);
  if (sFiles[strlen (sFiles) - 1] != '\\')
    strcat (sFiles, "\\");              /* add a trailing \ if none exists */
  strcat (sFiles, DRAGON_LOG "*");

  hFind = FindFirstFile (sFiles, &findData);
  while (hFind != INVALID_HANDLE_VALUE)
  {
    if (!(findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
      addStuff (findData.cFileName, m_pFilesExist);

	if (!FindNextFile (hFind, &findData))
    {
      FindClose (hFind);
      break;
    }
  }
#endif

#if !(defined HAVE_FTS_OPEN) && !(defined _WIN32)
  m_gFilesExist = m_pFilesExist;
  nftw (m_sPath, nftwCallback, 1, FTW_PHYS | FTW_MOUNT);
#endif

#ifdef HAVE_FTS_OPEN
  sPaths[0] = m_sPath;
  sPaths[1] = NULL;

  pFts = fts_open (sPaths, FTS_COMFOLLOW | FTS_NOCHDIR | FTS_PHYSICAL | FTS_XDEV, NULL);

  while ((pFtsEnt = fts_read (pFts)))
  {
    if (!(pFtsEnt->fts_statp->st_mode & S_IFREG) || pFtsEnt->fts_level != 1)
      continue;

    if (strncmp (pFtsEnt->fts_name, DRAGON_LOG, strlen(DRAGON_LOG)) == 0)
      addStuff (pFtsEnt->fts_name, m_pFilesExist);
  }

  fts_close (pFts);
#endif

  for (int j = 0; j < 1000; j++)
    if (m_pFilesExist[j])
	  m_nNumFilesExist ++;

  return m_nNumFilesExist != 0;
}

void CDragonLog::SeekLog (double nDate)
{
	m_nStartDate = nDate;
	m_nLastDate = nDate;
}

double CDragonLog::GetLastDate ()
{
	return m_nLastDate;
}
