# library-categories.sh: list categories in the revision library
# 
################################################################
# Copyright (C) 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "list categories in the revision library\\n"
                printf "usage: library-categories [options] [archive]"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
                printf " -r --reverse                  reverse the listing\\n"
                printf "\\n"
		printf "List all categories within a particular archive with records\\n"
		printf "in the revision library.\\n"
		printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
reverse=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "library-find: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "library-find: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    -r|--reverse)	shift
    			reverse=-r
			;;

    -*)                 printf "library-categories: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: library-categories [options] [archive]" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -gt 0 ; then
  archive="$1"
  shift
fi

################################################################
# Sanity Check and Process Defaults
# 

library="`larch my-revision-library -e library-categories`"

if test ! -z "$archive" ; then 
  larch valid-archive-name -e library-categories "$archive"
else
  archive="`larch my-default-archive -e library-categories`"
fi

################################################################
# Do It
# 

cd "$library"

if test ! -d "$archive" ; then
  printf "\\n" 1>&2
  printf "library-categories: archive not found in library\\n" 1>&2
  printf "  archive: %s\\n" "$archive" 1>&2
  printf "\\n" 1>&2
  exit 1
fi

cd "$archive"

for f in `ls | sort $reverse` ; do
  if larch valid-package-name --basename "$f" ; then
    printf "%s\\n" "$f"
  fi
done


# tag: Tom Lord Sun Jan 20 00:38:00 2002 (library/library-categories.sh)
#
