# 
# check-manifest.sh - check the inventory manifest
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "compare the inventory manifest to reality\\n"
                printf "usage: check-manifest [options] \\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -d --dir DIR                  cd to DIR first\\n"
                printf "\\n"
                printf "Compare the inventory manifest to the output of\\n"
                printf "\\n"
		printf "	larch inventory --source\\n"
                printf "\\n"
                printf "in the project tree root.  Report missing or added files.\\n"
                printf "\\n"
                printf "The inventory manifest is a list of files previously recorded\\n"
                printf "as the expected file inventory.\\n"
                printf "\\n"
                printf "See also \"larch manifest\" and \"larch set-manifest\".\\n"
                printf "\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
			if test $# -eq 0 ; then
			  printf "check-manifest: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir="$1"
			shift
			;;

    --)			shift
    			break
			;;
			
    -*)                 printf "check-manifest: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 0 ; then
  printf "usage: check-manifest [options]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

################################################################
# Sanity Check and Process Defaults
# 

here="`pwd`"
cd "$dir"
dir="`pwd`"

cd "$dir"
wdroot="`larch tree-root`"

manifest="$wdroot/{arch}/=manifest"

if test ! -e "$manifest" ; then

  printf "check-manifest: no manifest recorded\\n" 1>&2
  printf "\\n" 1>&2
  printf "Try \"larch set-manifest --help\".\\n" 1>&2
  exit 1

fi

################################################################
# Compute the Inventory and Print Missing Files
# 


want_ff=

page()
{
  if test ! -z "$want_ff" ; then
    printf "\\n\\f\\n"
  else
    printf "Manifest Report for %s\\n" "$wdroot"
    printf "\\n\\f\\n"
  fi

  want_ff=yes

  title="$1"
  shift

  title_width=${#title}

  if test $title_width -gt 64 ; then
    field_width=$title_width
  else
    field_width=$((32 + ($title_width / 2)))
  fi

  printf "%*s\\n" $field_width "$title"
  printf "%*s\\n" $field_width "$title" | sed -e "s/[^[:space:]]/=/g"
  printf "\\n"
  if test $# -gt 0 ; then
    printf "%s\\n" "$@"
  fi
  printf "\\n"
}



cd "$wdroot"

tmpdir=",,manifest.inventory.$$"

bail()
{
  cd "$wdroot"
  rm -rf "$tmpdir"
  exit 1
}

finish()
{
  cd "$wdroot"
  rm -rf "$tmpdir"
  exit 0
}

trap "printf \"check-manifest: interrupted -- cleaning up\\n\" 1>&2 ; bail" INT

rm -rf "$tmpdir"
mkdir "$tmpdir"

inventory="$tmpdir/inventory"

larch inventory --source --tags | sort -k 2 > "$inventory"

missing="$tmpdir/missing"
added="$tmpdir/added"
common="$tmpdir/common"

join -1 2 -2 2 -v 1 "$inventory" "$manifest" > "$added"
join -1 2 -2 2 -v 2 "$inventory" "$manifest" > "$missing"
join -1 2 -2 2 -o 1.1,2.1 "$inventory" "$manifest" > "$common"

if cat "$missing" \
   | cut -s -d ' ' -f 1 \
   | (   read first_line \
      && page "MISSING FILES" \
              "These files are in the manifest, but not in the project tree." \
      && printf "%s\\n" "$first_line" \
      && cat ) \
; then
  want_ff=yes
fi


if cat "$added" \
   | cut -s -d ' ' -f 1 \
   | (   read first_line \
      && page "ADDED FILES" \
              "These files are in the project tree, but not the manifest." \
      && printf "%s\\n" "$first_line" \
      && cat ) \
; then
  want_ff=yes
fi

if cat "$common" \
   | awk '{ if ( $1 != $2 ) print $2 " => " $1 }' \
   | (   read first_line \
      && page "MOVED FILES" \
              "These files are in the project tree and the manifest," \
              "but they are not at the location recorded in the manifest." \
      && printf "%s\\n" "$first_line" \
      && cat ) \
; then
  want_ff=yes
fi



finish
