#!/usr/bin/ruby -Ilib/

require 'test/unit'
require 'apt-listbugs/logic'

class Mock_config_apt
  def command()
    "apt"
  end
end

class Mock_config_list
  def command()
    "list"
  end
end

class TC_BugsFactory < Test::Unit::TestCase
  def setup
    Factory::config=Mock_config_apt.new
  end

  def teardown
    Factory::config=nil
  end

  def test_find_max_version_below_ver
    assert_equal("1.0",
                 Factory::BugsFactory.find_max_version_below_ver("0.8 1.0 1.5 2.0",
                                                                 "1.3",
                                                                 "test"),
                 "max version below version")

    assert_equal(nil,
                 Factory::BugsFactory.find_max_version_below_ver("0.8 1.0 1.5 2.0",
                                                                 "0.5",
                                                                 "test"),
                 "max version below version, nil")
  end

  def test_find_min_version_above_ver
    assert_equal("1.5",
                 Factory::BugsFactory.find_min_version_above_ver("0.8 1.0 1.5 2.0",
                                                                 "1.3",
                                                                 "test"),
                 "min version above version")
    assert_equal(nil,
                 Factory::BugsFactory.find_min_version_above_ver("0.8 1.0 1.5 2.0",
                                                                 "2.5",
                                                                 "test"),
                 "min version above version, nil")
  end

  def test_am_i_buggy
    # from francesco poli
# Legend:  V  version under consideration
#          o  version where the bug is found
#          x  version where the bug is fixed
#          ?  irrelevant data
# a)  ----------------V------------------  buggy
# b)  ----------------V------x-----------  buggy
# c)  ----------------V------o-----------  non-buggy
# d)  ----------------V------o----x------  non-buggy
# e)  ----------------V------x----o------  buggy
#          1    2     3      4    5

# f)  ----------o-----V--- ? -- ? -- ? --  buggy
# g)  ----------x-----V--- ? -- ? -- ? --  non-buggy
# h)  -----o----x-----V--- ? -- ? -- ? --  non-buggy
# i)  -----x----o-----V--- ? -- ? -- ? --  buggy

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 [""].join(" "), # fixed
                                                 [""].join(" ")  # found
                                                 ),
                 "a)  ----------------V------------------  buggy")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["4"].join(" "), # fixed
                                                 [""].join(" ")  # found
                                                 ),
                 "b)  ----------------V------x-----------  buggy")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 [""].join(" "), # fixed
                                                 ["4"].join(" ")  # found
                                                 ),
                 "c)  ----------------V------o-----------  non-buggy")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["5"].join(" "), # fixed
                                                 ["4"].join(" ")  # found
                                                 ),
                 "d)  ----------------V------o----x------  non-buggy")


    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["4"].join(" "), # fixed
                                                 ["4"].join(" ")  # found
                                                 ),
                 "d')  ----------------V----------(xo)------  ? buggy")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["4"].join(" "), # fixed
                                                 ["5"].join(" ")  # found
                                                 ),
                 "e)  ----------------V------x----o------  buggy")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 [""].join(" "), # fixed
                                                 ["2"].join(" ")  # found
                                                 ),
                 "f)  ----------o-----V--- ? -- ? -- ? --  buggy")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["2","4"].join(" "), # fixed
                                                 ["5"].join(" ")  # found
                                                 ),
                 "g)  ----------x-----V--- ? -- ? -- ? --  non-buggy")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["2","5"].join(" "), # fixed
                                                 ["1","4"].join(" ")  # found
                                                 ),
                 "h)  -----o----x-----V--- ? -- ? -- ? --  non-buggy")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3",
                                                 ["1","5"].join(" "), # fixed
                                                 ["2","4"].join(" ")  # found
                                                 ),
                 "i)  -----x----o-----V--- ? -- ? -- ? --  buggy")

    # is this a bug?
    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "1.5",
                                                 ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                 ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                 ),
                 "Test a case for buggy package")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "2.0",
                                                 ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                 ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                 ),
                 "Test a case for non-buggy package")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "1.0",
                                                 ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                 ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                 ),
                 "Test case for ooooold package which is before bug is initially found")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "1.0",
                                                 "apt-test/2.0", #fixed
                                                 nil #found
                                                 ),
                 "Test case for package which bug is found in ancient day and fixed sometime in the future.")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3.0",
                                                 "apt-test/2.0", # fixed
                                                 nil #found
                                                 ),
                 "Test case for package which bug is found in ancient day and fixed sometime in the past.")

    assert_equal(true,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3.0",
                                                 nil,
                                                 "apt-test/1.0"
                                                 ),
                 "Test case for package which bug is found and no fixed date is set.")

    assert_equal(false,
                 Factory::BugsFactory.am_i_buggy(
                                                 "3.0",
                                                 "4.0", #fixed
                                                 "3.5" #found
                                                 ),
                 "Bug is found and fixed in a future version")

  end

  def test_bug_is_irrelevant
    # bug should be 'true' if that is non-relevant i.e. non-embugging

    assert_equal(true,
                 # returns true if this bug can be ignored
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "1.5", "2.0",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test that normal case where upgrade fixes a bug.")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "1.0", "2.5",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test the case where upgrade skips over a bug-period")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "3.0", "3.5",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "Test the case where the latest upgrade fixes a bug")


    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "2.5", "3.0",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "Test the case where the latest upgrade introduces a bug")


    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "1.0", "3.0",
                                                        123458,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test the case where upgrade jumps from non-buggy to reopened buggy")

    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "1.0", "1.7",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test the case where upgrade jumps into a bug region")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("audacious",
                                                        "1.5.0-1", "1.5.0-2",
                                                        123456,
                                                        "",
                                                        ["audacious/1.4.6-2", "audacious/1.5.0-2"].join(" ")  # found
                                                        ),
                 "between two found versions, and I am already buggy in the previous installation, #485109")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "1.7", "3.0",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test the case where upgrade jumps from buggy-buggy")

    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        nil, "3.0",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test new install into buggy version")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        nil, "3.5",
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test new install into non-buggy version")

    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("perl",
                                                        "5.10.0-10", "5.10.0-11",
                                                        492816,
                                                        nil, # fixed
                                                        ["perl/5.10.0-11"].join(" ")  # found
                                                        ),
                 "Test an upgrade to a newly found unfixed bug.")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt",
                                                        "0.7.14", nil,
                                                        482476,
                                                        nil,
                                                        ["apt/0.7.11", "apt/0.6.46.4-0.1"].join(" "),  # found
                                                        "done"
                                                        ),
                 "Say this bug is 'done' without version info")

    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt",
                                                        "0.7.14", nil,
                                                        482476,
                                                        "0.7.15",
                                                        ["apt/0.7.11", "apt/0.6.46.4-0.1"].join(" "),  # found
                                                        "done"
                                                        ),
                 "Say this bug is 'done' with version info")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "0.1", "0.2",
                                                        4321,
                                                        "0.5", #fixed
                                                        "0.3",  # found
                                                        "done"
                                                        ),
                 "Test bug where bug is found and fixed in future version.")

  end
end

class TC_BugsFactory_List < Test::Unit::TestCase
  def setup
    Factory::config=Mock_config_list.new
  end

  def teardown
    Factory::config=nil
  end

  def test_bug_is_irrelevant_list
    assert_equal(false,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "3.0", nil,
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test apt-listbugs list with buggy version")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "3.5", nil,
                                                        123456,
                                                        ["apt-test/2.0", "apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/1.5", "apt-test/3.0"].join(" ")  # found
                                                        ),
                 "test apt-listbugs list with non-buggy version")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "0.7.14+b1", nil,
                                                        482476,
                                                        ["0.7.14"].join(" "), # fixed
                                                        ["apt/0.7.11", "apt/0.6.46.4-0.1"].join(" ")  # found
                                                        ),
                 "test apt-listbugs list apt, apt bug is fixed")

    assert_equal(true,
                 Factory::BugsFactory.bug_is_irrelevant("apt-test",
                                                        "3.0", nil,
                                                        123456,
                                                        ["apt-test/3.5"].join(" "), # fixed
                                                        ["apt-test/3.2"].join(" ")  # found
                                                        ),
                 "Ignore bug that is found newer than this version")


  end
end

class TC_AppConfig < Test::Unit::TestCase
  def setup
    @a=AppConfig.new
  end

  def teardown
  end

  def test_statmap
    assert_equal("my_undefined",
                 @a.statmap("my_undefined"),
                 "test how it handles undefined value")
    assert_equal("unfixed",
                 @a.statmap("pending"),
                 "test how translation works")
  end
end
