
#define LOCAL_DEBUG
#include "debug.h"

#include "meta.h"
#include "conn.h"
#include "job.h"
#include "header.h"
#include "dlcon.h"
#include "acbuf.h"
#include "tcpconnect.h"

#include <sys/select.h>
#include <signal.h>
#include <string.h>
#include <errno.h>
#include <iostream>

using namespace MYSTD;

con::con(int fdId, const char *c) :
	m_confd(fdId),
    m_bStopActivity(false),
    m_pDlClient(NULL),
    m_pTmpHead(NULL),
    m_nLastIn(0),
    m_nLastOut(0)
    //,m_bNoLogs(false)
    
{
	if(c) // if NULL, pick up later when sent by the wrapper
		m_sClientHost=c;
	
    LOGSTART2("con::con", "fd: " << fdId << ", clienthost: " << c);

    
#ifdef KILLABLE
    wakepipe[0]=wakepipe[1]=-1;
    
    if(pipe(wakepipe) == 0) {
    	set_nb(wakepipe[0]);
    	set_nb(wakepipe[1]);
    }
    else
    	m_bStopActivity=true;
#endif
    
};

#ifdef KILLABLE
void con::SignalStop() {
    setLockGuard;
    m_bStopActivity=true;
    POKE(wakepipe[1]); // for select
    notifyAll();
}
#endif


//void DumpItems();

con::~con() {
	LOGSTART("con::~con (Destroying connection...)");
	flushShutdownClose(m_confd);

	// our user's connection is released but the downloader task created here may still be serving others
	// tell it to stop when it gets the chance and delete it then

	MYSTD::list<job*>::iterator jit;
	for (jit=m_jobs2send.begin(); jit!=m_jobs2send.end(); jit++)
		delete *jit;

	_LogFlush();
	
    if(m_pDlClient) 
    {
    	m_pDlClient->SignalStop();
    	pthread_join(m_dlerthr, NULL);
    	
    	delete m_pDlClient;
    	m_pDlClient=NULL;
    }
    
    if(m_pTmpHead)
    {
    	delete m_pTmpHead;
    	m_pTmpHead=NULL;
    }
    				
    aclog::flush();
}

class SoapPassThrough : public tcpconnect
{
public:
	bool TrySoapPassThrough(acbuf &clientBufIn, int fdClient)
	{

#define POSTMARK "POST http://bugs.debian.org:80/"
#define FAKEMARK "POST                          /"

		if(clientBufIn.size()<_countof(POSTMARK)-1)
			return false;
		if(strncmp(clientBufIn.rptr(), POSTMARK, _countof(POSTMARK)-1))
			return false;

		string sErr;
		if(!_Connect("bugs.debian.org", sErr))
			return false;


		acbuf clientBufOut; // use custom buffers?
		clientBufOut.init(32*1024); // out to be enough for any BTS response

		// for convenience
		int fdServer = m_conFd;
		acbuf &serverBufOut = clientBufIn, &serverBufIn = clientBufOut;

		int maxfd=1+std::max(fdClient, fdServer);

		while (true)
		{
			fd_set rfds, wfds;
			FD_ZERO(&rfds);
			FD_ZERO(&wfds);

			// can send to client?
			if(clientBufOut.size()>0)
				FD_SET(fdClient, &wfds);

			// can receive from client?
			if(clientBufIn.freecapa()>0)
				FD_SET(fdClient, &rfds);

			if(serverBufOut.size()>0)
				FD_SET(fdServer, &wfds);

			if(serverBufIn.freecapa()>0)
				FD_SET(fdServer, &rfds);

			int nReady=select(maxfd, &rfds, &wfds, NULL, NULL);
			if (nReady<0)
				return false;

			if(FD_ISSET(fdServer, &wfds))
			{
				// if there is some other proxy, let it deal with the stuff
				if(!m_proxy)
				{
					serverBufOut.move();
					char *p = 0;
					while (0 != (p = strstr(serverBufOut.rptr(), POSTMARK)))
						memcpy(p, FAKEMARK, _countof(FAKEMARK) - 1);
				}

				if(serverBufOut.syswrite(fdServer)<0)
					return false;
			}

			if(FD_ISSET(fdClient, &wfds))
			{
				if(clientBufOut.syswrite(fdClient)<0)
					return false;
			}

			if(FD_ISSET(fdServer, &rfds))
			{
				if(serverBufIn.sysread(fdServer)<=0)
					return true;
			}

			if(FD_ISSET(fdClient, &rfds))
			{
				if(clientBufIn.sysread(fdClient)<=0)
					return true;
			}
		}

		return false;

	};
};

void con::WorkLoop() {

	LOGSTART("con::WorkLoop");
    
	signal(SIGPIPE, SIG_IGN);
	
    acbuf inBuf;
    inBuf.init(32*1024);
    
    int maxfd=m_confd;
    while(!m_bStopActivity) {
        fd_set rfds, wfds;
        FD_ZERO(&wfds);
        FD_ZERO(&rfds);
        
        FD_SET(m_confd, &rfds);
        if(inBuf.freecapa()==0)
        	return; // shouldn't even get here
        
        job *pjSender(NULL);
    
        if ( !m_jobs2send.empty())
		{
			pjSender=m_jobs2send.front();
			FD_SET(m_confd, &wfds);
		}
		
        
        ldbg("select con");

        struct timeval tv;
        tv.tv_sec = 90;
        tv.tv_usec = 0;
        int ready = select(maxfd+1, &rfds, &wfds, NULL, &tv);
        
        if(ready == 0)
        {
        	USRDBG(5, "Timeout occured, apt client disappeared silently?");
        	return;
        }
		else if (ready<0)
		{
			if (EINTR == errno)
				continue;
			
			ldbg("select error in con, errno: " << errno);
			return; // FIXME: good error message?
		}
        
        ldbg("select con back");

        if(FD_ISSET(m_confd, &rfds)) {
            int n=inBuf.sysread(m_confd);
            ldbg("got data: " << n <<", inbuf size: "<< inBuf.size());
            if(n<=0) // error, incoming junk overflow or closed connection
            {
              if(n==-EAGAIN)
                continue;
              else
                return;
            }
        }

        // split new data into requests
        while(inBuf.size()>0) {
        	MYTRY
        	{
				if(!m_pTmpHead)
					m_pTmpHead = new header();
				if(!m_pTmpHead)
					return; // no resources? whatever
				
				int nConsumed=m_pTmpHead->LoadFromBuf(inBuf.rptr(), inBuf.size());
				ldbg("header parsed how? " << nConsumed);
				if(nConsumed==0)
				{ // Either not enough data received, or buffer full; make space and retry
					inBuf.move();
					break;
				}
				if(nConsumed<0)
				{
					 // also must be identified before
					if( acfg::forwardsoap && ! m_sClientHost.empty() )
					{
						SoapPassThrough pt;
						if(pt.TrySoapPassThrough(inBuf, m_confd))
							return;
					}

					ldbg("Bad request: " << inBuf.rptr() );
					return;
				}
				if (nConsumed>0)
					inBuf.drop(nConsumed);
				
				if (m_sClientHost.empty()) // may come from wrapper... MUST identify itself
				{
					if(m_pTmpHead->h[header::XORIG] && *(m_pTmpHead->h[header::XORIG]))
					{
						m_sClientHost=m_pTmpHead->h[header::XORIG];
						continue; // OK
					}
					else
						return;
				}

				ldbg("Parsed REQUEST:" << m_pTmpHead->frontLine);
				ldbg("Rest: " << inBuf.size());
				job * j = new job(m_pTmpHead, this);
				m_pTmpHead=NULL; // owned by job
				j->PrepareDownload();
				m_jobs2send.push_back(j);
			}
        	MYCATCH(bad_alloc)
        	{
        		return;
        	}
        }
        
        if(inBuf.freecapa()==0)
        	return; // cannot happen unless being attacked

#ifdef KILLABLE
        if(FD_ISSET(wakepipe[0], &rfds))
		{
			//ldbg("Aufgeweckt, outbuf:" << outBuf);
			int tmp;
			while(read(wakepipe[0], &tmp, 1) > 0);
			continue;
		}
#endif

		if(FD_ISSET(m_confd, &wfds) && pjSender)
		{
			ldbg("Sending data of " << pjSender);
			switch(pjSender->SendData(m_confd, m_jobs2send.size()))
			{
				case(job::R_DISCON):
				{
					ldbg("Disconnect advise received, stopping connection");
					return;
				}
				case(job::R_DONE):
				{
					m_jobs2send.pop_front(); 						
					delete pjSender;
					pjSender=NULL;
		
					ldbg("Remaining jobs to send: " << m_jobs2send.size());
				}
				case(job::R_AGAIN):
				default:
					break;
			}
        }
	}
}

void * _StartDownloader(void *pVoidDler)
{
	static_cast<dlcon*>(pVoidDler) -> WorkLoop();
	return NULL;
}

bool con::SetupDownloader(const char *pszOrigin)
{
	if (m_pDlClient)
		return true;

	MYTRY
	{
		if(acfg::exporigin)
		{
			string sXff;
			if(pszOrigin)
			{
				sXff = *pszOrigin;
				sXff += ", ";
			}
			sXff+=m_sClientHost;
			m_pDlClient=new dlcon(false, &sXff);
		}
		else
			m_pDlClient=new dlcon(false);
		
		if(!m_pDlClient)
			return false;
	}
	MYCATCH(MYSTD::bad_alloc)
	{
		return false;
	}

	if (0==pthread_create(&m_dlerthr, NULL, _StartDownloader,
			(void *)m_pDlClient))
	{
		return true;
	}
	delete m_pDlClient;
	m_pDlClient=NULL;
	return false;
}


inline void con::_LogFlush()
{
	if (m_nLastIn>0)
		aclog::transfer(true, m_nLastIn, m_sLoggedClient.c_str(), m_sLoggedFile.c_str());
	
	if(m_nLastOut>0)
		aclog::transfer(false, m_nLastOut, m_sLoggedClient.c_str(), m_sLoggedFile.c_str());
	
	m_nLastIn=m_nLastOut=0;
}

void con::LogDataCounts(MYSTD::string & sFile, const char *xff, off_t nNewIn,
		off_t nNewOut)
{
	string sClient;
	if (!acfg::logxff || !xff) // not to be logged or not available
		sClient=m_sClientHost;
	else if (xff)
	{
		sClient=xff;
		trimString(sClient);
		string::size_type pos = sClient.find_last_of(SPACECHARS);
		if (pos!=stmiss)
			sClient.erase(0, pos+1);
	}
	
	if(sFile != m_sLoggedFile || sClient != m_sLoggedClient)
	{
		_LogFlush();
		m_sLoggedClient=sClient;
		m_sLoggedFile=sFile;
	}

	m_nLastIn+=nNewIn;
	m_nLastOut+=nNewOut;

}

