
#ifndef _acbuf_H
#define _acbuf_H

#include <limits.h>
#include <string.h>
#include "meta.h"

/*! \brief Helper class to maintain a memory buffer, i.e. a continuous block of bytes.
 * It also encapsulates some typical operations on it.
 */

class acbuf {
    public:
        inline acbuf() : r(0), w(0), m_nCapacity(0), m_buf(NULL) {};
    	inline ~acbuf() { if(m_buf) delete [] m_buf; }
    	inline bool empty() { return w==r;}
    	//! Count of the data inside
        inline unsigned int size() { return w-r;}
        //! Returns capacity
        inline unsigned int freecapa() { return m_nCapacity-w; }
        //! Erase all characters after a certain position  
        inline void erase(size_t pos) { w=r+pos; if(r==w) clear(); }
        //! Invalidate data on the beginning, move reading position  
        inline void drop(size_t count) {  r+=count; if(r==w) clear(); }
        //! Mark externally appended data as valid, move writing position
        inline void got(size_t count) { w+=count;}
        //! for the worst case, move data to beginning to make space to read a complete new header
        inline void move() { if(0==r) return; memmove(m_buf, m_buf+r, w-r); w-=r; r=0; }
        //! Return pointer where to append data
        inline char *wptr() { return m_buf+w;};
        //! Return pointer to read valid data from
        inline char *rptr() { return m_buf+r; }
        //! like rptr but appends a NULL terminator
        inline char *c_str() {	m_buf[w]=0x0; return rptr();}
        //! Equivalent to drop(size()), drops all data
        inline void clear() {w=r=0;}
        
        //! Allocate needed memory
        bool init(unsigned int capa);
        bool initFromFile(const char *path);
                
        /*!
         * Writes to a (non-blocking) file descriptor, cares about EAGAIN and updates
         * position indexes. Optional chunklen variable specifies
         * the maximum of data to write.
         *
         * \param fd File descriptor to write to
         * \param maxlen Maximum amount of data to write
         * \return Number of written bytes, negative on failures, see write(2)
         */
        int syswrite(int fd, unsigned int maxlen=UINT_MAX);

        /*
         * Reads from a file descriptor and append to buffered data, update position indexes.
         * \param fd File descriptor
         * \return Number of read bytes, negative on failures, see read(2)
         */
        int sysread(int fd);


    private:

        size_t r, w, m_nCapacity; // read/write positions, size
        char *m_buf;

    	acbuf(const acbuf&); // don't copy me
    	acbuf& operator=(const acbuf&); 
        
};

#endif
