// ---------------------------------------------------------------------------
// - InputFile.hpp                                                           -
// - standard object library - input file class definition                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_INPUTFILE_HPP
#define  ALEPH_INPUTFILE_HPP

#ifndef  ALEPH_INPUT_HPP
#include "Input.hpp"
#endif

namespace aleph {

  /// The InputFile class is derived from the Input base base class and 
  /// provide a facility for reading file. The file is open at construction
  /// and closed at destruction or after a specific call to the close method.
  /// Sequential access is provided with the lseek method.
  /// @author amaury darsch

  class InputFile : public Input {
  private:
    /// the file name
    String d_name;
    /// the stream id
    int    d_sid;

  public:
    /// create a new input file by name
    /// @param name the file name
    InputFile (const String& name);

    /// close and destroy this input stream
    ~InputFile (void);

    /// @return the class name
    String repr (void) const;

    /// @return the stream descriptor
    int getsid (void) const;

    /// @return the next available character
    char read (void);

    /// @return true if we are at the eof
    bool iseof (void) const;

    /// check if we can read a character
    /// @param tout the timeout value
    bool valid (const long tout) const;

    /// @return the file name for this file stream
    String getname (void) const;

    /// close this input file and mark the eof marker
    bool close (void);

    /// place the file marker at a certain position
    /// @param pos the position to go
    void lseek (const long pos);

    /// @return the input file size
    long length (void) const;

    /// generate a new input file
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// apply this input file with a set of arguments and a quark
    /// @param robj   robj the current runnable
    /// @param nset   the current nameset    
    /// @param quark  the quark to apply these arguments
    /// @param argv   the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);

  private:
    // make the copy constructor private
    InputFile (const InputFile&);
    // make the assignment operator private
    InputFile& operator = (const InputFile&);
  };
}

#endif
