// ---------------------------------------------------------------------------
// - Boolean.cpp                                                             -
// - standard object library - boolean class implementation                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Input.hpp"
#include "Vector.hpp"
#include "Recycle.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // the boolean supported quarks
  static const long QUARK_EQL      = String::intern ("==");
  static const long QUARK_NEQ      = String::intern ("!=");

  // the boolean recycler
  static Recycle recycler;

  // allocate a new boolean
  void* Boolean::operator new (const t_size size) {
    return recycler.pop (size);
  }

  // delete a boolean
  void Boolean::operator delete (void* handle) {
    recycler.push (handle);
  }

  // create a new boolean - the initial value is false
  
  Boolean::Boolean (void) {
    d_value = false;
  }
  
  // create a boolean from a native value
  
  Boolean::Boolean (const bool value) {
    d_value = value;
  }
  
  // create a boolean from a string

  Boolean::Boolean (const String& value) {
    if (value == "false") 
      d_value = false;
    else if (value == "true")
      d_value = true;
    else
      throw Exception ("literal-error","illegal boolean value",value);
  }

  // copy constructor for this boolean

  Boolean::Boolean (const Boolean& that) {
    d_value = that.d_value;
  }

  // return the class name

  String Boolean::repr (void) const {
    return "Boolean";
  }

  // return a literal representation of this boolean
  
  String Boolean::toliteral (void) const {
    return tostring ();
  }

  // return a string representation of this boolean
  
  String Boolean::tostring (void) const {
    return d_value ? "true" : "false";
  }

  // return a clone of this boolean

  Object* Boolean::clone (void) const {
    return new Boolean (*this);
  }

  // return the boolean serial code

  t_byte Boolean::serialid (void) const {
    return SERIAL_BOOL_ID;
  }

  // serialize this boolean

  void Boolean::wrstream (Output& os) const {
    rdlock ();
    char c = d_value ? 0x01 : nilc;
    os.write (c);
    unlock ();
  }

  // deserialize this boolean

  void Boolean::rdstream (Input& is) {
    wrlock ();
    char c = is.read ();
    d_value = (c == nilc) ? false : true;
    unlock ();
  }

  // return this boolean value

  bool Boolean::toboolean (void) const {
    rdlock ();
    bool result = d_value;
    unlock ();
    return result;
  }

  // assign a boolean with a native value

  Boolean& Boolean::operator = (const bool value) {
    d_value = value;
    return *this;
  }
  
  // assign a boolean with a boolean
  
  Boolean& Boolean::operator = (const Boolean& value) {
    d_value = value.d_value;
    return *this;
  }
  
  // compare a boolean with a native value
  
  bool Boolean::operator == (const bool value) const {
    return (d_value == value);
  }
  
  // compare two boolean
  
  bool Boolean::operator == (const Boolean& value) const {
    return (d_value == value.d_value);
  }
  
  // compare a boolean with a native value
  
  bool Boolean::operator != (const bool value) const {
    return (d_value != value);
  }
  
  // compare two boolean
  
  bool Boolean::operator != (const Boolean& value) const {
    return (d_value != value.d_value);
  }

  // evaluate an object to a boolean value

  bool Boolean::evalto (Runnable* robj, Nameset* nset, Object* object) {
    Object* obj = (object == nilp) ? nilp : object->eval (robj, nset);
    Boolean* val = dynamic_cast <Boolean*> (obj);
    if (val == nilp) throw Exception ("type-error", "nil object to evaluate");
    return val->toboolean ();
  }

  // create a new boolean in a generic way

  Object* Boolean::mknew (Vector* argv) {
    if ((argv == nilp) || (argv->length () == 0)) return new Boolean;
    if (argv->length () != 1) 
      throw Exception ("argument-error", 
		       "too many argument with boolean constructor");
    // try to map the boolean argument
    Object* obj = argv->get (0);
    if (obj == nilp) return new Boolean;

    // try a boolean object
    Boolean* bval = dynamic_cast <Boolean*> (obj);
    if (bval != nilp) return new Boolean (*bval);

    // try a string object
    String* sval = dynamic_cast <String*> (obj);
    if (sval != nilp) return new Boolean (*sval);

    // illegal object
    throw Exception ("type-error", "illegal object with boolean constructor",
		     obj->repr ());
  }

  // operate this boolean with another object

  Object* Boolean::oper (Runnable* robj, t_oper type, Object* object) {
    Boolean* bobj = dynamic_cast <Boolean*> (object);
    switch (type) {
    case Object::EQL:
      if (bobj != nilp) return new Boolean (d_value == bobj->d_value);
      break;
    case Object::NEQ:
      if (bobj != nilp) return new Boolean (d_value != bobj->d_value);
      break;
    default:
      throw Exception ("operator-error", "unsupported boolean operator");
    }
    throw Exception ("type-error", "invalid operand with boolean",
		     Object::repr (object));
  }

  // set an object to this boolean

  Object* Boolean::vdef (Runnable* robj, Nameset* nset, Object* object) {
    Boolean* bobj = dynamic_cast <Boolean*> (object);
    if (bobj != nilp) {
      d_value = bobj->d_value;
      return this;
    }
    throw Exception ("type-error", "invalid object with boolean vdef",
		     Object::repr (object));
  }

  // apply this boolean with a set of arguments and a quark

  Object* Boolean::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch one argument
    if (argc == 1) {
      if (quark == QUARK_EQL) return oper (robj, Object::EQL, argv->get (0));
      if (quark == QUARK_NEQ) return oper (robj, Object::NEQ, argv->get (0));
    }

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
