// ---------------------------------------------------------------------------
// - Cookie.cpp                                                              -
// - aleph:www library - http cookie class implementation                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Time.hpp"
#include "Vector.hpp"
#include "Cookie.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // the cookie supported quarks
  static const long QUARK_GETNAME    = String::intern ("get-name");
  static const long QUARK_SETNAME    = String::intern ("set-name");
  static const long QUARK_GETVALUE   = String::intern ("get-value");
  static const long QUARK_SETVALUE   = String::intern ("set-value");
  static const long QUARK_GETMAGE    = String::intern ("get-max-age");
  static const long QUARK_SETMAGE    = String::intern ("set-max-age");
  static const long QUARK_GETPATH    = String::intern ("get-path");
  static const long QUARK_SETPATH    = String::intern ("set-path");
  static const long QUARK_GETDOMAIN  = String::intern ("get-domain");
  static const long QUARK_SETDOMAIN  = String::intern ("set-domain");
  static const long QUARK_GETCOMMENT = String::intern ("get-comment");
  static const long QUARK_SETCOMMENT = String::intern ("set-comment");
  static const long QUARK_GETSECURE  = String::intern ("get-secure");
  static const long QUARK_SETSECURE  = String::intern ("set-secure");
  static const long QUARK_TOSTRING   = String::intern ("to-string");

  // create a cookie with a name/value pair

  Cookie::Cookie (const String& name, const String& value) {
    d_name   = name;
    d_value  = value;
    d_mage   = -1;
    d_secure = false;
  }

  // create a cookie with a name/value pair

  Cookie::Cookie (const String& name, const String& value, long mage) {
    d_name   = name;
    d_value  = value;
    d_mage   = (mage < 0) ? -1 : mage;
    d_secure = false;
  }

  // return the class name

  String Cookie::repr (void) const {
    return "Cookie";
  }

  // return a http cookie string representation

  String Cookie::tostring (void) const {
    rdlock ();
    if ((d_name.length () == 0) || (d_value.length () == 0)) {
      unlock ();
      throw Exception ("cookie-error", "invalid cookie name or value");
    }
    String result = "Set-Cookie: ";
    result = result + d_name.toliteral () + '=' + d_value.toliteral ();
    if (d_mage >= 0) {
      result = result + "; Max-Age=" + d_mage;
    }
    if (d_path.length () != 0) {
      result = result + "; Path=" + d_path.toliteral ();
    }
    if (d_domain.length () != 0) {
      result = result + "; Domain=" + d_domain.toliteral ();
    }
    if (d_comment.length () != 0) {
      result = result + "; Comment=" + d_comment.toliteral ();
    }
    if (d_mage > 0) {
      Time time;
      time.add (d_mage);
      result = result + "; Expires=" + time.fmtcookie ();
    }
    if (d_secure == true) {
      result = result + "; Secure";
    }
    unlock ();
    return result;
  }

  // set the cookie name
  
  void Cookie::setname (const String& name) {
    wrlock ();
    d_name = name;
    unlock ();
  }

  // get the cookie name

  String Cookie::getname (void) const {
    rdlock ();
    String result = d_name;
    unlock ();
    return result;
  }

  // set the cookie value
  
  void Cookie::setvalue (const String& value) {
    wrlock ();
    d_value = value;
    unlock ();
  }

  // get the cookie value

  String Cookie::getvalue (void) const {
    rdlock ();
    String result = d_value;
    unlock ();
    return result;
  }

  // set the cookie maximum age
  
  void Cookie::setmage (const long mage) {
    wrlock ();
    d_mage = (mage < 0) ? -1 : mage;
    unlock ();
  }

  // get the cookie maximum age

  long Cookie::getmage (void) const {
    rdlock ();
    long result = d_mage;
    unlock ();
    return result;
  }

  // set the cookie path
  
  void Cookie::setpath (const String& path) {
    wrlock ();
    d_path = path;
    unlock ();
  }

  // get the cookie path

  String Cookie::getpath (void) const {
    rdlock ();
    String result = d_path;
    unlock ();
    return result;
  }

  // set the cookie domain
  
  void Cookie::setdomain (const String& domain) {
    wrlock ();
    d_domain = domain;
    unlock ();
  }

  // get the cookie domain

  String Cookie::getdomain (void) const {
    rdlock ();
    String result = d_domain;
    unlock ();
    return result;
  }

  // set the cookie comment
  
  void Cookie::setcomment (const String& comment) {
    wrlock ();
    d_comment = comment;
    unlock ();
  }

  // get the cookie comment

  String Cookie::getcomment (void) const {
    rdlock ();
    String result = d_comment;
    unlock ();
    return result;
  }

  // set the cookie secure flag
  
  void Cookie::setsecure (const bool flag) {
    wrlock ();
    d_secure = flag;
    unlock ();
  }

  // get the cookie secure flag

  bool Cookie::getsecure (void) const {
    rdlock ();
    bool result = d_secure;
    unlock ();
    return result;
  }
  
  // create a new cookie object in a generic way

  Object* Cookie::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for two arguments
    if (argc == 2) {
      String name  = argv->getstring (0);
      String value = argv->getstring (1);
      return new Cookie (name, value);
    }
 
    // check for three arguments
    if (argc == 3) {
      String name  = argv->getstring (0);
      String value = argv->getstring (1);
      long   mage  = argv->getint    (2);
      return new Cookie (name, value, mage);
    }
    // wrong arguments
    throw Exception ("argument-error", "invalid arguments with cookie");
  }

  // apply this cookie object with a set of arguments and a quark

  Object* Cookie::apply (Runnable* robj, Nameset* nset, const long quark,
			 Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETNAME)    return new String  (getname    ());
      if (quark == QUARK_GETVALUE)   return new String  (getvalue   ());
      if (quark == QUARK_GETMAGE)    return new Integer (getmage    ());
      if (quark == QUARK_GETPATH)    return new String  (getpath    ());
      if (quark == QUARK_GETDOMAIN)  return new String  (getdomain  ());
      if (quark == QUARK_GETCOMMENT) return new String  (getcomment ());
      if (quark == QUARK_GETSECURE)  return new Boolean (getsecure  ());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_SETVALUE) {
	String value = argv->getstring (0);
	setvalue (value);
	return nilp;
      }
      if (quark == QUARK_SETMAGE) {
	long mage = argv->getint (0);
	setmage (mage);
	return nilp;
      }      
      if (quark == QUARK_SETPATH) {
	String path = argv->getstring (0);
	setpath (path);
	return nilp;
      }      
      if (quark == QUARK_SETDOMAIN) {
	String domain = argv->getstring (0);
	setdomain (domain);
	return nilp;
      }
      if (quark == QUARK_SETCOMMENT) {
	String comment = argv->getstring (0);
	setcomment (comment);
	return nilp;
      }
      if (quark == QUARK_SETSECURE) {
	bool flag = argv->getbool (0);
	setsecure (flag);
	return nilp;
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
