//==============================================
//  copyright            : (C) 2003-2005 by Will Stokes
//==============================================
//  This program is free software; you can redistribute it 
//  and/or modify it under the terms of the GNU General 
//  Public License as published by the Free Software 
//  Foundation; either version 2 of the License, or  
//  (at your option) any later version.         
//==============================================

//Projectwide includes
#include "panningPreviewInterface.h"

//==============================================
PanningPreviewInterface::PanningPreviewInterface( QString imageFilename, 
                                                  QWidget *parent, const char* name ) : 
                                                  SplitViewInterface (parent, name )
{
  //load full size image
  fullSizeImage = QImage( imageFilename );
          
  //a 0-width selection is invalid and prevents 
  //resize events from triggering painting
  //until the true selection region is set
  selection.setWidth( 0 );
}
//==============================================
QSize PanningPreviewInterface::sizeHint() const
{
  //subjetively chosen default size :)
  return QSize( 500, 400 );
}
//==============================================
QSize PanningPreviewInterface::paintingSize()
{
  return QSize( QMIN( fullSizeImage.width(),  size().width()  ),
                QMIN( fullSizeImage.height(), size().height() ) );
}
//==============================================
void PanningPreviewInterface::resizeEvent( QResizeEvent * )
{
  //center of new selection...
  QPoint center;
  
  //if selection not set then default to center of image
  if( selection.width() == 0)
  {
    //compute center selection center
    center = QPoint( fullSizeImage.width() / 2,
                     fullSizeImage.height() / 2 );
  }
  //else construct new selection that is centered over old selection
  else
  {
    //compute center selection center
    center = QPoint( selection.left() + selection.width()/2,
                     selection.top() + selection.height()/2 );
  }
  
  //determine width/height that will be used for painting
  QSize actualSize = paintingSize();

  //compute new selection area centerd over old selection region
  QRect newSelection;
  newSelection.setLeft( center.x() - actualSize.width() /2 );
  newSelection.setTop ( center.y() - actualSize.height()/2 );
  newSelection.setRight( newSelection.left() + actualSize.width()  - 1 );
  newSelection.setBottom( newSelection.top() + actualSize.height() - 1 );
  
  //set selection which will result in regenerating of orig and adjusted images
  setSelection( newSelection ); 
}
//==============================================
void PanningPreviewInterface::setSelection( QRect s )
{
  //set the selection
  selection = s; 

  //get the available painting size
  QSize actualSize = paintingSize();

  //if too wide or tall shrink selection
  if( selection.width() > actualSize.width() )
    selection.setRight( selection.left() + actualSize.width() - 1 );
  if( selection.height() > actualSize.height() )
    selection.setBottom( selection.top() + actualSize.height() - 1 );

 //shift selection area if it extends beyond image boundary
  if( selection.left() < 0 )
    selection.moveBy( -selection.left(), 0 );
  
  if( selection.top() < 0 )
    selection.moveBy( 0, -selection.top() );
  
  if( selection.right() > fullSizeImage.width()-1 )
    selection.moveBy( (fullSizeImage.width()-1) - selection.right(), 0 );

  if( selection.bottom() > fullSizeImage.height()-1 )
    selection.moveBy( 0, (fullSizeImage.height()-1) - selection.bottom() );
  
  //regenerate orig and adjusted images
  generateOrigImage();
}
//==============================================
void PanningPreviewInterface::generateOrigImage()
{
  //generate orig image
  //set adjusted image to null so repain won't occur until it is reset
  setImages( fullSizeImage.copy( selection.left(), selection.top(),
                                 selection.width(), selection.height() ),
             QImage() );
  
  //emit signal indicating adjusted image is out of date
  emit selectionChanged();
}
//==============================================
QRect PanningPreviewInterface::getSelection()
{ return selection; }
//==============================================

