#   Alacarte Menu Editor - Simple fd.o Compliant Menu Editor
#   Copyright (C) 2005  Travis Watkins
#
#   This library is free software; you can redistribute it and/or
#   modify it under the terms of the GNU Library General Public
#   License as published by the Free Software Foundation; either
#   version 2 of the License, or (at your option) any later version.
#
#   This library is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#   Library General Public License for more details.
#
#   You should have received a copy of the GNU Library General Public
#   License along with this library; if not, write to the Free Software
#   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

from config import *
import Alacarte.MenuHandler
import xdg.Config, xdg.MenuEditor, xdg.BaseDirectory, xdg.Menu
import re, xml.dom.minidom, os

class MenuHandler(Alacarte.MenuHandler.MenuHandler, xdg.MenuEditor.MenuEditor):
	def __init__(self, filename, options):
		super(MenuHandler, self).__init__(filename, options)
		xdg.Config.setWindowManager(options['desktop_environment'])
		xdg.MenuEditor.MenuEditor.__init__(
			self, filename, root=options['root_mode']
			)
		if self.menu.Name != 'Applications':
			path = os.path.join(
				xdg.BaseDirectory.xdg_config_dirs[0], 'menus',
				os.path.split(filename)[1]
				)
			if not os.path.isdir(os.path.split(path)[0]):
				os.makedirs(os.path.split(path)[0])
			open(path, 'w').write(
				"<!DOCTYPE Menu PUBLIC '-//freedesktop//DTD Menu 1.0//EN' " + 
				"'http://standards.freedesktop.org/menu-spec/menu-1.0.dtd'>" + 
				"\n<Menu>\n  <Name>" + self.menu.Name + "</Name>\n" +
				"  <MergeFile type=\"parent\">" + self.menu.Filename +
				"</MergeFile>\n</Menu>"
				)
			xdg.MenuEditor.MenuEditor.__init__(
				self, path, root=options['root_mode']
				)
		self.root = MenuItem(
			self.menu, ALACARTE_IS_ROOT_MENU, ALACARTE_IS_SYSTEM
			)
		self.root.handler = self

	def convertActionToFileType(self, item):
		if isinstance(item, xdg.Menu.Menu) and item.Name == 'Other':
			return ALACARTE_IS_LOCKED
		action = self.getAction(item)
		if action == "delete":
			return ALACARTE_IS_LOCAL
		elif action == "revert":
			return ALACARTE_IS_SYSTEM_WITH_LOCAL
		return ALACARTE_IS_SYSTEM

	def copyAppDirs(self, oldParent, newParent, menu=None):
		if menu:
			master = menu.AppDirs
		else:
			master = self.menu.AppDirs
		newMenu = self._MenuEditor__getXmlMenu(newParent.getPath(True, True))
		for appdir in oldParent.AppDirs:
			if appdir not in newParent.AppDirs:
				if menu or appdir not in master:
					self._MenuEditor__addXmlTextElement(
						newMenu, 'AppDir', appdir
						)

	def fixAppDirs(self, item):
		for parent in item.Parents:
			menu = self._MenuEditor__getXmlMenu(parent.getPath(True, True))
			appdir_path = os.path.join(
				xdg.BaseDirectory.xdg_data_dirs[0], 'applications'
				)
			for node in menu.childNodes:
				if node.nodeType == xml.dom.minidom.Node.ELEMENT_NODE:
					if node.tagName == 'AppDir':
						for child in node.childNodes:
							if child.nodeType == xml.dom.minidom.Node.TEXT_NODE:
								if child.nodeValue == appdir_path:
									return
			self._MenuEditor__addXmlTextElement(menu, 'AppDir', appdir_path)

	def copyDirectoryDirs(self, oldParent, newParent, menu=None):
		if menu:
			master = menu.DirectoryDirs
		else:
			master = self.menu.DirectoryDirs
		newMenu = self._MenuEditor__getXmlMenu(newParent.getPath(True, True))
		for dirdir in oldParent.DirectoryDirs:
			if dirdir not in newParent.DirectoryDirs:
				if menu or dirdir not in master:
					self._MenuEditor__addXmlTextElement(
						newMenu, 'DirectoryDir', dirdir
						)

	def fixDirectoryDirs(self, item):
		menu = self._MenuEditor__getXmlMenu(item.getPath(True, True))
		dirdir_path = os.path.join(
			xdg.BaseDirectory.xdg_data_dirs[0], 'desktop-directories'
			)
		for node in menu.childNodes:
			if node.nodeType == xml.dom.minidom.Node.ELEMENT_NODE:
				if node.tagName == 'DirectoryDir':
					for child in node.childNodes:
						if child.nodeType == xml.dom.minidom.Node.TEXT_NODE:
							if child.nodeValue == dirdir_path:
								return
		self._MenuEditor__addXmlTextElement(menu, 'DirectoryDir', dirdir_path)

	def revert(self):
		try:
			self.revertMenu(self.root)
		except:
			pass
		try:
			os.unlink(self.filename)
		except OSError:
			pass
		self.__init__(os.path.split(self.filename)[1], self.options)

	def revertMenu(self, menu):
		for child in menu.getChildren():
			if child.itemType == ALACARTE_IS_MENU:
				self.revertMenu(child)
			elif child.fileType == ALACARTE_IS_LOCAL:
				child.delete()
			elif child.fileType == ALACARTE_IS_SYSTEM_WITH_LOCAL:
				child.revert()
			elif child.fileType == ALACARTE_IS_SYSTEM:
				if child.itemType == ALACARTE_IS_SEPARATOR:
					child.delete()
		if menu.fileType == ALACARTE_IS_LOCAL:
			menu.delete()
		elif menu.fileType == ALACARTE_IS_SYSTEM_WITH_LOCAL:
			print 'Reverting', menu.getKey('Name'), 'menu.'
			menu.revert()
		elif menu.fileType == ALACARTE_IS_SYSTEM:
			return

	def save(self):
		nodes = self.doc.getElementsByTagName('Merge')
		for node in nodes:
			parent = node.parentNode
			parent.removeChild(node)
			parent.appendChild(node)
		xdg.MenuEditor.MenuEditor.save(self)
		return True

	def new(self, item, parent):
		#rather hackish way of allowing
		#  Alacarte.MenuHandler.new('Foo Menu', parent)
		#also allows drag-and-drop from gnome-vfs
		if parent.backendId.Name == 'Other':
			return False
		fd = open('/tmp/alacarte_temp.desktop', 'w')
		if '\n' in item:
			itemType = ALACARTE_IS_ENTRY
			fd.write(item)
		else:
			itemType = ALACARTE_IS_MENU
			temp = '[Desktop Entry]\nName=' + item + '\nType=Directory\n'
			temp += 'Encoding=UTF-8'
			fd.write(temp)
		fd.close()
		menuentry = xdg.Menu.MenuEntry('/tmp/alacarte_temp.desktop')
		item = MenuItem(menuentry, itemType, ALACARTE_IS_LOCAL)
		if itemType == ALACARTE_IS_MENU:
			item.backendId = self.createMenu(
				parent.backendId, item.getKey('Name'), None,
				item.getKey('Comment'),	item.getKey('Icon')
				)
		else:
			item.backendId = self.createMenuEntry(
				parent.backendId, item.getKey('Name'), item.getKey('Exec'),
				None, item.getKey('Comment'), item.getKey('Icon')
				)
		del menuentry
		os.unlink('/tmp/alacarte_temp.desktop')
		return item

	def newMenu(self, parent, name, comment, icon, before, after):
		if parent.backendId.Name == 'Other':
			return False
		if before:
			before = before.backendId
		if after:
			after = after.backendId
		item = MenuItem(None, ALACARTE_IS_MENU, ALACARTE_IS_LOCAL)
		item.backendId = self.createMenu(
			parent.backendId, name, None, comment, icon, after, before
			)
		return item

	def newEntry(
		self, parent, name, command, comment, icon, terminal, before, after
		):
		if parent.backendId.Name == 'Other':
			return False
		if before:
			before = before.backendId
		if after:
			after = after.backendId
		item = MenuItem(None, ALACARTE_IS_ENTRY, ALACARTE_IS_LOCAL)
		item.backendId = self.createMenuEntry(
			parent.backendId, name, command, None, comment, icon, terminal,
			after, before
			)
		return item

	def newSeparator(self, parent, before, after):
		if parent.backendId.Name == 'Other':
			return False
		if before:
			before = before.backendId
		if after:
			after = after.backendId
		item = MenuItem(None, ALACARTE_IS_SEPARATOR, ALACARTE_IS_SYSTEM)
		item.backendId = self.createSeparator(parent.backendId, after, before)
		return item

	#move entry from one menu tree to another
	def move(self, otherHandler, oldParent, newParent, item, before, after):
		oldParent = oldParent.backendId
		newParent = newParent.backendId
		if newParent.Name == 'Other':
			return False
		item = item.backendId
		if before:
			before = before.backendId
		if after:
			after = after.backendId
		otherHandler.editMenuEntry(item, hidden = True)
		deskentry = item.DesktopEntry
		entry = self.createMenuEntry(
			newParent, deskentry.getName(), deskentry.getExec(),
			deskentry.getGenericName(), deskentry.getComment(),
			deskentry.getIcon(), deskentry.getTerminal(), after, before
			)
		return MenuItem(entry, ALACARTE_IS_ENTRY, ALACARTE_IS_LOCAL)

	#UGLY AS SIN BUT IT WORKS SO DON'T TOUCH IT
	def _MenuEditor__undoMoves(self, element, old, new):
		nodes = []
		matches = []
		originalOld = old
		finalOld = old
		for node in self._MenuEditor__getXmlNodesByName(['Move'], element):
			nodes = [node,] + nodes
		for node in nodes:
			oldXml = node.getElementsByTagName('Old')[0]
			newXml = node.getElementsByTagName('New')[0]
			if newXml.childNodes[0].nodeValue == old:
				matches.append(node)
				old = newXml.childNodes[0].nodeValue
				finalOld = oldXml.childNodes[0].nodeValue
		for node in matches:
			element.removeChild(node)
		if len(matches) > 0:
			for node in nodes:
				oldXml = node.getElementsByTagName('Old')[0]
				newXml = node.getElementsByTagName('New')[0]
				path = os.path.split(newXml.childNodes[0].nodeValue)
				if path[0] == originalOld:
					element.removeChild(node)
					for node in self.doc.getElementsByTagName('Menu'):
						nameNode = node.getElementsByTagName('Name')[0]
						name = nameNode.childNodes[0].nodeValue
						if name == os.path.split(new)[1]:
							newMenu = self._MenuEditor__getXmlMenu(
								'Applications/' + new
								)
							for node2 in node.getElementsByTagName('AppDir'):
								newMenu.appendChild(node2)
							for node2 in node.getElementsByTagName(
								'DirectoryDir'
								):
								newMenu.appendChild(node2)
							parent = node.parentNode
							parent.removeChild(node)
					node = self.doc.createElement('Move')
					node.appendChild(
						self._MenuEditor__addXmlTextElement(
							node, 'Old', oldXml.childNodes[0].nodeValue
							)
						)
					node.appendChild(
						self._MenuEditor__addXmlTextElement(
							node, 'New', os.path.join(new, path[1])
							)
						)
					element.appendChild(node)
			if finalOld == new:
				return True
			node = self.doc.createElement('Move')
			node.appendChild(self.__addXmlTextElement(node, 'Old', finalOld))
			node.appendChild(self.__addXmlTextElement(node, 'New', new))
			return element.appendChild(node)

	def _MenuEditor__addXmlMove(self, element, old, new):
		if not self._MenuEditor__undoMoves(element, old, new):
			super(MenuHandler, self)._MenuEditor__addXmlMove(element, old, new)

class MenuItem(Alacarte.MenuHandler.MenuItem):
	def getKey(self, key):
		super(MenuItem, self).getKey(key)
		if self.itemType == ALACARTE_IS_SEPARATOR:
			return '-----'
		if key in ('Name', 'Comment', 'Icon'):
			if self.itemType == ALACARTE_IS_ENTRY:
				return self.backendId.DesktopEntry.get(key, locale=True)
			else:
				try:
					return self.backendId.Directory.DesktopEntry.get(
						key, locale=True
						)
				except:
					if key == 'Name':
						return self.backendId.Name
		else:
			if self.itemType == ALACARTE_IS_ENTRY:
				return self.backendId.DesktopEntry.get(key)
			else:
				try:
					return self.backendId.Directory.DesktopEntry.get(key)
				except:
					pass
		return ''

	def setKey(self, key, value):
		super(MenuItem, self).setKey(key, value)
		if key in ('Name', 'Comment', 'Icon'):
			if self.itemType == ALACARTE_IS_ENTRY:
				self.backendId.DesktopEntry.set(key, value, locale=True)
				self.backendId.DesktopEntry.set(key, value)
			else:
				self.backendId.Directory.DesktopEntry.set(
					key, value, locale=True
					)
				self.backendId.Directory.DesktopEntry.set(key, value)
		else:
			if self.itemType == ALACARTE_IS_ENTRY:
				self.backendId.DesktopEntry.set(key, value)
			else:
				self.backendId.Directory.DesktopEntry.set(key, value)

	def getChildren(self):
		super(MenuItem, self).getChildren()
		if self.itemType in (ALACARTE_IS_MENU, ALACARTE_IS_ROOT_MENU):
			for child in self.backendId.getEntries(True):
				fileType = self.handler.convertActionToFileType(child)
				if isinstance(child, xdg.Menu.Menu):
					if child.getName() != '':
						item = MenuItem(child, ALACARTE_IS_MENU, fileType)
						item.handler = self.handler
						yield item
				elif isinstance(child, xdg.Menu.MenuEntry):
					if '-usercustom' not in child.DesktopFileID:
						if child.Show == 'NoDisplay' or child.Show == True:
							desk = child.DesktopEntry
							item = MenuItem(child, ALACARTE_IS_ENTRY, fileType)
							item.handler = self.handler
							yield item
				elif isinstance(child, xdg.Menu.Separator):
					item = MenuItem(
						child, ALACARTE_IS_SEPARATOR, ALACARTE_IS_SYSTEM
						)
					item.handler = self.handler
					yield item

	def isVisible(self):
		super(MenuItem, self).isVisible()
		item = self.backendId
		if isinstance(item, xdg.Menu.Separator):
			return True
		if item.Show == True:
			return True
		return False

	def toggleVisibility(self):
		super(MenuItem, self).toggleVisibility()
		item = self.backendId
		if isinstance(item, xdg.Menu.MenuEntry):
			if item.Show == True:
				self.handler.hideMenuEntry(item)
			else:
				self.handler.unhideMenuEntry(item)
			self.handler.fixAppDirs(item)
		elif isinstance(item, xdg.Menu.Menu):
			if item.Name == 'Other':
				return False
			if item.Show == True:
				self.handler.hideMenu(item)
			elif item.Show == 'NoDisplay' or item.Show == 'Deleted':
				self.handler.unhideMenu(item)
			elif item.Show == 'Empty':
				return False
			self.handler.fixDirectoryDirs(item)
		else:
			return False
		return True

	def move(self, oldParent, newParent, before, after):
		super(MenuItem, self).move(oldParent, newParent, before, after)
		item = self.backendId
		oldParent = oldParent.backendId
		newParent = newParent.backendId
		if before != None:
			before = before.backendId
		if after != None:
			after = after.backendId
		if newParent.Name == 'Other':
			return False
		if isinstance(item, xdg.Menu.Menu):
			if item == newParent:
				return False
		if not isinstance(item, xdg.Menu.Separator):
			if oldParent != newParent:
				self.handler.copyAppDirs(oldParent, newParent)
			if isinstance(item, xdg.Menu.MenuEntry):
				self.handler.moveMenuEntry(
					item, oldParent, newParent, after, before
					)
			elif isinstance(item, xdg.Menu.Menu):
				parent = newParent
				while parent != self.handler.menu:
					if parent == item:
						return False
					parent = parent.Parent
				self.handler.moveMenu(item, oldParent, newParent, after, before)
				if oldParent != newParent:
					self.handler.copyDirectoryDirs(oldParent, newParent, item)
			else:
				return False
		else:
			if oldParent != newParent:
				return False
			self.backendId = self.handler.moveSeparator(
				item, oldParent, after, before
				)
		return True

	def delete(self):
		super(MenuItem, self).delete()
		item = self.backendId
		if isinstance(item, xdg.Menu.MenuEntry):
			self.handler.deleteMenuEntry(item)
		elif isinstance(item, xdg.Menu.Menu):
			self.handler.deleteMenu(item)
		elif isinstance(item, xdg.Menu.Separator):
			self.handler.deleteSeparator(item)
		else:
			return False
		return True

	def save(self):
		super(MenuItem, self).save()
		item = self.backendId
		if self.itemType == ALACARTE_IS_ENTRY:
			desk = item.DesktopEntry
			item = self.handler.editMenuEntry(
				item, desk.getName(), None, desk.getComment(),
				desk.getExec(), desk.getIcon(),
				desk.getTerminal()
				)
			self.handler.fixAppDirs(item)
			self.handler.save()
		elif self.itemType == ALACARTE_IS_MENU:
			item = self.handler.editMenu(
				item, self.getKey('Name'), None, self.getKey('Comment'),
				self.getKey('Icon')
				)
			self.handler.fixDirectoryDirs(item)
			self.handler.save()
		else:
			return False
		return True

	def revert(self):
		super(MenuItem, self).revert()
		item = self.backendId
		if isinstance(item, xdg.Menu.MenuEntry):
			self.backendId = self.handler.revertMenuEntry(item)
		elif isinstance(item, xdg.Menu.Menu):
			self.backendId = self.handler.revertMenu(item)
		else:
			return False
		return True
