/*
    This file is part of AirSnort.

    AirSnort is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    AirSnort is distributed in the hope that it will be useful,
      but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with AirSnort; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/errno.h>
#include <string.h>
#include <asm/types.h>
#include <net/if.h>
#include <linux/netlink.h>
#include <netinet/in.h>

#include <features.h>  //GLIBC version number
#if __GLIBC__ >= 2 && __GLIBC_MINOR >= 1
#include <netpacket/packet.h>
#include <net/ethernet.h>
#else
#include <asm/types.h>
#include <linux/if_packet.h>
#include <linux/if_ether.h>
#endif

#include "PacketSource.h"
#include "capture.h"
#define MCAST_GRP_SNIFF 2

#define ETH_P_80211_RAW (ETH_P_ECONET + 1)

int use_PF_PACKET = 0;

void CreatePacketSource(PacketSource *ps) {
  ps->fd = -1;
}

void DestroyPacketSource(PacketSource *ps) {
  closePacketSource(ps);
}

int openPacketSource(PacketSource *ps) {
  int r;
  struct sockaddr_ll addr_ll;
  struct sockaddr_nl addr_nl;

  struct ifreq req;
  int myset = 10;  // socket option to reuse address

  if (use_PF_PACKET) {
    /*
     * Create socket with fd handle
     */
    if (( ps->fd = socket (PF_PACKET, SOCK_RAW, htons(ETH_P_80211_RAW))) < 0) {
      return -1;
    }
    /* get the interface index */
    memset(&req, 0, sizeof(struct ifreq));
    strcpy(req.ifr_name, dev);
    /*
     * Set socket options to specify receive timeout value and enable
     * reuseaddr.
     */
    if ((r = setsockopt (ps->fd, SOL_SOCKET, SO_REUSEADDR, &myset, sizeof(myset))) < 0) {
      return -1;
    }
    memset (&addr_ll, 0, sizeof (struct sockaddr_ll));
    addr_ll.sll_ifindex = req.ifr_ifindex;
    addr_ll.sll_protocol = htons(ETH_P_ALL);
    addr_ll.sll_family = AF_PACKET;

    if (bind(ps->fd, (struct sockaddr *) &addr_ll, sizeof (struct sockaddr_ll)) < 0) {
      return -1;
    }
  }
  else {  //using PF_NETLINK
    ps->fd=socket(PF_NETLINK,SOCK_RAW,MCAST_GRP_SNIFF);
    if (ps->fd < 0) {
      return -1;
    }
    addr_nl.nl_family=PF_NETLINK;
    addr_nl.nl_pid=getpid();
    addr_nl.nl_groups=MCAST_GRP_SNIFF;
    if (bind(ps->fd,(struct sockaddr *) &addr_nl, sizeof(struct sockaddr_nl)) < 0) {
      return -1;
    }
  }
  return(ps->fd);
}

void closePacketSource(PacketSource *ps) {
  if (ps->fd >= 0) {
    close(ps->fd);
  }
}

int getPacket(PacketSource *ps, char *buf, int maxlen, int timeout) {
  fd_set rs;
  int r;
  struct timeval tm;
  struct timeval *ptm;
  if (ps->fd < 0) {
    return(ERR_IO);
  }
  FD_ZERO(&rs);
  FD_SET(ps->fd, &rs);
  if (timeout >= 0) {
    tm.tv_sec = timeout / 1000;
    tm.tv_usec = (timeout % 1000) * 1000;
    ptm = &tm; 
  } else {
    ptm = NULL;
  }

  r = select(ps->fd + 1, &rs, NULL, NULL, ptm);

  if (r < 0) {
    return ERR_IO;
  }
  if (r == 0 && timeout >= 0) {
    return ERR_TIMEOUT;
  }
  if (FD_ISSET(ps->fd, &rs)) {
    if ((r = recv(ps->fd, buf, maxlen, 0)) < 0) {
      return ERR_IO;
    }
    return r;
  }	      
  return(ERR_TIMEOUT);

}

