/**********************************************************************
 ** generator - the generator class that takes in a pseudo-data file and
 **             generates an AIME data file
 **
 **
 **********************************************************************/

#ifndef GENERATOR_C
#define GENERATOR_C

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "generator.h"
#include "../../include/strings.h"
#include "../../include/lexer.h"
#include "../../include/config.h"
#include "../../include/mudtypes.h"
#include "../../include/sysdep.h"
#include "../../include/utils.h"

char *dir[10] = {"North", "South", "East", "West", "Up", "Down", "Northeast", 
		 "Northwest", "Southeast", "Southwest"};

#if 0
/***********************************************************************
 ** convert_file - the function that calls the appropriate conversion
 **                file based on what we are converting from
 **
 ** Parameters: read_file - the file to read in from
 **             write_filename - the filename to write the converted data to
 **             format - the format, if any, we are forcing recognition of
 **
 ** Returns: a pointer to the created string
 **
 ***********************************************************************/

int convert_file(FILE *read_file, char *write_filename, char *format, 
                                                            char *filename)
{
   Strings the_version;
   token_record *the_token;
   FILE *write_file;

   if (format != NULL)
   {
      the_version = format;
      if (!strcmp(the_version.str_show(), FULLVERSION))
      {
         printf("\nVersion already current.\n");
         return 0;
      }

/*      if (!verify_version(the_version.str_show()))
      {
         printf("\n\nThe stated version '%s' is not supported.\n\n",
                                            the_version.str_show());
         ask_version(&the_version);
         
      }
*/
      if ((write_file = fopen(write_filename, "w")) == NULL)
      {
         printf("Could not open write file '%s'\n", write_filename);
         return -1;
      }


   /* next read in the version number this template converts to */
   the_token = get_token(read_file, '\n');
   if (the_token->token_type != T_CARROT)
   {
      printf("No version detected.\n");
      ask_version(&the_version);
      if (!strcasecmp(the_version.str_show(), "none"))
         return 0;

      if ((write_file = fopen(write_filename, "w")) == NULL)
      {
         printf("Could not open write file '%s'\n", write_filename);
         return -1;
      }

      return execute_convert(the_version.str_show(), read_file, write_file,
                                                               filename);
   }
   the_token = get_token(read_file, '^');
   the_version = the_token->the_string;

   if (the_version.str_show() == NULL)
   {
      printf("No version was detected.\n");
      ask_version(&the_version);
   }
   else if (!verify_version(the_version.str_show()))
   {
      the_version = format;
      if (!strcasecmp(the_version.str_show(), FULLVERSION))
      {
         printf("\nVersion already current.\n");
         return 0;
      }

      printf("The detected version '%s' is not supported.\n", 
                                                 the_version.str_show());
      ask_version(&the_version);
   }

   printf("Version detected: %s\n", the_version.str_show());

   if ((write_file = fopen(write_filename, "w")) == NULL)
   {
      printf("Could not open write file '%s'\n", write_filename);
      return -1;
   }

   return execute_convert(the_version.str_show(), read_file, write_file,
                                                                 filename);
}

#endif

/***********************************************************************
 ** Generator (constructor) - initializes this generator with the write
 **                           file
 **
 ** Parameters: write_file - the file to write the results to
 **
 ***********************************************************************/

Generator::Generator(FILE *write_to, int is_long_header)
{
	write_file = write_to;
	long_header = is_long_header;
}

/***********************************************************************
 ** generate_element - reads in an element from the pseudocode file and
 **                    writes it back as AIME data
 **
 ** Parameters: the_file - the file we read from
 **			    write_file - the file to write to
 **
 ** Returns: -1 for could not find element header, 1 for success
 **
 ***********************************************************************/

int Generator::generate_element(FILE *the_file)
{
	token_record *the_token;

	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
		return 0;

	if (!strcmp(the_token->the_string, "StartHeader"))
	{
		conv_Header *tmp_header;

		if ((tmp_header = read_header(the_file)) == NULL)
			return -1;
		write_header(tmp_header);
		delete tmp_header;
	}
	else if (!strcmp(the_token->the_string, "StartLocation"))
	{
		conv_Location *tmp_loc;

		if ((tmp_loc = read_location(the_file)) == NULL)
			return -1;
		write_location(tmp_loc);
		delete tmp_loc->tmp_mudobj;
		delete tmp_loc;
	}
	else if (!strcmp(the_token->the_string, "StartMobile"))
	{
		conv_Mobile *tmp_mob;

		if ((tmp_mob = read_mobile(the_file)) == NULL)
			return -1;
		write_mobile(tmp_mob);
		delete_mobile(tmp_mob);
	}
	else if (!strcmp(the_token->the_string, "StartWeapon"))
	{
		conv_Weapon *tmp_weapon;

		if ((tmp_weapon = read_weapon(the_file)) == NULL)
			return -1;
		write_weapon(tmp_weapon);
		delete_moveable(tmp_weapon->the_move);
		delete tmp_weapon;
	}
	else if (!strcmp(the_token->the_string, "StartMarker"))
	{
		conv_Item *tmp_marker;

		if ((tmp_marker = read_marker(the_file)) == NULL)
			return -1;
		write_marker(tmp_marker);
		delete tmp_marker->the_mudobj;
		delete tmp_marker;
	}
	else if (!strcmp(the_token->the_string, "StartWearable"))
	{
		conv_Wearable *tmp_wearable;

		if ((tmp_wearable = read_wearable(the_file)) == NULL)
			return -1;
		write_wearable(tmp_wearable);
		delete_moveable(tmp_wearable->the_move);
		delete tmp_wearable;
	}
	else if (!strcmp(the_token->the_string, "StartMoveable"))
	{
		conv_Moveable *tmp_moveable;

		if ((tmp_moveable = read_moveable(the_file)) == NULL)
			return -1;
		write_moveable(tmp_moveable);
		delete_moveable(tmp_moveable);
	}
	else if (!strcmp(the_token->the_string, "StartDoor"))
	{
		conv_Door *tmp_door;

		if ((tmp_door = read_door(the_file)) == NULL)
			return -1;
		write_door(tmp_door);
		delete tmp_door->tmp_item->the_mudobj;
		delete tmp_door->tmp_item;
		delete tmp_door;
	}
	else if (!strcmp(the_token->the_string, "StartBook"))
	{
		conv_Book *tmp_book;

		if ((tmp_book = read_book(the_file)) == NULL)
			return -1;
		write_book(tmp_book);
		delete_moveable(tmp_book->tmp_move);
		delete tmp_book;
	}
	else if (!strcmp(the_token->the_string, "StartBoat"))
	{
		conv_Boat *tmp_boat;

		if ((tmp_boat = read_boat(the_file)) == NULL)
			return -1;
		write_boat(tmp_boat);
		delete_moveable(tmp_boat->tmp_move);
		delete tmp_boat;
	}
	else if (!strcmp(the_token->the_string, "StartRope"))
	{
		conv_Rope *tmp_rope;

		if ((tmp_rope = read_rope(the_file)) == NULL)
			return -1;
		write_rope(tmp_rope);
		delete_moveable(tmp_rope->tmp_move);
		delete tmp_rope;
	}
	else if (!strcmp(the_token->the_string, "StartMerger"))
	{
		conv_Merger *tmp_merger;

		if ((tmp_merger = read_merger(the_file)) == NULL)
			return -1;
		write_merger(tmp_merger);
		delete_moveable(tmp_merger->the_move);
		delete tmp_merger;
	}
	else if (!strcmp(the_token->the_string, "StartFood"))
	{
		conv_Food *tmp_food;

		if ((tmp_food = read_food(the_file)) == NULL)
			return -1;
		write_food(tmp_food);
		delete_moveable(tmp_food->the_merger->the_move);
		delete tmp_food->the_merger;
		delete tmp_food;
	}
	else if (!strcmp(the_token->the_string, "StartMoney"))
	{
		conv_Money *tmp_money;

		if ((tmp_money = read_money(the_file)) == NULL)
			return -1;
		write_money(tmp_money);
		delete_moveable(tmp_money->the_merger->the_move);
		delete tmp_money->the_merger;
		delete tmp_money;
	}
	else if (!strcmp(the_token->the_string, "StartSpecial"))
	{
		conv_Specials *tmp_special;

		if ((tmp_special = read_specials(the_file)) == NULL)
			return -1;
		write_special(tmp_special);
		delete tmp_special;
	}
	else if (!strcmp(the_token->the_string, "StartSpell"))
	{
		conv_Spell *tmp_spell;

		if ((tmp_spell = read_spell(the_file)) == NULL)
			return -1;
		write_spell(tmp_spell);
		delete tmp_spell;
	}
	else if (!strcmp(the_token->the_string, "StartSkill"))
	{
		conv_Skill *tmp_skill;

		if ((tmp_skill = read_skill(the_file)) == NULL)
			return -1;
		write_skill(tmp_skill);
		delete tmp_skill;
	}
	else if (!strcmp(the_token->the_string, "StartAction"))
	{
		conv_Action *tmp_action;

		if ((tmp_action = read_action(the_file)) == NULL)
			return -1;
		write_action(tmp_action);
		delete tmp_action;
	}
	else if (!strcmp(the_token->the_string, "StartRace"))
	{
		conv_Race *tmp_race;

		if ((tmp_race = read_race(the_file)) == NULL)
			return -1;
		write_race(tmp_race);
		delete tmp_race;
	}
	else if (!strcmp(the_token->the_string, "StartQuest"))
	{
		conv_Quest *tmp_quest;

		if ((tmp_quest = read_quest(the_file)) == NULL)
			return -1;
		write_quest(tmp_quest);
		delete tmp_quest;
	}
	else if (!strcmp(the_token->the_string, "StartLevel"))
	{
		conv_Level *tmp_level;

		if ((tmp_level = read_level(the_file)) == NULL)
			return -1;
		write_levels(tmp_level);
		delete tmp_level;
	}
	else if (!strcmp(the_token->the_string, "StartMask"))
	{
		conv_Mask *tmp_mask;

		if ((tmp_mask = read_mask(the_file)) == NULL)
			return -1;
		write_mask(tmp_mask);
		delete tmp_mask;
	}
	else if (!strcmp(the_token->the_string, "StartInclination"))
	{
		conv_Inclination *tmp_incl;

		if ((tmp_incl = read_inclination(the_file)) == NULL)
			return -1;
		write_inclination(tmp_incl);
		delete tmp_incl;
	}
	else if (!strcmp(the_token->the_string, "StartTalent"))
	{
		conv_Talent *tmp_talent;

		if ((tmp_talent = read_talent(the_file)) == NULL)
			return -1;
		write_talent(tmp_talent);
		delete tmp_talent;
	}
	else if (!strcmp(the_token->the_string, "StartChatLine"))
	{
		conv_ChatLine *tmp_chatline;

		if ((tmp_chatline = read_chatline(the_file)) == NULL)
			return -1;
		write_chatline(tmp_chatline);
		delete tmp_chatline;
	}

	else
	{
	   // This just burns us to the end of elements we don't want to read
	   while ((the_token->token_type > 0) && (strncmp(the_token->the_string, "End", 3)))
	  	  the_token = get_token(the_file, '\0');
	}
	return 1;
}


/***********************************************************************
 ** read_sl_string - reads in a single line string and passes the info
 **                  into the passed in the_string object
 **
 ** Parameters: the_file - the file we read from
 **             the_string - the string object to place the results into
 **
 ** Returns: -1 for a problem in the string, 1 for succcess
 **
 ***********************************************************************/

int Generator::read_sl_string(FILE *the_file, Strings *the_string)
{
	token_record *the_token;

	the_token = get_token(the_file, '\0');
	if ((the_token->token_type <= 0) || (the_token->token_type != T_CARROT))
		return -1;

	the_token = get_token(the_file, '^');
	if (the_token->token_type <= 0)
		return -1;

	the_string->str_copy(the_token->the_string);
	return 1;
}


/***********************************************************************
 ** read_int - reads in an integer value
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the integer, value for success
 **
 ***********************************************************************/

int Generator::read_int(FILE *the_file)
{
	token_record *the_token;
	int sign = 1;

	the_token = get_token(the_file, '\0');

	if (the_token->token_type == T_MINUS)
	{
		sign = -1;
		the_token = get_token(the_file, '\0');
	}

	if ((the_token->token_type <= 0) || (the_token->token_type != T_NUMERICAL))
		return -1;

	return (atoi(the_token->the_string) * sign);
}

/***********************************************************************
 ** read_long - reads in a long value
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the long, value for success
 **
 ***********************************************************************/

long Generator::read_long(FILE *the_file)
{
	token_record *the_token;
	long sign = 1;

	the_token = get_token(the_file, '\0');

    if (the_token->token_type == T_MINUS)
    {
	  sign = -1;
	  the_token = get_token(the_file, '\0');
    }

	if ((the_token->token_type <= 0) || (the_token->token_type != T_NUMERICAL))
		return -1;

	return (atol(the_token->the_string) * sign);
}


/***********************************************************************
 ** check_mudobj_fields - checks all fields that may be in a mudobject
 **
 ** Parameters: the_file - the file we read from
 **             the_token - the token that has the field name
 **             tmp_mudobj - a pointer to the mudobject convert struct
 **
 ** Returns: -1 for a problem in the object, 1 for succcess, 0 for no fields
 **          found
 **
 ***********************************************************************/

int Generator::check_mudobj_fields(FILE *the_file, token_record *the_token, 
													conv_MudObject *tmp_mudobj)
{
	Strings holder;

	if (!STRCASECMP(the_token->the_string, "Title"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->title = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Name"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->name = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Altname"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->altname = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Clones"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->clones = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Location"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->location = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SpecialStr"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->special_str = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Desc"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_mudobj->desc = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Capacity"))
	{
		if ((tmp_mudobj->capacity = read_int(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}

	else
		return 0;
}


/***********************************************************************
 ** check_ability_fields - checks all fields that may be in an ability
 **
 ** Parameters: the_file - the file we read from
 **             the_token - the token that has the field name
 **             tmp_mudobj - a pointer to the mudobject convert struct
 **
 ** Returns: -1 for a problem in the object, 1 for succcess, 0 for no fields
 **          found
 **
 ***********************************************************************/

int Generator::check_ability_fields(FILE *the_file, token_record *the_token, 
													conv_Ability *tmp_ability)
{
	Strings holder;

	if (!STRCASECMP(the_token->the_string, "Name"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->name = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "ActingActor"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->acting_str[0] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "ActingTarget"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->acting_str[1] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "ActingRoom"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->acting_str[2] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SuccessActor"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->success_str[0] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SuccessTarget"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->success_str[1] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SuccessRoom"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->success_str[2] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "FailureActor"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->fail_str[0] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "FailureTarget"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->fail_str[1] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "FailureRoom"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->fail_str[2] = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SpecialName"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->special_name = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "SuccessTrigger"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->succ_trig = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "FailTrigger"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->fail_trig = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "AttemptTrigger"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_ability->attempt_trig = holder.str_show();		
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Drain"))
	{
		if ((tmp_ability->drain = read_int(the_file)) <= 0)
		{
			return -1;
		}
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "StartDependencyList"))
	{
		Strings holder;

		while (STRCASECMP(the_token->the_string, "EndDependencyList"))
		{
			if (!STRCASECMP(the_token->the_string, "StartDependency"))
			{
				conv_dependency *new_dep = new conv_dependency;
				conv_dependency *tmp_dep;

				new_dep->the_type = (depend_type) 0;
				new_dep->num_value = 0;
				new_dep->str_value = "";
				new_dep->next_depend = NULL;

				while (STRCASECMP(the_token->the_string, "EndDependency"))
				{
					if (!STRCASECMP(the_token->the_string, "DependencyType"))
					{
						if ((new_dep->the_type = (depend_type) read_int(the_file)) < 0)
						{
							delete new_dep;
							return -1;
						}
					}
					else if (!STRCASECMP(the_token->the_string, "DependencyInt"))
					{
						if ((new_dep->num_value = read_int(the_file)) < 0)
						{
							delete new_dep;
							return -1;
						}
					}
					else if (!STRCASECMP(the_token->the_string, "DependencyString"))
					{
						if (read_sl_string(the_file, &holder) <= 0)
						{
							delete new_dep;
							return -1;
						}
						new_dep->str_value = holder.str_show();
					}
					the_token = get_token(the_file, '\0');
				}

				// Add our new one to the list
				tmp_dep = tmp_ability->depend_list;
				if (tmp_dep == NULL)
					tmp_ability->depend_list = new_dep;
				else
				{
					while (tmp_dep->next_depend != NULL)
						tmp_dep = tmp_dep->next_depend;
					tmp_dep->next_depend = new_dep;
				}
			}
			the_token = get_token(the_file, '\0');
		}
		return 1;
	}
	else
		return 0;

}


/***********************************************************************
 ** check_merger_fields - checks all fields that may be in a merger
 **
 ** Parameters: the_file - the file we read from
 **             the_token - the token that has the field name
 **             tmp_merger - a pointer to the merger convert struct
 **
 ** Returns: -1 for a problem in the object, 1 for succcess, 0 for no fields
 **          found
 **
 ***********************************************************************/

int Generator::check_merger_fields(FILE *the_file, token_record *the_token, 
													conv_Merger *tmp_merger)
{
	Strings holder;

	if (!STRCASECMP(the_token->the_string, "NumberOf"))
	{
		if ((tmp_merger->number_of = read_int(the_file)) < 0)
		{
			return -1;
		}
		return 1;
	}
	else
		return 0;
}

/***********************************************************************
 ** check_item_fields - checks all fields that may be in an item
 **
 ** Parameters: the_file - the file we read from
 **             the_token - the token that has the field name
 **             tmp_item - a pointer to the item convert struct
 **
 ** Returns: -1 for a problem in the object, 1 for succcess, 0 for no fields
 **          found
 **
 ***********************************************************************/

int Generator::check_item_fields(FILE *the_file, token_record *the_token, 
													conv_Item *tmp_item)
{
	Strings holder;

	if (!STRCASECMP(the_token->the_string, "Guard"))
	{
		if (read_sl_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_item->guard = holder.str_show();
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "ItemFlag1"))
	{
		if ((tmp_item->flag[0] = read_long(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "ItemFlag2"))
	{
		if ((tmp_item->flag[1] = read_long(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else
		return 0;
}


/***********************************************************************
 ** check_moveable_fields - checks all fields that may be in a moveable item
 **
 ** Parameters: the_file - the file we read from
 **             the_token - the token that has the field name
 **             tmp_move - a pointer to the moveable convert struct
 **
 ** Returns: -1 for a problem in the object, 1 for succcess, 0 for no fields
 **          found
 **
 ***********************************************************************/

int Generator::check_moveable_fields(FILE *the_file, token_record *the_token, 
														conv_Moveable *tmp_move)
{
	Strings holder;

	if (!STRCASECMP(the_token->the_string, "Brief1"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_move->brief[0] = holder.str_show();
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Brief2"))
	{
		if (read_ml_string(the_file, &holder) <= 0)
		{
			return -1;
		}
		tmp_move->brief[1] = holder.str_show();
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Size"))
	{
		if ((tmp_move->size = read_int(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "Weight"))
	{
		if ((tmp_move->weight = read_int(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "NumAllowed"))
	{
		if ((tmp_move->num_allowed = read_int(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else if (!STRCASECMP(the_token->the_string, "PercentAllowed"))
	{
		if ((tmp_move->percent_allowed = read_int(the_file)) == -1)
		{
			return -1;
		}
		return 1;
	}
	else
		return 0;
}


/***********************************************************************
 ** read_ml_string - reads in a multi-line string
 **
 ** Parameters: the_file - the file we read from
 **             the_string - the string to place the results into
 **
 ** Returns: -1 for a problem in the object, 1 for success
 **
 ***********************************************************************/

int Generator::read_ml_string(FILE *the_file, Strings *the_string)
{
	char *tmp_char;

	tmp_char = read_desc_type(the_file, NULL, NULL);
	if (tmp_char == NULL)
		return -1;
	the_string->str_copy(tmp_char);
	delete tmp_char;

	return 1;
}


/***********************************************************************
 ** read_header - reads in a header element that contains things such as
 **               read permissions, etc
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

conv_Header *Generator::read_header(FILE *the_file)
{
	token_record *the_token;
	conv_Header  *tmp_header;
	Strings      holder;

	tmp_header = new conv_Header;

	// First we initialize the header with default data
	tmp_header->write_allow = "";
	tmp_header->read_deny = "";
	tmp_header->flag1 = 0;
	tmp_header->flag2 = 0;
	tmp_header->reload_secs = 1800;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete tmp_header;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndHeader")))
	{
		if (!strcmp(the_token->the_string, "ReadDeny"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete tmp_header;
				return NULL;
			}
			tmp_header->read_deny = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "WriteAllow"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete tmp_header;
				return NULL;
			}
			tmp_header->write_allow = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Flag1"))
		{
			if ((tmp_header->flag1 = read_long(the_file)) == -1)
			{
				delete tmp_header;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Flag2"))
		{
			if ((tmp_header->flag2 = read_long(the_file)) == -1)
			{
				delete tmp_header;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "ReloadSecs"))
		{
			if ((tmp_header->reload_secs = read_long(the_file)) == -1)
			{
				delete tmp_header;
				return NULL;
			}
		}
		else
		{
			// We need to add code here to ignore unknown entries instead of doing this
			delete tmp_header;
			return NULL;
		}
		the_token = get_token(the_file, '\0');
	}
	return tmp_header;
}


/***********************************************************************
 ** write_header - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             conv_Header - the header struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_header(conv_Header *the_header)
{
	fprintf(write_file, "^%s^\n", FULLVERSION);
	fprintf(write_file, "^%s^\n", the_header->read_deny.str_show());
	fprintf(write_file, "^%s^\n", the_header->write_allow.str_show());
	if (long_header)
	{
		fprintf(write_file, "%d\n", the_header->flag1);
		fprintf(write_file, "%d\n", the_header->flag2);
		fprintf(write_file, "%d\n", the_header->reload_secs);
	}
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** read_location - reads in a location object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Location *Generator::read_location(FILE *the_file)
{
	conv_Location *the_loc;
	token_record *the_token;
	int found = 0;
	Strings holder;
	int i;

	the_loc = new conv_Location;

	for (i=0; i<10; i++)
		the_loc->exit[i] = "";

	the_loc->extra_desc[0] = "";
	the_loc->extra_desc[1] = "";
	the_loc->extra_dir = "";
	the_loc->listen = "";
	the_loc->smell = "";
	the_loc->room_size = 30;
	the_loc->lighting = 5;
	the_loc->terrain = 5;
	the_loc->flag[0] = 0;
	the_loc->flag[1] = 0;
	the_loc->tmp_mudobj = new conv_MudObject;

	the_loc->tmp_mudobj->altname = "";
	the_loc->tmp_mudobj->title = "";
	the_loc->tmp_mudobj->capacity = 0;
	the_loc->tmp_mudobj->clones = "";
	the_loc->tmp_mudobj->desc = "No Desc.\n";
	the_loc->tmp_mudobj->location = "NoLoc";
	the_loc->tmp_mudobj->special_str = "";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_loc->tmp_mudobj;
		delete the_loc;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndLocation")))
	{
		for (i=0; i<10; i++)
		{
			if (!STRCASECMP(the_token->the_string, dir[i]))
			{
				if (read_sl_string(the_file, &holder) <= 0)
				{
					delete the_loc->tmp_mudobj;
					delete the_loc;
					return NULL;
				}
				the_loc->exit[i] = holder.str_show();
				found = 1;
				break;
			}
		}
		if ((found) || (check_mudobj_fields(the_file, the_token, the_loc->tmp_mudobj) > 0))
		{
			the_token = get_token(the_file, '\0');
			found = 0;
			continue;
		}

		else if (!strcmp(the_token->the_string, "Listen"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
			the_loc->listen = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Smell"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
			the_loc->smell = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "ExtraDir"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
			the_loc->extra_dir = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "ExtraDesc1"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
			the_loc->extra_desc[0] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "ExtraDesc2"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
			the_loc->extra_desc[1] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "HorizSize"))
		{
			if ((the_loc->room_size = read_int(the_file)) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "VertSize"))
		{
			if (read_int(the_file) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Height"))
		{
			if (read_int(the_file) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}

		else if (!strcmp(the_token->the_string, "Terrain"))
		{
			if ((the_loc->terrain = read_int(the_file)) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Lighting"))
		{
			if ((the_loc->lighting = read_int(the_file)) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "LocFlag1"))
		{
			if ((the_loc->flag[0] = read_long(the_file)) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "LocFlag2"))
		{
			if ((the_loc->flag[1] = read_long(the_file)) == -1)
			{
				delete the_loc->tmp_mudobj;
				delete the_loc;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_loc;
}




/***********************************************************************
 ** delete_mobile - deletes the passed in mobile and all attached to it
 ***********************************************************************/

void Generator::delete_mobile(conv_Mobile *the_mob)
{
	conv_Com_List *tmp_com;
	conv_shop_item *tmp_item;

	delete the_mob->the_mudobj;

	while (the_mob->com_list != NULL)
	{
		tmp_com = the_mob->com_list->next_com;
		delete the_mob->com_list;
		the_mob->com_list = tmp_com;
	}

	if (the_mob->the_shop != NULL)
	{
	
		while (the_mob->the_shop->item_list != NULL)
		{
			tmp_item = the_mob->the_shop->item_list->next_item;
			delete the_mob->the_shop->item_list;
			the_mob->the_shop->item_list = tmp_item;
		}

		delete the_mob->the_shop;
	}

	delete the_mob;
}


/***********************************************************************
 ** delete_race - deletes the passed in race and all attached to it
 ***********************************************************************/

void Generator::delete_race(conv_Race *the_race)
{
	conv_tutorial *tmp_tut = the_race->tutorial_list;

	while (the_race->tutorial_list != NULL)
	{
		tmp_tut = the_race->tutorial_list->next_tut;
		delete the_race->tutorial_list;
		the_race->tutorial_list = tmp_tut;
	}

	delete the_race;
}


/***********************************************************************
 ** delete_moveable - deletes a moveable struct
 ***********************************************************************/

void Generator::delete_moveable(conv_Moveable *the_move)
{
	delete the_move->the_item->the_mudobj;
	delete the_move->the_item;
	delete the_move;
}

/***********************************************************************
 ** delete_ability - deletes ability fields
 ***********************************************************************/

void Generator::delete_ability(conv_Ability *the_ability)
{
	conv_dependency *tmp_dep;

	while (the_ability->depend_list != NULL)
	{
		tmp_dep = the_ability->depend_list;
		the_ability->depend_list = tmp_dep->next_depend;
		delete tmp_dep;
	}

	delete the_ability;
}


/***********************************************************************
 ** init_mudobject - initializes a mudobject with standard init settings
 ***********************************************************************/

void Generator::init_mudobject(conv_MudObject *the_mudobj)
{
	the_mudobj->altname = "";
	the_mudobj->title = "";
	the_mudobj->capacity = 0;
	the_mudobj->clones = "";
	the_mudobj->desc = "No Desc.\n";
	the_mudobj->location = "NoLoc";
	the_mudobj->special_str = "";
}

/***********************************************************************
 ** init_ability - initializes an ability with standard init settings
 ***********************************************************************/

void Generator::init_ability(conv_Ability *the_ability)
{
  int i;
  for (i=0; i<3; i++)
     the_ability->acting_str[i] = "";

  for (i=0; i<3; i++)
     the_ability->success_str[i] = "";

  for (i=0; i<3; i++)
     the_ability->fail_str[i] = "";

  the_ability->attempt_trig = "";
  the_ability->fail_trig = "";
  the_ability->succ_trig = "";
  the_ability->drain = 0;
  the_ability->special_name = "";
  the_ability->depend_list = NULL;
}


/***********************************************************************
 ** init_item - initializes an item with standard init settings
 ***********************************************************************/

void Generator::init_item(conv_Item *the_item)
{
	the_item->flag[0] = 0;
	the_item->flag[1] = 0;
	the_item->guard = "";
}


/***********************************************************************
 ** init_merger - initializes a merger with standard init settings
 ***********************************************************************/

void Generator::init_merger(conv_Merger *the_merger)
{
	the_merger->number_of = 0;
}


/***********************************************************************
 ** init_moveable - initializes a moveable with standard init settings
 ***********************************************************************/

void Generator::init_moveable(conv_Moveable *the_move)
{
	the_move->brief[0] = "No Brief";
	the_move->brief[1] = "No Brief";
	the_move->num_allowed = 9999;
	the_move->percent_allowed = 0;
	the_move->size = 0;
	the_move->weight = 0;
}

/***********************************************************************
 ** ignore_entry - skips to the end of an entry after detecting the type
 ***********************************************************************/

void Generator::ignore_entry(FILE *the_file)
{
	token_record *the_token = get_token(the_file, '\0');

	// If it is a number, our job is finished
	if (the_token->token_type == T_NUMERICAL)
		return;

	// If it is a string, skip to the end
	if (the_token->token_type == T_CARROT)
	{
		the_token = get_token(the_file, '^');
	}

}

/***********************************************************************
 ** read_mobile - reads in a mobile object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Mobile * Generator::read_mobile(FILE *the_file)
{
	conv_Mobile *the_mob;
	token_record *the_token;
	Strings holder;

	the_mob = new conv_Mobile;
	the_mob->the_mudobj = new conv_MudObject;

	init_mudobject(the_mob->the_mudobj);

	the_mob->brief = "None";
	the_mob->subarea = "";
	the_mob->start_wield[0] = "";
	the_mob->start_wield[1] = "";
	the_mob->start_wear = "";
	the_mob->bare_weapon = "";
	the_mob->strength = 5;
	the_mob->dexterity = 5;
	the_mob->intel = 5;
	the_mob->flag[0] = the_mob->flag[1] = 0;
	the_mob->speed = 0;
	the_mob->aggression = 0;
	the_mob->wisdom = 0;
	the_mob->charisma = 0;
	the_mob->com_list = NULL;
	the_mob->the_shop = NULL;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_mobile(the_mob);
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMobile")))
	{
		if (check_mudobj_fields(the_file, the_token, the_mob->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "Brief"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->brief = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "BareWeapon"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->bare_weapon = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "StartWear"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->start_wear = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "StartWieldLeft"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->start_wield[0] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "StartWieldRight"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->start_wield[1] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Subarea"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
			the_mob->subarea = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Aggression"))
		{
			if ((the_mob->aggression = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Constitution"))
		{
			if ((the_mob->constitution = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Dexterity"))
		{
			if ((the_mob->dexterity = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MobFlag1"))
		{
			if ((the_mob->flag[0] = read_long(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MobFlag2"))
		{
			if ((the_mob->flag[1] = read_long(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Intel"))
		{
			if ((the_mob->intel = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Wisdom"))
		{
			if ((the_mob->wisdom = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Charisma"))
		{
			if ((the_mob->charisma = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Speed"))
		{
			if ((the_mob->speed = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Strength"))
		{
			if ((the_mob->strength = read_int(the_file)) < 0)
			{
				delete_mobile(the_mob);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "StartTellReplies"))
		{
			conv_Com_List *tmp_comm;

			while (strcmp(the_token->the_string, "EndTellReplies"))
			{
				the_token = get_token(the_file, '\0');
				if (!strcmp(the_token->the_string, "StartReply"))
				{
					if (the_mob->com_list == NULL)
					{
						the_mob->com_list = new conv_Com_List;
						tmp_comm = the_mob->com_list;
					}
					else
					{
						tmp_comm = the_mob->com_list;
						while (tmp_comm->next_com != NULL)
							tmp_comm = tmp_comm->next_com;
						tmp_comm->next_com = new conv_Com_List;
						tmp_comm = tmp_comm->next_com;
					}
					tmp_comm->next_com = NULL;

					the_token = get_token(the_file, '\0');
					while (strcmp(the_token->the_string, "EndReply"))
					{
						if (!strcmp(the_token->the_string, "Keywords"))
						{
							if (read_sl_string(the_file, &holder) <= 0)
							{
								delete_mobile(the_mob);
								return NULL;
							}
							tmp_comm->the_keywords = holder.str_show();
						}
						else if (!strcmp(the_token->the_string, "Reply"))
						{
							if (read_ml_string(the_file, &holder) <= 0)
							{
								delete_mobile(the_mob);
								return NULL;
							}
							tmp_comm->the_reply = holder.str_show();
						}
						else
						{
							ignore_entry(the_file);
						}
						the_token = get_token(the_file, '\0');
					}
				}
			}
		}

		else if (!strcmp(the_token->the_string, "StartShop"))
		{
			conv_shop_item *tmp_item;

			the_mob->the_shop = new conv_Shop;
			the_mob->the_shop->item_list = NULL;

			while (strcmp(the_token->the_string, "EndShop"))
			{
				the_token = get_token(the_file, '\0');
				if (!strcmp(the_token->the_string, "Name"))
				{
					if (read_sl_string(the_file, &holder) <= 0)
					{
						delete_mobile(the_mob);
						return NULL;
					}
					the_mob->the_shop->name = holder.str_show();
				}
				else if (!strcmp(the_token->the_string, "Currency"))
				{
					if (read_sl_string(the_file, &holder) <= 0)
					{
						delete_mobile(the_mob);
						return NULL;
					}
					the_mob->the_shop->currency = holder.str_show();
				}
				else if (!strcmp(the_token->the_string, "StartShopItems"))
				{
					while (strcmp(the_token->the_string, "EndShopItems"))
					{
						the_token = get_token(the_file, '\0');
						if (!strcmp(the_token->the_string, "StartItem"))
						{
							if (the_mob->the_shop->item_list == NULL)
							{
								the_mob->the_shop->item_list = new conv_shop_item;
								tmp_item = the_mob->the_shop->item_list;
							}
							else
							{
								tmp_item = the_mob->the_shop->item_list;
								while (tmp_item->next_item != NULL)
										tmp_item = tmp_item->next_item;
								tmp_item->next_item = new conv_shop_item;
								tmp_item = tmp_item->next_item;
							}
							tmp_item->next_item = NULL;

							the_token = get_token(the_file, '\0');
							while (strcmp(the_token->the_string, "EndItem"))
							{
								if (!strcmp(the_token->the_string, "ItemName"))	
								{
									if (read_sl_string(the_file, &holder) <= 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
									tmp_item->itemname = holder.str_show();
								}
								else if (!strcmp(the_token->the_string, "AliasName"))	
								{
									if (read_sl_string(the_file, &holder) <= 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
									tmp_item->alias = holder.str_show();
								}
								else if (!strcmp(the_token->the_string, "Desc"))	
								{
									if (read_ml_string(the_file, &holder) <= 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
									tmp_item->desc = holder.str_show();
								}
								else if (!strcmp(the_token->the_string, "Value"))
								{
									if ((tmp_item->value = read_int(the_file)) < 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
								}
								else if (!strcmp(the_token->the_string, "ItemType"))
								{
									if ((tmp_item->item_type = read_int(the_file)) < 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
								}
								else if (!strcmp(the_token->the_string, "NumberOf"))
								{
									if ((tmp_item->number_of = read_int(the_file)) < 0)
									{
										delete_mobile(the_mob);
										return NULL;
									}
								}

								else
								{
									ignore_entry(the_file);
								}
								the_token = get_token(the_file, '\0');
							}
						}
					}
				}
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_mob;

}


/***********************************************************************
 ** read_door - reads in a door object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Door * Generator::read_door(FILE *the_file)
{
	conv_Door *the_door;
	token_record *the_token;
	Strings holder;
	int i;

	the_door = new conv_Door;
	the_door->tmp_item = new conv_Item;
	the_door->tmp_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_door->tmp_item->the_mudobj);
	init_item(the_door->tmp_item);

	the_door->distance = 0;
	the_door->door_state = 0;
	for (i=0; i<4; i++)
		the_door->inside_brief[i] = "No Brief";
	for (i=0; i<4; i++)
		the_door->outside_brief[i] = "No Brief";
	the_door->outside_desc = "No Desc";
	the_door->key_names = "";
	the_door->outside_loc = "noloc";
	the_door->rope_name = "";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_door->tmp_item->the_mudobj;
		delete the_door->tmp_item;
		delete the_door;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndDoor")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_door->tmp_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (check_item_fields(the_file, the_token, 
			                                the_door->tmp_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (!strcmp(the_token->the_string, "KeyNames"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->key_names = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Brief0"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[0] = holder.str_show();
			the_door->outside_brief[0] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Brief1"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[1] = holder.str_show();
			the_door->outside_brief[1] = holder.str_show();
		}

		else if (!strcmp(the_token->the_string, "InsideBrief0"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[0] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InsideBrief1"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[1] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InsideBrief2"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[2] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InsideBrief3"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->inside_brief[3] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideBrief0"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_brief[0] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideBrief1"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_brief[1] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideBrief2"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_brief[2] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideBrief3"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_brief[3] = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideLoc"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_loc = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "OutsideDesc"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->outside_desc = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "RopeName"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
			the_door->rope_name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "DoorState"))
		{
			if ((the_door->door_state = read_int(the_file)) < 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Distance"))
		{
			if ((the_door->distance = read_int(the_file)) < 0)
			{
				delete the_door->tmp_item->the_mudobj;
				delete the_door->tmp_item;
				delete the_door;
				return NULL;
			}
		}

		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_door;
}


/***********************************************************************
 ** read_book - reads in a book object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Book * Generator::read_book(FILE *the_file)
{
	conv_Book *the_book;
	token_record *the_token;
	Strings holder;

	the_book = new conv_Book;
	the_book->tmp_move = new conv_Moveable;
	the_book->tmp_move->the_item = new conv_Item;
	the_book->tmp_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_book->tmp_move->the_item->the_mudobj);
	init_item(the_book->tmp_move->the_item);
	init_moveable(the_book->tmp_move);

	the_book->subject = "";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_book->tmp_move);
		delete the_book;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndBook")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_book->tmp_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_book->tmp_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_book->tmp_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "Subject"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_moveable(the_book->tmp_move);
				delete the_book;
				return NULL;
			}
			the_book->subject = holder.str_show();
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_book;

}


/***********************************************************************
 ** read_boat - reads in a boat object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Boat * Generator::read_boat(FILE *the_file)
{
	conv_Boat *the_boat;
	token_record *the_token;
	Strings holder;

	the_boat = new conv_Boat;
	the_boat->tmp_move = new conv_Moveable;
	the_boat->tmp_move->the_item = new conv_Item;
	the_boat->tmp_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_boat->tmp_move->the_item->the_mudobj);
	init_item(the_boat->tmp_move->the_item);
	init_moveable(the_boat->tmp_move);

	the_boat->move_str = "No String";
	the_boat->water_brief = "None";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_boat->tmp_move);
		delete the_boat;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndBoat")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_boat->tmp_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_boat->tmp_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_boat->tmp_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "MoveStr"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_moveable(the_boat->tmp_move);
				delete the_boat;
				return NULL;
			}
			the_boat->move_str = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "WaterBrief"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_moveable(the_boat->tmp_move);
				delete the_boat;
				return NULL;
			}
			the_boat->water_brief = holder.str_show();
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_boat;
}


/***********************************************************************
 ** read_rope - reads in a rope object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for success
 **
 ***********************************************************************/

conv_Rope * Generator::read_rope(FILE *the_file)
{
	conv_Rope *the_rope;
	token_record *the_token;
	Strings holder;

	the_rope = new conv_Rope;
	the_rope->tmp_move = new conv_Moveable;
	the_rope->tmp_move->the_item = new conv_Item;
	the_rope->tmp_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_rope->tmp_move->the_item->the_mudobj);
	init_item(the_rope->tmp_move->the_item);
	init_moveable(the_rope->tmp_move);

	the_rope->rope_len = 1;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_rope->tmp_move);
		delete the_rope;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndRope")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_rope->tmp_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_rope->tmp_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_rope->tmp_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "RopeLength"))
		{
			if ((the_rope->rope_len = read_int(the_file)) < 0)
			{
				delete_moveable(the_rope->tmp_move);
				delete the_rope;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_rope;
}


/***********************************************************************
 ** read_weapon - reads in a weapon object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Weapon * Generator::read_weapon(FILE *the_file)
{
	conv_Weapon *the_weapon;
	token_record *the_token;
	Strings holder;

	the_weapon = new conv_Weapon;
	the_weapon->the_move = new conv_Moveable;
	the_weapon->the_move->the_item = new conv_Item;
	the_weapon->the_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_weapon->the_move->the_item->the_mudobj);
	init_item(the_weapon->the_move->the_item);
	init_moveable(the_weapon->the_move);

	the_weapon->damage = 0;
	the_weapon->dep_dex = 0;
	the_weapon->dep_strength = 0;
	the_weapon->weapon_class = "";
	the_weapon->wield_type = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_weapon->the_move);
		delete the_weapon;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndWeapon")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_weapon->the_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_weapon->the_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_weapon->the_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "WeaponClass"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_moveable(the_weapon->the_move);
				delete the_weapon;
				return NULL;
			}
			the_weapon->weapon_class = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "DepStrength"))
		{
			if ((the_weapon->dep_strength = read_int(the_file)) < 0)
			{
				delete_moveable(the_weapon->the_move);
				delete the_weapon;
				return NULL;
			}			
		}
		else if (!strcmp(the_token->the_string, "DepDexterity"))
		{
			if ((the_weapon->dep_dex = read_int(the_file)) < 0)
			{
				delete_moveable(the_weapon->the_move);
				delete the_weapon;
				return NULL;
			}			
		}
		else if (!strcmp(the_token->the_string, "Damage"))
		{
			if ((the_weapon->damage = read_int(the_file)) < 0)
			{
				delete_moveable(the_weapon->the_move);
				delete the_weapon;
				return NULL;
			}			
		}
		else if (!strcmp(the_token->the_string, "WieldType"))
		{
			if ((the_weapon->wield_type = read_int(the_file)) < 0)
			{
				delete_moveable(the_weapon->the_move);
				delete the_weapon;
				return NULL;
			}			
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_weapon;

}


/***********************************************************************
 ** read_marker - reads in a marker object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Item * Generator::read_marker(FILE *the_file)
{
	conv_Item *the_marker;
	token_record *the_token;
	Strings holder;

	the_marker = new conv_Item;
	the_marker->the_mudobj = new conv_MudObject;

	init_mudobject(the_marker->the_mudobj);

	the_marker->flag[0] = 0;
	the_marker->flag[1] = 0;
	the_marker->guard = "";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_marker->the_mudobj;
		delete the_marker;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMarker")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_marker->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (check_item_fields(the_file, the_token, 
			                                the_marker) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_marker;

}


/***********************************************************************
 ** read_merger - reads in a merger object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Merger * Generator::read_merger(FILE *the_file)
{
	conv_Merger *the_merger;
	token_record *the_token;
	Strings holder;

	the_merger = new conv_Merger;
	the_merger->the_move = new conv_Moveable;
	the_merger->the_move->the_item = new conv_Item;
	the_merger->the_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_merger->the_move->the_item->the_mudobj);
	init_item(the_merger->the_move->the_item);
	init_moveable(the_merger->the_move);
	init_merger(the_merger);

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_merger->the_move);
		delete the_merger;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMerger")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_merger->the_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_merger->the_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_merger->the_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (check_merger_fields(the_file, the_token, 
														the_merger) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_merger;
}


/***********************************************************************
 ** read_wearable - reads in a wearable object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Wearable * Generator::read_wearable(FILE *the_file)
{
	conv_Wearable *the_wearable;
	token_record *the_token;
	Strings holder;

	the_wearable = new conv_Wearable;
	the_wearable->the_move = new conv_Moveable;
	the_wearable->the_move->the_item = new conv_Item;
	the_wearable->the_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_wearable->the_move->the_item->the_mudobj);
	init_item(the_wearable->the_move->the_item);
	init_moveable(the_wearable->the_move);

	the_wearable->the_move->brief[0] = "No Brief";
	the_wearable->the_move->brief[1] = "No Brief";
	the_wearable->the_move->num_allowed = 9999;
	the_wearable->the_move->percent_allowed = 0;
	the_wearable->the_move->size = 0;
	the_wearable->the_move->weight = 0;

	the_wearable->armor = 0;
	the_wearable->flag = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_wearable->the_move);
		delete the_wearable;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndWearable")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_wearable->the_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_wearable->the_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_wearable->the_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (!strcmp(the_token->the_string, "Armor"))
		{
			if ((the_wearable->armor = read_int(the_file)) < 0)
			{
				delete_moveable(the_wearable->the_move);
				delete the_wearable;
				return NULL;
			}			
		}
		else if (!strcmp(the_token->the_string, "WearFlag1"))
		{
			if ((the_wearable->flag = read_long(the_file)) < 0)
			{
				delete_moveable(the_wearable->the_move);
				delete the_wearable;
				return NULL;
			}			
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_wearable;
}


/***********************************************************************
 ** read_moveable - reads in a moveable object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Moveable * Generator::read_moveable(FILE *the_file)
{
	conv_Moveable *the_moveable;
	token_record *the_token;
	Strings holder;

	the_moveable = new conv_Moveable;
	the_moveable->the_item = new conv_Item;
	the_moveable->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_moveable->the_item->the_mudobj);
	init_item(the_moveable->the_item);
	init_moveable(the_moveable);

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_moveable);
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMoveable")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_moveable->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_moveable->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_moveable) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_moveable;
}

/***********************************************************************
 ** read_food - reads in a food object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Food * Generator::read_food(FILE *the_file)
{
	conv_Food *the_food;
	token_record *the_token;
	Strings holder;

	the_food = new conv_Food;
	the_food->the_merger = new conv_Merger;
	the_food->the_merger->the_move = new conv_Moveable;
	the_food->the_merger->the_move->the_item = new conv_Item;
	the_food->the_merger->the_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_food->the_merger->the_move->the_item->the_mudobj);
	init_item(the_food->the_merger->the_move->the_item);
	init_moveable(the_food->the_merger->the_move);
	init_merger(the_food->the_merger);

	the_food->energy = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_food->the_merger->the_move);
		delete the_food->the_merger;
		delete the_food;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndFood")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_food->the_merger->the_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_food->the_merger->the_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_food->the_merger->the_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (check_merger_fields(the_file, the_token, 
														the_food->the_merger) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (!strcmp(the_token->the_string, "Energy"))
		{
			if ((the_food->energy = read_int(the_file)) < 0)
			{
				delete_moveable(the_food->the_merger->the_move);
				delete the_food->the_merger;
				delete the_food;
				return NULL;
			}			
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_food;

}


/***********************************************************************
 ** read_money - reads in a money object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Money * Generator::read_money(FILE *the_file)
{
	conv_Money *the_money;
	token_record *the_token;
	Strings holder;

	the_money = new conv_Money;
	the_money->the_merger = new conv_Merger;
	the_money->the_merger->the_move = new conv_Moveable;
	the_money->the_merger->the_move->the_item = new conv_Item;
	the_money->the_merger->the_move->the_item->the_mudobj = new conv_MudObject;

	init_mudobject(the_money->the_merger->the_move->the_item->the_mudobj);
	init_item(the_money->the_merger->the_move->the_item);
	init_moveable(the_money->the_merger->the_move);
	init_merger(the_money->the_merger);

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_moveable(the_money->the_merger->the_move);
		delete the_money->the_merger;
		delete the_money;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMoney")))
	{
		if (check_mudobj_fields(the_file, the_token, 
			                                the_money->the_merger->the_move->the_item->the_mudobj) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_item_fields(the_file, the_token, 
			                                the_money->the_merger->the_move->the_item) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		else if (check_moveable_fields(the_file, the_token, 
			                                the_money->the_merger->the_move) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (check_merger_fields(the_file, the_token, 
														the_money->the_merger) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}
		else if (!strcmp(the_token->the_string, "MoneyIndex"))
		{
			if ((the_money->money_index = read_int(the_file)) < 0)
			{
				delete_moveable(the_money->the_merger->the_move);
				delete the_money->the_merger;
				delete the_money;
				return NULL;
			}			
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_money;

}


/***********************************************************************
 ** read_specials - reads in a specials object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Specials * Generator::read_specials(FILE *the_file)
{
	conv_Specials *the_specials;
	token_record *the_token;
	Strings holder;

	the_specials = new conv_Specials;

	the_specials->activation = 0;
	the_specials->counter = 0;
	the_specials->interval = 0;
	the_specials->num_times = 0;
	the_specials->trigger = "none";
	the_specials->code = "";
	the_specials->comments = "";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_specials;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndSpecial")))
	{
		if (!strcmp(the_token->the_string, "Trigger"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_specials;
				return NULL;
			}
			the_specials->trigger = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_specials;
				return NULL;
			}
			the_specials->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Activation"))
		{
			if ((the_specials->activation = read_long(the_file)) == -1)
			{
				delete the_specials;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Interval"))
		{
			if ((the_specials->interval = read_long(the_file)) == -1)
			{
				delete the_specials;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "NumTimes"))
		{
			if ((the_specials->interval = read_int(the_file)) == -1)
			{
				delete the_specials;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Counter"))
		{
			if ((the_specials->counter = read_long(the_file)) == -1)
			{
				delete the_specials;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Code"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_specials;
				return NULL;
			}
			the_specials->code = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Comments"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_specials;
				return NULL;
			}
			the_specials->comments = holder.str_show();
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}

	return the_specials;
}




/***********************************************************************
 ** read_spell - reads in a spell object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Spell * Generator::read_spell(FILE *the_file)
{
	conv_Spell *the_spell;
	token_record *the_token;
	Strings holder;

	the_spell = new conv_Spell;
	the_spell->the_abil = new conv_Ability;

	init_ability(the_spell->the_abil);
	the_spell->flag = 0;


	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_ability(the_spell->the_abil);
		delete the_spell;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndSpell")))
	{
		if (check_ability_fields(the_file, the_token, the_spell->the_abil) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		if (!strcmp(the_token->the_string, "SpellFlag1"))
		{
			if ((the_spell->flag = read_long(the_file)) == -1)
			{
				delete_ability(the_spell->the_abil);
				delete the_spell;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_spell;
}


/***********************************************************************
 ** read_skill - reads in a skill object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Skill * Generator::read_skill(FILE *the_file)
{
	conv_Skill *the_skill;
	token_record *the_token;
	Strings holder;

	the_skill = new conv_Skill;
	the_skill->the_abil = new conv_Ability;

	init_ability(the_skill->the_abil);
	the_skill->flag = 0;


	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_ability(the_skill->the_abil);
		delete the_skill;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndSkill")))
	{
		if (check_ability_fields(the_file, the_token, the_skill->the_abil) > 0)
		{
			the_token = get_token(the_file, '\0');
			continue;
		}

		if (!strcmp(the_token->the_string, "SkillFlag1"))
		{
			if ((the_skill->flag = read_long(the_file)) == -1)
			{
				delete_ability(the_skill->the_abil);
				delete the_skill;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_skill;

}


/***********************************************************************
 ** read_action - reads in an action object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Action * Generator::read_action(FILE *the_file)
{
	conv_Action *the_action;
	token_record *the_token;
	Strings holder;

	the_action = new conv_Action;

	the_action->flag = 0;


	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_action;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndAction")))
	{

		if (!strcmp(the_token->the_string, "ActFlag1"))
		{
			if ((the_action->flag = read_long(the_file)) == -1)
			{
				delete the_action;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Actor"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->actor = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Crowd"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->crowd = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Target"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->target = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Sender"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->sender = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Bystander"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_action;
				return NULL;
			}
			the_action->bystander = holder.str_show();
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_action;
}


/***********************************************************************
 ** read_race - reads in a race object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Race * Generator::read_race(FILE *the_file)
{
	conv_Race *the_race;
	token_record *the_token;
	Strings holder;

	the_race = new conv_Race;

	the_race->init_dex = 0;
	the_race->init_intel = 0;
	the_race->init_str = 0;
	the_race->death_location = "none";
	the_race->death_text = "You have died.";
	the_race->description = "None";
	the_race->init_brief = "A creature is here.";
	the_race->init_location = "none";
	the_race->tutorial_list = NULL;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete_race(the_race);
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndRace")))
	{

		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Description"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->description = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InitialDesc"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->init_desc = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InitialBrief"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->init_brief = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InitialAbility"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->init_ability = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InitialLocation"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->init_location = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "DeathLocation"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->init_brief = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "DeathText"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->death_text = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "AllowInclinations"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->allow_incl = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "AllowTalents"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete_race(the_race);
				return NULL;
			}
			the_race->allow_talent = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "InitialStrength"))
		{
			if ((the_race->init_str = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "InitialDexterity"))
		{
			if ((the_race->init_dex = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "InitialIntel"))
		{
			if ((the_race->init_intel = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "InitialCon"))
		{
			if ((the_race->init_con = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "InitialWis"))
		{
			if ((the_race->init_wis = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "InitialCha"))
		{
			if ((the_race->init_cha = read_int(the_file)) < 0)
			{
				delete_race(the_race);
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "StartTutorialList"))
		{
			conv_tutorial *tmp_tut;

			while (strcmp(the_token->the_string, "EndTutorialList"))
			{
				the_token = get_token(the_file, '\0');
				if (!strcmp(the_token->the_string, "StartTutorial"))
				{
					if (the_race->tutorial_list == NULL)
					{
						the_race->tutorial_list = new conv_tutorial;
						tmp_tut = the_race->tutorial_list;
					}
					else
					{
						tmp_tut = the_race->tutorial_list;
						while (tmp_tut->next_tut != NULL)
							tmp_tut = tmp_tut->next_tut;
						tmp_tut->next_tut = new conv_tutorial;
						tmp_tut = tmp_tut->next_tut;
					}
					tmp_tut->next_tut = NULL;

					the_token = get_token(the_file, '\0');
					while (strcmp(the_token->the_string, "EndTutorial"))
					{
						if (!strcmp(the_token->the_string, "Name"))
						{
							if (read_sl_string(the_file, &holder) <= 0)
							{
								delete_race(the_race);
								return NULL;
							}
							tmp_tut->name = holder.str_show();
						}
						else if (!strcmp(the_token->the_string, "TutorialSpecial"))
						{
							if (read_sl_string(the_file, &holder) <= 0)
							{
								delete_race(the_race);
								return NULL;
							}
							tmp_tut->tutorial_special = holder.str_show();
						}
						else if (!strcmp(the_token->the_string, "Prompt"))
						{
							if (read_sl_string(the_file, &holder) <= 0)
							{
								delete_race(the_race);
								return NULL;
							}
							tmp_tut->prompt = holder.str_show();
						}
						else if (!strcmp(the_token->the_string, "Sequence"))
						{
							if ((tmp_tut->sequence = read_int(the_file)) < 0)
							{
								delete_race(the_race);
								return NULL;
							}
						}
						else if (!strcmp(the_token->the_string, "Mandatory"))
						{
							if ((tmp_tut->mandatory = read_int(the_file)) < 0)
							{
								delete_race(the_race);
								return NULL;
							}
						}
						else
						{
							ignore_entry(the_file);
						}
						the_token = get_token(the_file, '\0');
					}
				}
			}

		}

		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_race;
}


/***********************************************************************
 ** read_quest - reads in a quest object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Quest * Generator::read_quest(FILE *the_file)
{
	conv_Quest *the_quest;
	token_record *the_token;
	Strings holder;

	the_quest = new conv_Quest;

	the_quest->combat = 0;
	the_quest->difficulty = 0;
	the_quest->puzzles = 0;
	the_quest->qpoints = 0;
	the_quest->required = 0;
	the_quest->description = "None";

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_quest;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndQuest")))
	{

		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_quest;
				return NULL;
			}
			the_quest->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Description"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_quest;
				return NULL;
			}
			the_quest->description = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Difficulty"))
		{
			if ((the_quest->difficulty = read_int(the_file)) < 0)
			{
				delete the_quest;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Puzzles"))
		{
			if ((the_quest->puzzles = read_int(the_file)) < 0)
			{
				delete the_quest;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Combat"))
		{
			if ((the_quest->combat = read_int(the_file)) < 0)
			{
				delete the_quest;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "QuestPoints"))
		{
			if ((the_quest->qpoints = read_int(the_file)) < 0)
			{
				delete the_quest;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "Required"))
		{
			if ((the_quest->required = read_int(the_file)) < 0)
			{
				delete the_quest;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_quest;
}


/***********************************************************************
 ** read_level - reads in a level object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Level * Generator::read_level(FILE *the_file)
{
	conv_Level *the_level;
	token_record *the_token;
	Strings holder;

	the_level = new conv_Level;

	the_level->award_str = "None";
	the_level->chain_name = "none";
	the_level->level_str = "None";
	the_level->lvl_num = 0;
	the_level->min_dex = 0;
	the_level->min_str = 0;
	the_level->min_intel = 0;
	the_level->min_exp = 0;
	the_level->min_con = 0;
	the_level->min_wis = 0;
	the_level->min_cha = 0;
	the_level->flag = 0;


	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_level;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndLevel")))
	{

		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "ChainName"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->chain_name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "RequiredAbilities"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->req_abilities = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "AwardString"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->award_str = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "LevelString"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->level_str = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "SpecialRequirements"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->special_req_str = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "WhenAwarded"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_level;
				return NULL;
			}
			the_level->when_awarded_str = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "LevelNumber"))
		{
			if ((the_level->lvl_num = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinStrength"))
		{
			if ((the_level->min_str = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinIntel"))
		{
			if ((the_level->min_intel = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinDexterity"))
		{
			if ((the_level->min_dex = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinExperience"))
		{
			if ((the_level->min_exp = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinConstitution"))
		{
			if ((the_level->min_con = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinWisdom"))
		{
			if ((the_level->min_wis = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "MinCharisma"))
		{
			if ((the_level->min_cha = read_int(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else if (!strcmp(the_token->the_string, "LevelFlag1"))
		{
			if ((the_level->flag = read_long(the_file)) < 0)
			{
				delete the_level;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_level;
}


/***********************************************************************
 ** read_chatline - reads in a chatline object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_ChatLine * Generator::read_chatline(FILE *the_file)
{
   conv_ChatLine *the_chatline;
   token_record *the_token;
   Strings holder;

   the_chatline = new conv_ChatLine;

   the_chatline->lead_in_actor = "";
   the_chatline->lead_in_crowd = "";
   the_chatline->text_color = "";
   the_chatline->border = "";
   the_chatline->flag1 = 0;

   // Now read in the pseudo-data and place into the struct
    the_token = get_token(the_file, '\0');
    if (the_token->token_type <= 0)
    {
       delete the_chatline;
       return NULL;
    }

    while ((the_token->token_type > 0) && 
           (strcmp(the_token->the_string, "EndChatLine")))
    {
       if (!strcmp(the_token->the_string, "Name"))
       {
	  if (read_sl_string(the_file, &holder) <= 0)
	  {
	    delete the_chatline;
	    return NULL;
	  }
	  the_chatline->name = holder.str_show();
       }
       else if (!strcmp(the_token->the_string, "LeadInActor"))
       {
	 if (read_sl_string(the_file, &holder) <= 0)
	 {
	   delete the_chatline;
	   return NULL;
	 }
	 the_chatline->lead_in_actor = holder.str_show();
       }
       else if (!strcmp(the_token->the_string, "LeadInCrowd"))
       {
	 if (read_sl_string(the_file, &holder) <= 0)
	 {
	   delete the_chatline;
	   return NULL;
	 }
	 the_chatline->lead_in_crowd = holder.str_show();
       }
       else if (!strcmp(the_token->the_string, "TextColor"))
       {
	 if (read_sl_string(the_file, &holder) <= 0)
	 {
	   delete the_chatline;
	   return NULL;
	 }
	 the_chatline->text_color = holder.str_show();
       }
       else if (!strcmp(the_token->the_string, "Border"))
       {
	 if (read_sl_string(the_file, &holder) <= 0)
	 {
	   delete the_chatline;
	   return NULL;
	 }
	 the_chatline->border = holder.str_show();
       }
       else if (!strcmp(the_token->the_string, "ChatFlag1"))
       {
	 if ((the_chatline->flag1 = read_long(the_file)) < 0)
	   {
	     delete the_chatline;
	     return NULL;
	   }
       }
       else
       {
	 ignore_entry(the_file);
       }
       the_token = get_token(the_file, '\0');
    }
    return the_chatline;
}


/***********************************************************************
 ** read_mask - reads in a mask object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Mask * Generator::read_mask(FILE *the_file)
{
	conv_Mask *the_mask;
	token_record *the_token;
	Strings holder;
	int i;

	the_mask = new conv_Mask;

	for (i=0; i<6; i++)
		the_mask->comflag[i] = 0;
	for (i=0; i<6; i++)
		the_mask->adminflag[i] = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_mask;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndMask")))
	{
		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_mask;
				return NULL;
			}
			the_mask->name = holder.str_show();
		}
		else if (!strncmp(the_token->the_string, "ComFlag", 7))
		{
			int flagnum;

			int result = atoi(the_token->the_string + 7);
			if (((flagnum = atoi(((the_token->the_string) + 7))) < 1) ||
				(flagnum > 6))
			{
				ignore_entry(the_file);

			}
			the_mask->comflag[flagnum-1] = read_long(the_file);
		}
		else if (!strncmp(the_token->the_string, "AdminFlag", 9))
		{
			int flagnum;

			if (((flagnum = atoi((the_token->the_string + 9))) < 1) ||
				 (flagnum > 6))
			{
				ignore_entry(the_file);

			}
			the_mask->adminflag[flagnum-1] = read_long(the_file);
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_mask;
}


/***********************************************************************
 ** read_inclination - reads in an inclination object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Inclination * Generator::read_inclination(FILE *the_file)
{
	conv_Inclination *the_incl;
	token_record *the_token;
	Strings holder;

	the_incl = new conv_Inclination;

	the_incl->desc = "None";
	the_incl->special = "none";
	the_incl->con_offset = 0;
	the_incl->dex_offset = 0;
	the_incl->intel_offset = 0;
	the_incl->str_offset = 0;
	the_incl->wisdom_offset = 0;
	the_incl->charisma_offset = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_incl;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndInclination")))
	{

		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_incl;
				return NULL;
			}
			the_incl->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Description"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_incl;
				return NULL;
			}
			the_incl->desc = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Special"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_incl;
				return NULL;
			}
			the_incl->special = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "StrengthOffset"))
		{
			the_incl->str_offset = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "DexterityOffset"))
		{
			the_incl->dex_offset = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "IntelOffset"))
		{
			the_incl->intel_offset = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "ConstitutionOffset"))
		{
			the_incl->con_offset = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "WisdomOffset"))
		{
			the_incl->wisdom_offset = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "CharismaOffset"))
		{
			the_incl->charisma_offset = read_int(the_file);
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_incl;
}


/***********************************************************************
 ** read_talent - reads in an talent object
 **
 ** Parameters: the_file - the file we read from
 **
 ** Returns: -1 for a problem in the object, 1 for succcess
 **
 ***********************************************************************/

conv_Talent * Generator::read_talent(FILE *the_file)
{
	conv_Talent *the_talent;
	token_record *the_token;
	Strings holder;

	the_talent = new conv_Talent;

	the_talent->desc = "None";
	the_talent->special = "none";
	the_talent->cost = 0;
	the_talent->allowed = 0;

	// Now read in the pseudo-data and place into the struct
	the_token = get_token(the_file, '\0');
	if (the_token->token_type <= 0)
	{
		delete the_talent;
		return NULL;
	}

	while ((the_token->token_type > 0) && (strcmp(the_token->the_string, "EndTalent")))
	{

		if (!strcmp(the_token->the_string, "Name"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_talent;
				return NULL;
			}
			the_talent->name = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Description"))
		{
			if (read_ml_string(the_file, &holder) <= 0)
			{
				delete the_talent;
				return NULL;
			}
			the_talent->desc = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Special"))
		{
			if (read_sl_string(the_file, &holder) <= 0)
			{
				delete the_talent;
				return NULL;
			}
			the_talent->special = holder.str_show();
		}
		else if (!strcmp(the_token->the_string, "Cost"))
		{
			the_talent->cost = read_int(the_file);
		}
		else if (!strcmp(the_token->the_string, "Allowed"))
		{
			if ((the_talent->allowed = read_int(the_file)) < 0)
			{
				delete the_talent;
				return NULL;
			}
		}
		else
		{
			ignore_entry(the_file);
		}
		the_token = get_token(the_file, '\0');
	}
	return the_talent;
}


/***********************************************************************
 ** write_location - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_loc - the location struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_location(conv_Location *the_loc)
{
  int i;
	write_mudobj_fields("loc", the_loc->tmp_mudobj);

    for (i=0; i<10; i++)
      fprintf(write_file, "%s ", 
                ((the_loc->exit[i].str_show() == NULL) ? "0" : the_loc->exit[i].str_show())); 

	fprintf(write_file, "\n^%s^\n", (the_loc->extra_dir.str_show() == NULL) ? "" : 
												the_loc->extra_dir.str_show());
	fprintf(write_file, "^%s^\n", (the_loc->listen.str_show() == NULL) ? "" : 
												the_loc->listen.str_show());
	fprintf(write_file, "^%s^\n", (the_loc->smell.str_show() == NULL) ? "" : 
												the_loc->smell.str_show());
	fprintf(write_file, "^%s^\n", (the_loc->extra_desc[0].str_show() == NULL) ? "" : 
												the_loc->extra_desc[0].str_show());
	fprintf(write_file, "^%s^\n", (the_loc->extra_desc[1].str_show() == NULL) ? "" : 
												the_loc->extra_desc[1].str_show());
	fprintf(write_file, "%d\n", the_loc->room_size);
	fprintf(write_file, "%d\n", the_loc->terrain);
	fprintf(write_file, "%d\n", the_loc->lighting);
	fprintf(write_file, "%d\n", the_loc->flag[0]);
	fprintf(write_file, "%d\n", the_loc->flag[1]);
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_specials - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_special - the special struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_special(conv_Specials *the_special)
{
	fprintf(write_file, "special %s\n", the_special->name.str_show());
	fprintf(write_file, "%s\n", (the_special->trigger.str_show() == NULL) ? "none" : 
													the_special->trigger.str_show());

	fprintf(write_file, "%lu\n", the_special->activation);
	fprintf(write_file, "%lu\n", the_special->interval);
	fprintf(write_file, "%d\n", the_special->num_times);
	fprintf(write_file, "%lu\n", the_special->counter);

	fprintf(write_file, "^%s^\n", (the_special->code.str_show() == NULL) ? "" : 
													the_special->code.str_show());
	fprintf(write_file, "^%s^\n", (the_special->comments.str_show() == NULL) ? "" : 
													the_special->comments.str_show());
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_spell - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_spell - the spell struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_spell(conv_Spell *the_spell)
{
	write_ability_fields("spell", the_spell->the_abil);
	fprintf(write_file, "%ld\n", the_spell->flag);
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_skill - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_skill - the skill struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_skill(conv_Skill *the_skill)
{
	write_ability_fields("skill", the_skill->the_abil);
	fprintf(write_file, "%ld\n", the_skill->flag);
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_mudobj_fields - writes the fields for a mudobject
 **
 ** Parameters: the_mudobj - the mudobject struct to write
 **
 ***********************************************************************/

void Generator::write_mudobj_fields(char *objtype, conv_MudObject *the_mudobj)
{
	fprintf(write_file, "%s %s\n", objtype, the_mudobj->name.str_show());
	fprintf(write_file, "^%s^\n", (the_mudobj->title.str_show() == NULL) ? "" : 
														the_mudobj->title.str_show());
	fprintf(write_file, "^%s^\n", (the_mudobj->altname.str_show() == NULL) ? "" : 
														the_mudobj->altname.str_show());
	fprintf(write_file, "^%s^\n", (the_mudobj->clones.str_show() == NULL) ? "" : 
														the_mudobj->clones.str_show());
	fprintf(write_file, "%s\n", (the_mudobj->location.str_show() == NULL) ? "" : 
														the_mudobj->location.str_show());
	fprintf(write_file, "^%s^\n", (the_mudobj->special_str.str_show() == NULL) ? "" : 
														the_mudobj->special_str.str_show());
	fprintf(write_file, "^%s^\n", (the_mudobj->desc.str_show() == NULL) ? "" : 
														the_mudobj->desc.str_show());
    fprintf(write_file, "%d\n", the_mudobj->capacity);
}


/***********************************************************************
 ** write_ability_fields - writes the fields for an ability object
 **
 ** Parameters: the_mudobj - the mudobject struct to write
 **
 ***********************************************************************/

void Generator::write_ability_fields(char *objtype, conv_Ability *the_ability)
{
   conv_dependency *tmp_dep;

   fprintf(write_file, "%s %s\n", objtype, the_ability->name.str_show());

   fprintf(write_file, "^%s^\n", (the_ability->acting_str[0].str_show() == NULL) ? "" : 
											the_ability->acting_str[0].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->acting_str[1].str_show() == NULL) ? "" : 
											the_ability->acting_str[1].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->acting_str[2].str_show() == NULL) ? "" : 
											the_ability->acting_str[2].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->success_str[0].str_show() == NULL) ? "" : 
											the_ability->success_str[0].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->success_str[1].str_show() == NULL) ? "" : 
											the_ability->success_str[1].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->success_str[2].str_show() == NULL) ? "" : 
											the_ability->success_str[2].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->fail_str[0].str_show() == NULL) ? "" : 
											the_ability->fail_str[0].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->fail_str[0].str_show() == NULL) ? "" : 
											the_ability->fail_str[1].str_show());
   fprintf(write_file, "^%s^\n", (the_ability->fail_str[0].str_show() == NULL) ? "" : 
											the_ability->fail_str[2].str_show());
   fprintf(write_file, "^^\n");  // Remove later
   fprintf(write_file, "^%s^\n", (the_ability->special_name.str_show() == NULL) ? "" : 
											the_ability->special_name.str_show());
   fprintf(write_file, "^%s^\n", (the_ability->succ_trig.str_show() == NULL) ? "" : 
											the_ability->succ_trig.str_show());
   fprintf(write_file, "^%s^\n", (the_ability->fail_trig.str_show() == NULL) ? "" : 
											the_ability->fail_trig.str_show());
   fprintf(write_file, "^%s^\n", (the_ability->attempt_trig.str_show() == NULL) ? "" : 
											the_ability->attempt_trig.str_show());
   fprintf(write_file, "%d\n", the_ability->drain);

   tmp_dep = the_ability->depend_list;
   while (tmp_dep != NULL)
   {
		fprintf(write_file, "+\n");
		fprintf(write_file, "%d\n", (int) tmp_dep->the_type);
		fprintf(write_file, "^%s^\n", (tmp_dep->str_value.str_show() == NULL) ? "" : 
												tmp_dep->str_value.str_show());
		fprintf(write_file, "%d\n", tmp_dep->num_value);
		tmp_dep = tmp_dep->next_depend;
   }
   fprintf(write_file, "#\n");
}


/***********************************************************************
 ** write_item_fields - writes the fields for an item
 **
 ** Parameters: the_item - the item struct to write
 **
 ***********************************************************************/

void Generator::write_item_fields(conv_Item *the_item)
{
   fprintf(write_file, "^%s^\n", (the_item->guard.str_show() == NULL) ? "" : 
														the_item->guard.str_show());
   fprintf(write_file, "%d\n", the_item->flag[0]);
   fprintf(write_file, "%d\n", the_item->flag[1]);
}


/***********************************************************************
 ** write_moveable_fields - writes the fields for a moveable object
 **
 ** Parameters: the_moveable - the moveable struct to write
 **
 ***********************************************************************/

void Generator::write_moveable_fields(conv_Moveable *the_move)
{
   fprintf(write_file, "^%s^\n", (the_move->brief[0].str_show() == NULL) ? "" : 
														the_move->brief[0].str_show());
   fprintf(write_file, "^%s^\n", (the_move->brief[1].str_show() == NULL) ? "" : 
														the_move->brief[1].str_show());
   fprintf(write_file, "%d\n", the_move->size);
   fprintf(write_file, "%d\n", the_move->weight);
   fprintf(write_file, "%d\n", the_move->num_allowed);
   fprintf(write_file, "%d\n", the_move->percent_allowed);
}


/***********************************************************************
 ** write_merger_fields - writes the fields for a merger object
 **
 ** Parameters: the_merger - the merger struct to write
 **
 ***********************************************************************/

void Generator::write_merger_fields(conv_Merger *the_merger)
{
   fprintf(write_file, "%d\n", the_merger->number_of);
}


/***********************************************************************
 ** write_weapon - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_mob - the weapon struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_weapon(conv_Weapon *the_weapon)
{
	write_mudobj_fields("weapon", the_weapon->the_move->the_item->the_mudobj);
	write_item_fields(the_weapon->the_move->the_item);
	write_moveable_fields(the_weapon->the_move);

	fprintf(write_file, "%s\n", (the_weapon->weapon_class.str_show() == NULL) ? "none" : 
													the_weapon->weapon_class.str_show());
	fprintf(write_file, "%d\n", the_weapon->damage);
	fprintf(write_file, "%d\n", the_weapon->dep_strength);
	fprintf(write_file, "%d\n", the_weapon->dep_dex);
	fprintf(write_file, "%d\n", the_weapon->wield_type);
    fprintf(write_file, "\n\n");

}


/***********************************************************************
 ** write_wearable - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_wearable - the wearable struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_wearable(conv_Wearable *the_wearable)
{
	write_mudobj_fields("wearable", the_wearable->the_move->the_item->the_mudobj);
	write_item_fields(the_wearable->the_move->the_item);
	write_moveable_fields(the_wearable->the_move);

	fprintf(write_file, "%d\n", the_wearable->armor);
	fprintf(write_file, "%d\n", the_wearable->flag);
    fprintf(write_file, "\n\n");

}



/***********************************************************************
 ** write_moveable - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_wearable - the wearable struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_moveable(conv_Moveable *the_moveable)
{
	write_mudobj_fields("moveable", the_moveable->the_item->the_mudobj);
	write_item_fields(the_moveable->the_item);
	write_moveable_fields(the_moveable);
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_merger - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_merger - the merger struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_merger(conv_Merger *the_merger)
{
	write_mudobj_fields("merger", the_merger->the_move->the_item->the_mudobj);
	write_item_fields(the_merger->the_move->the_item);
	write_moveable_fields(the_merger->the_move);
	write_merger_fields(the_merger);
    fprintf(write_file, "\n\n");
}

/***********************************************************************
 ** write_food - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_food - the food struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_food(conv_Food *the_food)
{
	write_mudobj_fields("food", the_food->the_merger->the_move->the_item->the_mudobj);
	write_item_fields(the_food->the_merger->the_move->the_item);
	write_moveable_fields(the_food->the_merger->the_move);
	write_merger_fields(the_food->the_merger);
	fprintf(write_file, "%d\n", the_food->energy);
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_money - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_money - the money struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_money(conv_Money *the_money)
{
	write_mudobj_fields("money", the_money->the_merger->the_move->the_item->the_mudobj);
	write_item_fields(the_money->the_merger->the_move->the_item);
	write_moveable_fields(the_money->the_merger->the_move);
	write_merger_fields(the_money->the_merger);
	fprintf(write_file, "%d\n", the_money->money_index);
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_book - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_book - the book struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_book(conv_Book *the_book)
{
	write_mudobj_fields("book", the_book->tmp_move->the_item->the_mudobj);
	write_item_fields(the_book->tmp_move->the_item);
	write_moveable_fields(the_book->tmp_move);

	fprintf(write_file, "^%s^\n", (the_book->subject.str_show() == NULL) ? "" : 
											the_book->subject.str_show());
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_boat - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_boat - the boat struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_boat(conv_Boat *the_boat)
{
	write_mudobj_fields("boat", the_boat->tmp_move->the_item->the_mudobj);
	write_item_fields(the_boat->tmp_move->the_item);
	write_moveable_fields(the_boat->tmp_move);

	fprintf(write_file, "^%s^\n", (the_boat->move_str.str_show() == NULL) ? "" : 
											the_boat->move_str.str_show());
	fprintf(write_file, "^%s^\n", (the_boat->water_brief.str_show() == NULL) ? "" : 
											the_boat->water_brief.str_show());
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_rope - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_rope - the rope struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_rope(conv_Rope *the_rope)
{
	write_mudobj_fields("rope", the_rope->tmp_move->the_item->the_mudobj);
	write_item_fields(the_rope->tmp_move->the_item);
	write_moveable_fields(the_rope->tmp_move);

	fprintf(write_file, "%d\n", the_rope->rope_len);
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_marker - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_mob - the weapon struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_marker(conv_Item *the_marker)
{
	write_mudobj_fields("marker", the_marker->the_mudobj);
	write_item_fields(the_marker);
    fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_action - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_action - the action struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_action(conv_Action *the_action)
{
	fprintf(write_file, "%s\n", the_action->name.str_show());
	fprintf(write_file, "^%s^\n", (the_action->actor.str_show() == NULL) ? "" : 
														the_action->actor.str_show());
	fprintf(write_file, "^%s^\n", (the_action->crowd.str_show() == NULL) ? "" : 
														the_action->crowd.str_show());
	fprintf(write_file, "^%s^\n", (the_action->target.str_show() == NULL) ? "" : 
														the_action->target.str_show());
	fprintf(write_file, "^%s^\n", (the_action->sender.str_show() == NULL) ? "" : 
														the_action->sender.str_show());
	fprintf(write_file, "^%s^\n", (the_action->bystander.str_show() == NULL) ? "" : 
														the_action->bystander.str_show());

	fprintf(write_file, "%ld\n", the_action->flag);
	fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_race - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_race - the race struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_race(conv_Race *the_race)
{
   conv_tutorial *tmp_tutorial;

   fprintf(write_file, "race %s\n", the_race->name.str_show());

   fprintf(write_file, "^%s^\n", (the_race->description.str_show() == NULL) ? "" : 
														the_race->description.str_show());
   fprintf(write_file, "^%s^\n", (the_race->init_desc.str_show() == NULL) ? "" : 
														the_race->init_desc.str_show());
   fprintf(write_file, "^%s^\n", (the_race->init_brief.str_show() == NULL) ? "" : 
														the_race->init_brief.str_show());
   fprintf(write_file, "^%s^\n", (the_race->init_ability.str_show() == NULL) ? "" : 
														the_race->init_ability.str_show());
   fprintf(write_file, "%s\n", (the_race->init_location.str_show() == NULL) ? "none" : 
														the_race->init_location.str_show());
   fprintf(write_file, "%s\n", (the_race->death_location.str_show() == NULL) ? "none" : 
														the_race->death_location.str_show());
   fprintf(write_file, "^%s^\n", (the_race->death_text.str_show() == NULL) ? "" : 
														the_race->death_text.str_show());
   fprintf(write_file, "^%s^\n", (the_race->allow_incl.str_show() == NULL) ? "" : 
	                                        the_race->allow_incl.str_show());
   fprintf(write_file, "^%s^\n", (the_race->allow_talent.str_show() == NULL) ? "" : 
	                                        the_race->allow_talent.str_show());
   fprintf(write_file, "%d\n", the_race->init_str);
   fprintf(write_file, "%d\n", the_race->init_dex);
   fprintf(write_file, "%d\n", the_race->init_intel);
   fprintf(write_file, "%d\n", the_race->init_con);
   fprintf(write_file, "%d\n", the_race->init_wis);
   fprintf(write_file, "%d\n", the_race->init_cha);


   tmp_tutorial = the_race->tutorial_list;

   while (tmp_tutorial != NULL)
   {
      fprintf(write_file, "+\n%s\n", tmp_tutorial->name.str_show());
      fprintf(write_file, "%d\n", tmp_tutorial->sequence);
      fprintf(write_file, "^%s^\n", (tmp_tutorial->prompt.str_show() == NULL) ? 
                                       "" : tmp_tutorial->prompt.str_show());
      fprintf(write_file, "%s\n", 
             (tmp_tutorial->tutorial_special.str_show() == NULL) ? 
                          "none" : tmp_tutorial->tutorial_special.str_show());
      fprintf(write_file, "%d\n", tmp_tutorial->mandatory);
      tmp_tutorial = tmp_tutorial->next_tut;
   }
   fprintf(write_file, "#\n\n");
}


/***********************************************************************
 ** write_quest - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_quest - the quest struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_quest(conv_Quest *the_quest)
{
   fprintf(write_file, "%s\n", the_quest->name.str_show());
	
   fprintf(write_file, "^%s^\n", (the_quest->description.str_show() == NULL) ? "" : 
														the_quest->description.str_show());
   fprintf(write_file, "%d\n", the_quest->difficulty);
   fprintf(write_file, "%d\n", the_quest->puzzles);
   fprintf(write_file, "%d\n", the_quest->combat);
   fprintf(write_file, "%d\n", the_quest->qpoints);
   fprintf(write_file, "%d\n", the_quest->required);

   fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_levels - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_level - the levels struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_levels(conv_Level *the_level)
{
   fprintf(write_file, "level %s\n", the_level->name.str_show());

   fprintf(write_file, "^%s^\n", (the_level->chain_name.str_show() == NULL) ? "" : 
														the_level->chain_name.str_show());
   fprintf(write_file, "^%s^\n", (the_level->award_str.str_show() == NULL) ? "" : 
														the_level->award_str.str_show());
   fprintf(write_file, "^%s^\n", (the_level->level_str.str_show() == NULL) ? "" : 
														the_level->level_str.str_show());
   fprintf(write_file, "%d\n", the_level->lvl_num);
   fprintf(write_file, "%d\n", the_level->min_str);
   fprintf(write_file, "%d\n", the_level->min_dex);
   fprintf(write_file, "%d\n", the_level->min_intel);
   fprintf(write_file, "%d\n", the_level->min_exp);
   fprintf(write_file, "%d\n", the_level->min_con);
   fprintf(write_file, "%d\n", the_level->min_wis);
   fprintf(write_file, "%d\n", the_level->min_cha);
   fprintf(write_file, "^%s^\n", (the_level->req_abilities.str_show() == NULL) ? "" : 
														the_level->req_abilities.str_show());
   fprintf(write_file, "^%s^\n", (the_level->special_req_str.str_show() == NULL) ? "" : 
														the_level->special_req_str.str_show());
   fprintf(write_file, "^%s^\n", (the_level->when_awarded_str.str_show() == NULL) ? "" : 
														the_level->when_awarded_str.str_show());
   fprintf(write_file, "%ld\n", the_level->flag);

   fprintf(write_file, "\n\n");


}


/***********************************************************************
 ** write_chatline - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_chatline - the chatline we are writing
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_chatline(conv_ChatLine *the_chatline)
{
   fprintf(write_file, "chatline %s\n", the_chatline->name.str_show());

   fprintf(write_file, "^%s^\n", (the_chatline->lead_in_actor.str_show() == NULL) 
	   ? "" : the_chatline->lead_in_actor.str_show());
   fprintf(write_file, "^%s^\n", (the_chatline->lead_in_crowd.str_show() == NULL) 
	   ? "" : the_chatline->lead_in_crowd.str_show());
   fprintf(write_file, "^%s^\n", (the_chatline->text_color.str_show() == NULL) 
	   ? "" : the_chatline->text_color.str_show());
   fprintf(write_file, "^%s^\n", (the_chatline->border.str_show() == NULL) 
	   ? "" : the_chatline->border.str_show());
   fprintf(write_file, "%ld\n", the_chatline->flag1);
   fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_mask - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_mask - the mask struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_mask(conv_Mask *the_mask)
{
  int i;
   fprintf(write_file, "%s\n", the_mask->name.str_show());

   for (i=0; i<6; i++)
      fprintf(write_file, "%ld\n", the_mask->comflag[i]);
   for (i=0; i<6; i++)
      fprintf(write_file, "%ld\n", the_mask->adminflag[i]);

   fprintf(write_file, "\n\n");

}


/***********************************************************************
 ** write_inclination - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_incl - the inclination struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_inclination(conv_Inclination *the_incl)
{
   fprintf(write_file, "incl %s\n", the_incl->name.str_show());

   fprintf(write_file, "^%s^\n", (the_incl->desc.str_show() == NULL) ? "" : 
														the_incl->desc.str_show());
   fprintf(write_file, "%s\n", (the_incl->special.str_show() == NULL) ? "none" : 
														the_incl->special.str_show());
   fprintf(write_file, "%d\n", the_incl->str_offset);
   fprintf(write_file, "%d\n", the_incl->dex_offset);
   fprintf(write_file, "%d\n", the_incl->con_offset);
   fprintf(write_file, "%d\n", the_incl->intel_offset);
   fprintf(write_file, "%d\n", the_incl->wisdom_offset);
   fprintf(write_file, "%d\n", the_incl->charisma_offset);

   fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_talent - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_talent - the talent struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_talent(conv_Talent *the_talent)
{
   fprintf(write_file, "talent %s\n", the_talent->name.str_show());

   fprintf(write_file, "^%s^\n", (the_talent->desc.str_show() == NULL) ? "" : 
														the_talent->desc.str_show());
   fprintf(write_file, "%s\n", (the_talent->special.str_show() == NULL) ? "none" : 
																the_talent->special.str_show());
   fprintf(write_file, "%d\n", the_talent->cost);
   fprintf(write_file, "%d\n", the_talent->allowed);

   fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_mobile - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_mob - the mobile struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_mobile(conv_Mobile *the_mob)
{
	conv_Com_List *tmp_list;
	conv_shop_item *tmp_item;

	write_mudobj_fields("mob", the_mob->the_mudobj);

	fprintf(write_file, "^%s^\n", (the_mob->brief.str_show() == NULL) ? "" : 
															the_mob->brief.str_show());
	fprintf(write_file, "^%s^\n", (the_mob->subarea.str_show() == NULL) ? "" : 
															the_mob->subarea.str_show());
	fprintf(write_file, "%s\n", (the_mob->start_wield[0].str_show() == NULL) ? "none" : 
															the_mob->start_wield[0].str_show());
	fprintf(write_file, "%s\n", (the_mob->start_wield[1].str_show() == NULL) ? "none" : 
															the_mob->start_wield[1].str_show());
	fprintf(write_file, "^%s^\n", (the_mob->start_wear.str_show() == NULL) ? "" : 
															the_mob->start_wear.str_show());
	fprintf(write_file, "^%s^\n", (the_mob->bare_weapon.str_show() == NULL) ? "" : 
															the_mob->bare_weapon.str_show());
	fprintf(write_file, "%d\n", the_mob->strength);
	fprintf(write_file, "%d\n", the_mob->dexterity);
	fprintf(write_file, "%d\n", the_mob->constitution);
	fprintf(write_file, "%d\n", the_mob->intel);
	fprintf(write_file, "%d\n", the_mob->wisdom);
	fprintf(write_file, "%d\n", the_mob->charisma);
	
	fprintf(write_file, "%ld\n", the_mob->flag[0]);
	fprintf(write_file, "%ld\n", the_mob->flag[1]);

	fprintf(write_file, "%d\n", the_mob->speed);
	fprintf(write_file, "%d\n", the_mob->aggression);

   tmp_list = the_mob->com_list;
   while (tmp_list != NULL)
   {
      fprintf(write_file, "+\n^%s^\n^%s^\n", tmp_list->the_keywords.str_show(), 
                                           tmp_list->the_reply.str_show());
      tmp_list = tmp_list->next_com;
   }
   fprintf(write_file, "#\n\n");

   if (the_mob->the_shop != NULL)
   {

      fprintf(write_file, "^%s^\n", (the_mob->the_shop->name.str_show() == NULL) ? 
		                                       "NoName" : the_mob->the_shop->name.str_show());
      fprintf(write_file, "%s\n", (the_mob->the_shop->currency.str_show() == NULL) ? 
		                                       "None" : the_mob->the_shop->currency.str_show());
      tmp_item = the_mob->the_shop->item_list;
      while (tmp_item != NULL)
	  {
	     fprintf(write_file, "+\n%s\n", (tmp_item->itemname.str_show() == NULL) ? 
                                "None" : tmp_item->itemname.str_show());
		 fprintf(write_file, "%s\n", (tmp_item->alias.str_show() == NULL) ? 
                                "None" : tmp_item->alias.str_show());
		 fprintf(write_file, "%d\n", tmp_item->value);
		 fprintf(write_file, "%d\n", (int) tmp_item->item_type);
		 fprintf(write_file, "%d\n", tmp_item->number_of);
		 fprintf(write_file, "^\n%s\n^\n", 
                          (tmp_item->desc.str_show() == NULL) ? 
                          "No Desc" : tmp_item->desc.str_show());
         tmp_item = tmp_item->next_item;
	  }
   }

   fprintf(write_file, "#\n");

   fprintf(write_file, "\n\n");
}


/***********************************************************************
 ** write_door - writes a struct to file based on a passed in conv struct
 **
 ** Parameters: write_file - the file we read from
 **             the_door - the door struct to write
 **
 ** Returns: -1 for a problem in the element, 1 for succcess
 **
 ***********************************************************************/

void Generator::write_door(conv_Door *the_door)
{
  int i;
   write_mudobj_fields("door", the_door->tmp_item->the_mudobj);
   write_item_fields(the_door->tmp_item);

   fprintf(write_file, "^%s^\n", (the_door->key_names.str_show() == NULL) ? "" : 
                                                          the_door->key_names.str_show());
   fprintf(write_file, "%d\n", the_door->door_state);

   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      fprintf(write_file, "^%s^\n", (the_door->inside_brief[i].str_show() == NULL) ? "" : 
                                                        the_door->inside_brief[i].str_show());
   }
   fprintf(write_file, "%s\n", (the_door->outside_loc.str_show() == NULL) ? "needloc" : 
                                                       the_door->outside_loc.str_show());
   fprintf(write_file, "^%s^\n", (the_door->outside_desc.str_show() == NULL) ? "" : 
                                                          the_door->outside_desc.str_show());

   for (i=0; i<=DOOR_MAXSTATE; i++)
   {
      fprintf(write_file, "^%s^\n", (the_door->outside_brief[i].str_show() == NULL) ? "" : 
                                                        the_door->outside_brief[i].str_show());
   }

   fprintf(write_file, "%d\n", the_door->distance);
   fprintf(write_file, "%s\n", (the_door->rope_name.str_show() == NULL) ? "none" : 
                                                       the_door->rope_name.str_show());
   fprintf(write_file, "\n\n");
}



#endif
