(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*    Pierre Weis and Jun Furuse, projet Cristal, INRIA Rocquencourt   *)
(*                                                                     *)
(*  Copyright 2001 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  All rights reserved.  This file is distributed    *)
(*  under the terms of the GNU Library General Public License.         *)
(*                                                                     *)
(***********************************************************************)

(* $Id: graphicsY11.mli,v 1.14 2002/02/13 17:15:38 weis Exp $ *)

(* Module [GraphicsX11]: additional graphics primitives for the X Windows system *)

type window_id = string;;

val get_window_id : unit -> window_id;;
        (* Return the unique identifier of the Caml graphics window.
           The returned string is an unsigned 32 bits integer 
           in decimal form. *)

val open_subwindow : x:int -> y:int -> width:int -> height:int -> window_id;;
         (* Create a sub-window of the current Caml graphics window
            and return its identifier. *)

val close_subwindow : window_id -> unit;;
         (* Close the sub-window having the given identifier. *)

val map_subwindow : window_id -> unit;;
val unmap_subwindow : window_id -> unit;;
         (* Map or unmap the sub-window having the given identifier. *)

val move_subwindow : window_id -> int -> int -> unit;;
         (* [move_subwindow wid x y] moves the sub-window having the
            given identifier to point [x, y]. *)
val resize_subwindow : window_id -> int -> int -> unit;;
         (* [resize_subwindow wid w h] resizes the sub-window having
            the given identifier to height [h] and width [w]. *)

external bstore_id : unit -> int32 = "gr_bstore";;
        (* return the X pixmap of the bstore window as an integer *)
external window_id : unit -> int32 = "gr_window";;
        (* return the X pixmap of the bstore window as an integer *)


external flush : unit -> unit = "gr_flush";;
        (* flush pending events *)

external sync : unit -> unit = "gr_sync";;
        (* flush pending events and wait until all have been processed *)

external bsize_x : unit -> int = "gr_bsize_x";;
external bsize_y : unit -> int = "gr_bsize_y";;
        (* Idem, but return the size of the backing store. *)
external screen_x : unit -> int = "gr_screen_x";;
external screen_y : unit -> int = "gr_screen_y";;
external origin_x : unit -> int = "gr_origin_x";;
external origin_y : unit -> int = "gr_origin_y";;
        (* Return the size of the screen. *)
external reposition : int -> int -> int -> int -> unit = "gr_reposition";;

external set_named_atom_property : string -> string -> unit
    = "gr_set_named_atom_property";;
        (* make_atom_property ATOM STRING define an X atom ATOM with
           property STRING *)

(* setting cursor *)
(* check by fd -fn cursor *)
type cursor = 
  | Cursor_id of int
  | Cursor_X_cursor
  | Cursor_arrow
  | Cursor_based_arrow_down
  | Cursor_based_arrow_up
  | Cursor_boat
  | Cursor_bogosity
  | Cursor_bottom_left_corner
  | Cursor_bottom_right_corner
  | Cursor_bottom_side
  | Cursor_bottom_tee
  | Cursor_box_spiral
  | Cursor_center_ptr
  | Cursor_circle
  | Cursor_clock
  | Cursor_coffee_mug
  | Cursor_cross
  | Cursor_cross_reverse
  | Cursor_crosshair
  | Cursor_diamond_cross
  | Cursor_dot
  | Cursor_dotbox
  | Cursor_double_arrow
  | Cursor_draft_large
  | Cursor_draft_small
  | Cursor_draped_box
  | Cursor_exchange
  | Cursor_fleur
  | Cursor_gobbler
  | Cursor_gumby
  | Cursor_hand1
  | Cursor_hand2
  | Cursor_heart
  | Cursor_icon
  | Cursor_iron_cross
  | Cursor_left_ptr
  | Cursor_left_side
  | Cursor_left_tee
  | Cursor_leftbutton
  | Cursor_ll_angle
  | Cursor_lr_angle
  | Cursor_man
  | Cursor_middlebutton
  | Cursor_mouse
  | Cursor_pencil
  | Cursor_pirate
  | Cursor_plus
  | Cursor_question_arrow
  | Cursor_right_ptr
  | Cursor_right_side
  | Cursor_right_tee
  | Cursor_rightbutton
  | Cursor_rtl_logo
  | Cursor_sailboat
  | Cursor_sb_down_arrow
  | Cursor_sb_h_double_arrow
  | Cursor_sb_left_arrow
  | Cursor_sb_right_arrow
  | Cursor_sb_up_arrow
  | Cursor_sb_v_double_arrow
  | Cursor_shuttle
  | Cursor_sizing
  | Cursor_spider
  | Cursor_spraycan
  | Cursor_star
  | Cursor_target
  | Cursor_tcross
  | Cursor_top_left_arrow
  | Cursor_top_left_corner
  | Cursor_top_right_corner
  | Cursor_top_side
  | Cursor_top_tee
  | Cursor_trek
  | Cursor_ul_angle
  | Cursor_umbrella
  | Cursor_ur_angle
  | Cursor_watch
  | Cursor_xterm
;;

val set_cursor : cursor -> unit;;
        (* set the cursor for the graphics window *)

val unset_cursor : unit -> unit;;
        (* unset the cursor (use the parent's cursor) *)

external get_geometry : unit -> int * int * int * int = "gr_get_geometry";;
        (* returns width, height, x, y of the graphics window *)

external get_modifiers : unit -> int = "gr_get_modifiers";;
        (* returns the list of modifiers as an integer *)

val button1 : int;;
val button2 : int;;
val button3 : int;;
val button4 : int;;
val button5 : int;;
val shift : int;;
val control : int;;
val mod1 : int;;
val mod2 : int;;
val mod3 : int;;
val mod4 : int;;
val mod5 : int;;
        (* mask for modifiers *)

external cut : string -> unit = "gr_cut";;
        (* paste string to the cut and paste buffer *)

val draw_area : 
    ima:Graphics.image -> srcx:int -> srcy:int -> width:int -> height:int ->
      destx:int -> desty:int -> unit;;


(* Redefinition of events *)

(*** Mouse and keyboard events *)

type status =
  { mouse_x : int;              (* X coordinate of the mouse *)
    mouse_y : int;              (* Y coordinate of the mouse *)
    button : bool;              (* true if a mouse button is pressed *)
    keypressed : bool;          (* true if a key has been pressed *)
    key : char ;                (* the character for the key pressed *)
    modifiers : int;
  };;
(* To report events. *)

type event =
    Button_down                 (* A mouse button is pressed *)
  | Button_up                   (* A mouse button is released *)
  | Key_pressed                 (* A key is pressed *)
  | Mouse_motion                (* The mouse is moved *)
  | Poll                        (* Don't wait; return immediately *)
        (* To specify events to wait for. *)
;;

external wait_next_event : event list -> status = "gry_wait_event";;
        (* Wait until one of the events specified in the given event list
           occurs, and return the status of the mouse and keyboard at
           that time. If [Poll] is given in the event list, return immediately
           with the current status. If the mouse cursor is outside of the
           graphics window, the [mouse_x] and [mouse_y] fields of the event are
           outside the range [0..size_x()-1, 0..size_y()-1]. Keypresses
           are queued, and dequeued one by one when the [Key_pressed]
           event is specified. *)
val retrieve_events : unit -> unit;;
        (* Instead of having the event retrieving periodically called by
	   the interval timer like Graphics, we have this manual event 
	   retrieving function. Call [initalize] first to disable 
	   the original Graphics's interval timer call. *)

(*** Mouse and keyboard polling *)

val mouse_pos : unit -> int * int;;
        (* Return the position of the mouse cursor, relative to the
           graphics window. If the mouse cursor is outside of the graphics
           window, [mouse_pos()] returns a point outside of the range
           [0..size_x()-1, 0..size_y()-1]. *)
val button_down : unit -> bool;;
        (* Return [true] if the mouse button is pressed, [false] otherwise. *)
val read_key : unit -> char;;
        (* Wait for a key to be pressed, and return the corresponding
           character. Keypresses are queued. *)
val key_pressed : unit -> bool;;
        (* Return [true] if a keypress is available; that is, if [read_key]
           would not block. *)

(** It enables/disables the command display_mode and synchronize according
 to the value [false]/[true]. When disable, [display_mode] and [synchronize]
 commands will be ignored. *)
val global_display_mode : bool -> unit;;
(** Same as [Graphics.synchronize] but according to [global_display_mode] *)
val synchronize : unit -> unit;;
(** Same as [Graphics.display_mode] but according to [global_display_mode] *)
val display_mode : bool -> unit;;
(** As [point_color] but according to values of [global_display_mode] *)
val point_color : int -> int -> int;;

val init : unit -> unit;;
(** We have to call this function to disable the original Graphics
   event retrieving facility. *)
