/*
   Project: Adun

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#define _GNU_SOURCE
#include <fenv.h>
#include "AdunKernel/AdunCore.h"
#include "AdunKernel/AdunController.h"

static id appCore = nil;
NSString* divider = @"-------------------------------------------------------------------------------\n";

@interface AdCore (AdCheckpointing)
/**
Sets up checkpointing related variables, timers etc.
*/
- (void) setupCheckpointing;
/**
Checkpoints the current energy
*/
- (void) checkpointEnergy;
/**
Checkpoints the current dynamic state.
*/
- (void) checkpointTrajectory;
/**
Opens a frame for the current configuration iteration if
one is not already open. Closes any previous frame.
*/
- (void) openFrame;
/**
Called when a system reloads its data so it can
be checkpointed.
*/
- (void) handleSystemContentsChange: (NSNotification*) aNotification;
/**
Returns a dictionary containing data on the production step and frame correpsonding
to the last trajectory checkpoint made upto and including step number \e step. The dictionary keys are
"Step", "Frame" and "Checkpoint". The method uses \e dataReader to access the trajectory data.
*/
- (NSDictionary*) dataForLastTrajectoryCheckpoint: (int) step dataReader: (AdSimulationData*) dataReader;
@end

@implementation AdCore

- (void) _createMinimiser
{
	minimiser = [[AdMinimiser alloc]
			initWithSystems: [configurationGenerator systems]
			forceFields: [configurationGenerator forceFields]
			absoluteTolerance: 0.01
			numberOfSteps: 15
			algorithm: @"BFGS"
			stepSize: 0.1
			tolerance: 0.1];
}

- (void) _minimise
{
	AdForceFieldCollection* forceFieldCollection;
	NSEnumerator *forceFieldEnum;
	id forceField;

	forceFieldCollection = [configurationGenerator forceFields];

	GSPrintf(stdout, @"%@", divider);
	GSPrintf(stdout, @"Initial energies:\n");
	[forceFieldCollection evaluateEnergies];
	forceFieldEnum = [[forceFieldCollection forceFields] objectEnumerator];
	while((forceField = [forceFieldEnum nextObject]))
		if([forceFieldCollection isActive: forceField])
			GSPrintf(stdout, @"System %@. Energy %lf\n",
				[[forceField system] systemName],
				[forceField totalEnergy]);
	
	fflush(stdout);
	[minimiser production: NULL];

	GSPrintf(stdout, @"\nFinal energies:\n");
	[forceFieldCollection evaluateEnergies];
	forceFieldEnum = [[forceFieldCollection forceFields] objectEnumerator];
	while((forceField = [forceFieldEnum nextObject]))
		if([forceFieldCollection isActive: forceField])
			GSPrintf(stdout, @"System %@. Energy %lf\n",
				[[forceField system] systemName],
				[forceField totalEnergy]);

	//Reset all the timers that were incremented
	//during the minmisation. 
	[[AdMainLoopTimer mainLoopTimer] resetAll];	
	GSPrintf(stdout, @"%@", divider);		
}

//Prints out information on the current simulator configuration
- (void) _printSummary
{
	NSEnumerator* enumerator;
	id object;

	GSPrintf(stdout, @"%@", divider);
	GSPrintf(stdout, @"\nDATA SOURCES\n\n");
	enumerator = [[systems fullSystems] objectEnumerator];
	while(object = [enumerator nextObject])
		GSPrintf(stdout, @"%@\n", [[object dataSource] description]);
	
	GSPrintf(stdout, @"\nSYSTEMS\n\n");
	enumerator = [[systems allSystems] objectEnumerator];
	while(object = [enumerator nextObject])
		GSPrintf(stdout, @"%@\n", [object description]);
		
	GSPrintf(stdout, @"\nFORCE FIELDS\n\n");
	enumerator = [[forceFields forceFields] objectEnumerator];
	while(object = [enumerator nextObject])
		GSPrintf(stdout, @"%@\n", [object description]);
		
	GSPrintf(stdout, @"\nCONFIGURATION GENERATOR\n\n");
	GSPrintf(stdout, @"%@\n", [configurationGenerator description]);
	GSPrintf(stdout, @"\nCONTROLLER\n\n");
	GSPrintf(stdout, @"%@\n", [controller description]);
	GSPrintf(stdout, @"%@", divider);
}

/*
 * Creation/Destruction
 */

+ (id) appCore
{
	if(appCore == nil)
		return [AdCore new];
	else
		return appCore;
}

- (id) init
{
	if(appCore != nil)
		return appCore;

	if(self = [super init])
	{	
		corePool = [[NSAutoreleasePool alloc] init];
		commandErrors = [NSMutableDictionary new];
		//\note This may not be necessary - Leaving as a test
		commandResults = [NSMutableDictionary new];
		validCommands = [NSArray arrayWithObjects: 
					@"flushEnergies", 
					@"status", 
					@"reload",
					@"endSimulation", 
					@"controllerResults",
					 nil];
		[validCommands retain];

		//If the core has to exit for any reason this
		//notification will be posted
		[[NSNotificationCenter defaultCenter] addObserver: self
			selector: @selector(simulationFinished:)
			name: @"AdSimulationDidFinishNotification"
			object: nil];

		[[NSUserDefaults standardUserDefaults]
			registerDefaults: 
				[NSDictionary dictionaryWithObjectsAndKeys:
					[NSNumber numberWithInt: 100], 
					@"RollBackLimit", 
					[NSNumber numberWithInt: 5],
					@"MaximumRestartAttempts",
					[NSNumber numberWithBool: YES],
					@"InitialMinimisation",
					nil]];

		memoryManager = [AdMemoryManager new];
		ioManager = [AdIOManager appIOManager];
		runLoopIsRunning = NO;
		endSimulation = NO;
		checkpointTopology = YES;
		terminationError = nil;

		appCore = self;
	}	
	return self;
}

- (void) dealloc
{	
	NSDebugLLog(@"AdCore", @"Beginning deallocation");
	[corePool release];
	
	if([[NSUserDefaults standardUserDefaults]
		boolForKey: @"RunInteractive"])
	{	
		/*
		 * We need to make sure the thread gets a chance to 
		 * finish as if it doesn't it wont be deallocated.
		 * This is a problem since the thread retains the object 
		 * it was detached from (i.e. the controller) and hence 
		 * that object wont be released when the core exits if
		 * the thread hasn't already exited.
		 *
		 * If the controller retain count is not equal to one 
		 * the thread has not exited. Therefore we wait until 
		 * it is one before proceeding with deallocation so we can 
		 * be sure it will be released.
		*/

		if(controller != nil)
			while([controller retainCount] != 1)
				sleep(1);
	}			

	[productionCheckpoints release];
	[date release];
	[minimiser release];
	[dataWriter release];
	[configurationGenerator release];
	[controller release];
	[externalObjects release];
	[validCommands release];
	[terminationError release];
	[memoryManager release];
	NSDebugLLog(@"AdCore", @"Deallocation complete");
	[super dealloc];
}

- (BOOL) setup: (NSError**) error
{
	NSDictionary* template;
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	
	GSPrintf(stdout, @"%@", divider);
	template = [ioManager template];
	externalObjects = [ioManager externalObjects];
	
	templateProcessor = [AdTemplateProcessor new];
	GSPrintf(stdout, @"Validating simulation template\n");
	if(![templateProcessor validateTemplate: &template error: error])
		return NO;

	GSPrintf(stdout, @"Validation complete\n\n");
	[templateProcessor setTemplate: template];
	[templateProcessor setExternalObjects: externalObjects];
	GSPrintf(stdout, @"Processing simulation template\n");
	if(![templateProcessor processTemplate: error])
		return NO;
	
	configurationGenerator = [[templateProcessor configurationGenerator] retain];
	systems = [configurationGenerator systems];
	//FIXME Will all configuration generators use force fields?
	forceFields = [configurationGenerator forceFields];
	controller = [[templateProcessor controller] retain];
	externalObjects = [[templateProcessor externalObjects] retain];
	GSPrintf(stdout, @"Processing complete\n");
	GSPrintf(stdout, @"%@", divider);

	//Add references to the external object to the simulation data
	[ioManager setSimulationReferences: externalObjects];

	//Print a summary of the simulators state
	[self _printSummary];

	//Create a minimiser that will deal with exploding simulations
	//and perform an inital minimisation before beginning (if requested)	
	[self _createMinimiser];
	if([[NSUserDefaults standardUserDefaults] 
		boolForKey: @"InitialMinimisation"])
	{	
		[self _minimise];
	}	
	else
		GSPrintf(stdout, @"Skipping initial minimisation as requested\n");
	
	GSPrintf(stdout, @"%@", divider);
	[self setupCheckpointing];
	GSPrintf(stdout, @"%@", divider);
		
	[templateProcessor release];
	[pool release];
	AdLogMemoryUsage();

	return YES;
}

- (id) main: (NSDictionary*) dict
{
	GSPrintf(stdout, @"Calling controller %@\n", NSStringFromClass([controller class]));

	[controller coreWillStartSimulation: self];
	date = [[NSDate date] retain]; 
	terminationError = nil;

	if([[NSUserDefaults standardUserDefaults]
		boolForKey: @"RunInteractive"])
	{	
		//Run threaded
		[controller runThreadedController];
		GSPrintf(stdout, @"Entering run loop\n");
		[self startRunLoop];
	}
	else
		//Run normally
		[controller runController];
	
	/*
	 * Check if the controller exited due to an error. Thiw will
	 * be posted to AdunServer (or logged) when clean up is called.
	*/	
	terminationError = [[controller controllerError] retain];
	if(terminationError !=  nil)
		NSWarnLog(@"Error %@", [terminationError userInfo]);
	
	return nil;
}

- (void) cleanUp
{
	AdSimulationData* dataReader;
	id storage;

	GSPrintf(stdout, @"Requesting controller clean up\n");
	[controller cleanUp];
	GSPrintf(stdout, @"Saving system state\n");

	//Close any frame that may be open
	[dataWriter closeFrame];
	[dataWriter synchToStore];
	
	//Output information on what was collected
	GSPrintf(stdout, @"Simulation data summary -\n\n");
	dataReader = [AdSimulationData new];
	storage = [[AdFileSystemSimulationStorage alloc]
			initForReadingSimulationDataAtPath: 
	[[dataWriter dataStorage] storagePath]];
	[dataReader setDataStorage: storage];			
	[dataReader loadData];
	GSPrintf(stdout, @"%@", [dataReader description]);
	[dataReader release];
		
	GSPrintf(stdout, @"Outputting controller results (if any)\n");
	/*
	 * This will raise an exception if the entires in results are not
	 * all AdDataSets. We do not catch this error since this indicates
	 * a programmatic problem with the controller used.
	 */
	NSDebugLLog(@"AdCore", @"Outputting controller results\n");
	[ioManager saveResults: [controller simulationResults]];
	NSDebugLLog(@"AdCore", @"Complete");
}

- (id) flushEnergies: (NSDictionary*) aDict
{
	[dataWriter synchToStore];
	return nil;
}

- (id) controllerResults: (NSDictionary*) options
{
	return [controller simulationResults];
}

- (NSError*) terminationError
{
	return [[terminationError retain] autorelease];
}

- (AdConfigurationGenerator*) configurationGenerator
{
	return [[configurationGenerator retain] autorelease];
}

- (id) controller
{
	return [[controller retain] autorelease];
}

- (void) checkpointTopology: (BOOL) value
{
	checkpointTopology = value;
}

/*
 * AdCoreCommand Protocol Methods
 */

- (NSMutableDictionary*) optionsForCommand: (NSString*) name
{
	SEL methodSelector;
	NSString* methodName;

	methodName = [NSString stringWithFormat: @"%@Options", name];
	methodSelector = NSSelectorFromString(methodName);
	if(![self respondsToSelector: methodSelector])
		return nil;
	else
		return [self performSelector: methodSelector];
}

- (BOOL) validateCommand: (NSString*) name
{
	SEL command;

	NSDebugLLog(@"Execute", @"Validating command %@", name);

	command = NSSelectorFromString([NSString stringWithFormat: @"%@:", name]);
	return [self respondsToSelector: command];
}

- (NSError*) errorForCommand: (NSString*) name
{
	return [commandErrors objectForKey: name];
}

- (NSArray*) validCommands
{
	return validCommands;
}

- (void) setErrorForCommand: (NSString*) name description: (NSString*) description
{
	id error;

	error = AdCreateError(AdunCoreErrorDomain,
			AdCoreCommandError,
			@"A command error has occured",
			description,
			nil);
	
	[commandErrors setValue: error forKey: name];
}

@end


//Methods dealing with interactivity and threading
@implementation AdCore (AdCoreInteractionModeExtensions)

- (void) startRunLoop
{
	/*
	 * We stay in the run loop until the simulation finishes.
	 * When this happens we disconnect from the server and the runloop should end.
	 * Unfortunately I cannot get the runloop to exit gracefully. 
	 * Even though all the connections are invalidated limitDateForMode: still
	 * returns [NSDate distantFuture] for NSConnectionReplyMode. This implies
	 * there is still an input source in the runloop however its impossible	
	 * to find what it is. 
	 * Instead we have to use a flag which is set to true when the core 
	 * receives a simulationFinished notification - less elegant but what
	 * can you do?
	**/

	if(!runLoopIsRunning)
	{
		runLoopIsRunning = YES;
		//send server message indicating core is ready to recieve requests
		[ioManager acceptRequests];
		//End simulation is set to YES when simulationFinished is called.
		while(!endSimulation)
			[[NSRunLoop currentRunLoop] 
				runMode: NSDefaultRunLoopMode
				beforeDate: nil];

		runLoopIsRunning = NO;
	}
}

- (BOOL) simulationIsRunning
{
	return runLoopIsRunning;
}

/**
This method is called when the core recieves a AdSimulationDidFinishNotification. 
Set endSimulation to YES so we will break out of the runloop if it is running.
Note that this method is usually only sent by an AdController instance when it
has been running in a threaded mode.
*/
- (void) simulationFinished: (NSNotification*) aNotification
{
	endSimulation = YES;
}
@end


/*
Each checkpoint frame is associated with a production step.
It correpsonds to the state of the systems at the end of the step.
Since the first production step is 0 any frame associated with it
describes the systems when it has finished.

The state of a system before the production loop begins is
given an production step of -1. Since this state is
captured when an AdConfigurationGeneratorWillBeginProductionNotification 
is received one frame always corresponds to production step -1.
*/

@implementation AdCore (AdCheckpointing)

/**
Opens a checkpointing frame for the current iteration,
closing the previously opened frame. This method
has no effect if called more than once for a given
iteration of the configuration generator.
This method also adds an entry to the productionCheckpoints array
corresponding to the closed frame.
*/
- (void) openFrame
{
	int iteration;
	NSNumber* number; 

	iteration = [configurationGenerator currentStep];
	if(iteration == frameIteration)
		return;		//We have already opened a frame for this iteration
	else
	{
		//This is a different iteration to the last one recorded
		//Close the last frame and open a new one.
		NSDebugLLog(@"AdCore",
			@"Opening new frame : Iteration %d - last iteration %d",
			iteration, frameIteration);
		[dataWriter closeFrame];
		//Add an entry for the closed frame in productionCheckpoints	
		[productionCheckpoints insertObject: 
			[NSDictionary dictionaryWithObjectsAndKeys:
				[NSNumber numberWithInt: frameIteration], 
				@"Step",
				[NSNumber numberWithInt: [dataWriter lastFrame]],
				@"Frame", nil]
			atIndex: 0];
		//Update		
		frameIteration = iteration;
		if([configurationGenerator respondsToSelector: @selector(timeStep)])
		{
			number = [NSNumber numberWithDouble: 
					iteration*[(id)configurationGenerator timeStep]];
		}			
		else
			number = [NSNumber numberWithDouble: iteration];
			

		[dataWriter openFrame: number];
	}
}

//Recieved when a production loop starts
- (void) handleProductionStart: (NSNotification*) notification
{
	//Close any frame open since the last
	[dataWriter closeFrame];
	//Remove all entries from productionCheckpoints 
	[productionCheckpoints removeAllObjects];
	//Write a new frame corresponding to the start in production
	[forceFields evaluateEnergies]; 
	/*
	  Intial production checkpoint.
	  The timer is fired at the end of every step of the 
	  production loop. However since the first step is 0
	  this means iteration "0" will be the checkpoint
	  at the end of the first integration. Therefore we use -1
	  to indicate the initial energies/configuration
	 */
	[dataWriter openFrame: [NSNumber numberWithDouble: -1]];
	[dataWriter addTrajectoryCheckpoint];
	[dataWriter addEnergyCheckpoint];
	[dataWriter closeFrame];
	
	//Set the production boundary - We cant roll back past this point
	productionBoundary = [dataWriter lastFrame];
	frameIteration = -1;
	[productionCheckpoints addObject: 
		[NSDictionary dictionaryWithObjectsAndKeys:
			[NSNumber numberWithInt: frameIteration], 
			@"Step",
			[NSNumber numberWithInt: productionBoundary],
			@"Frame", nil]];

	GSPrintf(stdout, @"Checkpointed starting configuration for production loop\n"); 
}

//Recieved when a system reloads its data
- (void) handleSystemContentsChange: (NSNotification*) aNotification
{
	id object;

	//Check if this is an AdSystem object
	//FIXME: This will have to change when AdInteractionSystems
	//can handle bonded interactions
	object = [aNotification object];
	if(![object isKindOfClass: [AdSystem class]])
		return;

	if(checkpointTopology)
	{
		[self openFrame];
		NSDebugLLog(@"AdCore",
			@"Adding topology checkpoint to frame %d", frameIteration);
		[dataWriter addTopologyCheckpointForSystem: object];
	}
	else
		NSDebugLLog(@"AdCore",
			@"Requested not to checkpoint topologies");
}

- (void) setupCheckpointing
{
	int energyInterval, configurationInterval, flushInterval;
	NSString* iterationHeader;
	AdMainLoopTimer* scheduler;
	id template, simulationStorage;
	
	GSPrintf(stdout, @"Setting up checkpointing\n\n");
	
	scheduler = [AdMainLoopTimer mainLoopTimer];
	template = [ioManager template];
	energyInterval = [[template valueForKeyPath: @"checkpoint.energy"] intValue];
	configurationInterval = [[template valueForKeyPath: @"checkpoint.configuration"] intValue];
	flushInterval = [[template valueForKeyPath: @"checkpoint.energyDump"] intValue];
	
	GSPrintf(stdout, @"Energy interval - %d\n", energyInterval);
	GSPrintf(stdout, @"Configuration interval - %d\n", configurationInterval);
	GSPrintf(stdout, @"Energy flush interval - %d\n", flushInterval);
	
	[scheduler sendMessage: @selector(checkpointEnergy)
		toObject: self
		interval: energyInterval
		name: @"EnergyCheckpoint"];
	[scheduler sendMessage: @selector(checkpointTrajectory)
		toObject: self
		interval: configurationInterval 
		name: @"TrajectoryCheckpoint"];

	if([configurationGenerator respondsToSelector: @selector(timeStep)])
		iterationHeader = @"Time";
	else
		iterationHeader = @"Iteration";

	/*
	 * Create the simulation data writer
	 */
	//FIXME: See simulationWriteStorage docs
	simulationStorage = [ioManager simulationWriteStorage];
	dataWriter = [[AdSimulationDataWriter alloc]
			initWithDataStorage: simulationStorage
			systems: systems
			forceFields: forceFields
			iterationHeader: iterationHeader];
	
	if(flushInterval < (int)[configurationGenerator numberOfSteps])
		[scheduler
			sendMessage: @selector(synchToStore)
			toObject: dataWriter
			interval: flushInterval	
			name: @"StoreSynch"];

	//Register for AdSystemContentsDidChangeNotification's
	[[NSNotificationCenter defaultCenter]
		addObserver: self
		selector: @selector(handleSystemContentsChange:)
		name: @"AdSystemContentsDidChangeNotification"
		object: nil];

	//Register for starting of production loops
	//This enables us to record an initial checkpoint
	[[NSNotificationCenter defaultCenter]
		addObserver: self
		selector: @selector(handleProductionStart:)
		name: @"AdConfigurationGeneratorWillBeginProductionNotification"
		object: configurationGenerator];

	productionCheckpoints = [NSMutableArray new];	
	
	GSPrintf(stdout, @"\nComplete\n");
}

- (void) checkpointEnergy
{
	//Opens the frame if its not already open
	[self openFrame];
	[dataWriter addEnergyCheckpoint];
}

- (void) checkpointTrajectory
{
	[self openFrame];
	[dataWriter addTrajectoryCheckpoint];
}

- (NSDictionary*) dataForLastTrajectoryCheckpoint: (int) limit
			dataReader: (AdSimulationData*) dataReader
{
	int i, step, frame, index;
	NSMutableDictionary* dict;

	//We should always find at least the first frame (step  -1)
	for(frame = 0, i=0; i<(int)[productionCheckpoints count]; i++)
	{
		frame = [[[productionCheckpoints objectAtIndex: i] 
				objectForKey: @"Frame"] intValue];
		step = [[[productionCheckpoints objectAtIndex: i] 
				objectForKey: @"Step"] intValue];
		if(step <= limit)
		{
			if([[dataReader dataRecordedInFrame: frame]
				containsObject: @"Trajectory"])
				break;
		}		
	}

	dict = [[productionCheckpoints objectAtIndex: i] mutableCopy];
	
	//Find the trajectory checkpointing corresponding to frame
	for(i=(int)[dataReader numberTrajectoryCheckpoints] -1; i>=0; i--)
	{
		index = [dataReader frameForTrajectoryCheckpoint: i];
		if(index == frame)
			break;
	}

	[dict setObject: [NSNumber numberWithInt: i]
		forKey: @"Checkpoint"];

	return [dict autorelease];
}

@end

@implementation AdCore (AdCoreRollBackExtensions)

- (int) rollBackAndMinimise
{
	int restartPoint, energyInterval, configurationInterval;
	AdMainLoopTimer* scheduler;
	id template;

	restartPoint = [self rollBackToLastCheckpoint];
	
	/*
	 * Perform a minimisation on the starting state for the restartStep
	 *
	 * We have to remove and reset checkpoint timers 
	 * to avoid polluting the trajectory with the minimisation data.
	 * We also have to remove ourselves from observation of
	 * AdConfigurationGeneratorWillBeginProductionNotification so 
	 * handleProductionStart is not called.
	 */
	scheduler = [AdMainLoopTimer mainLoopTimer];
	[scheduler removeMessageWithName: @"EnergyCheckpoint"];
	[scheduler removeMessageWithName: @"TrajectoryCheckpoint"];
	[[NSNotificationCenter defaultCenter]
		removeObserver: self
		name: @"AdConfigurationGeneratorWillBeginProduction"
		object: configurationGenerator];
	
	NSWarnLog(@"Beginning minimisation of rolled back state");
	GSPrintf(stdout, @"Beginning minimisation of rolled back state\n");
	
	[minimiser setNumberOfSteps: 1000];
	[self _minimise];
	
	NSWarnLog(@"Minimisation complete - Resetting notifications");

	//Re-add all timers and notifications
	template = [ioManager template];
	energyInterval = [[template valueForKeyPath: @"checkpoint.energy"] intValue];
	configurationInterval = [[template valueForKeyPath: @"checkpoint.configuration"] intValue];
	[[NSNotificationCenter defaultCenter]
		addObserver: self
		selector: @selector(handleProductionStart:)
		name: @"AdConfigurationGeneratorWillBeginProduction"
		object: configurationGenerator];
	[scheduler sendMessage: @selector(checkpointEnergy)
		toObject: self
		interval: energyInterval
		name: @"EnergyCheckpoint"];
	[scheduler sendMessage: @selector(checkpointTrajectory)
		toObject: self
		interval: configurationInterval 
		name: @"TrajectoryCheckpoint"];

	NSWarnLog(@"Done");
	GSPrintf(stdout, @"Minimisation complete\n");
	
	return restartPoint;
}

- (int) rollBackToLastCheckpoint
{
	int captureMask;
	int currentStep, limitStep, rollBackLimit, index;
	int restartStep, restartFrame, topologyCheckpoint, restartCheckpoint;
	NSEnumerator* systemEnum;
	NSMutableArray* changedSystems = [NSMutableArray array];
	NSDictionary* dict;
	id dataSource, memento, system;
	AdSystemCollection* initialCollection;
	AdSimulationData* dataReader;
	AdFileSystemSimulationStorage* storage;
	
	//Before beginning make sure all data is written out
	//and create a data reader.
	[dataWriter synchToStore];
	
	dataReader = [AdSimulationData new];
	storage = [[AdFileSystemSimulationStorage alloc]
			initForReadingSimulationDataAtPath: 
				[[dataWriter dataStorage] storagePath]];
	[dataReader setDataStorage: storage];			
	[dataReader loadData];
	
	/*
	 * The rolling back is complicated by a number of factors ....
	 * The steps:
	 * 1) Get the last trajectory checkpoint at least RollBackLimit steps before now
	 *    without crossing any production boundaries. We dont cross these boundaries
	 *    because we cannot be sure of the relevance of any previous production run
	 *    to the current one.
	 * 2) Check for any topology changes in the interval between then and now.
	 *	A) If there was a change then:
	 *		i) Find who changed
	 *		ii) Find the last topology checkpoint for it before the choosen trajectory checkpoint
	 *		iii) If there is none use starting topology
	 * 3) Reset the system with the coordinates and possibly velocities/topology
	 * 	A) If no velocities reinitialise them
	 * 4) Roll back the simulation data
	 * 5) Reset all related AdCore instance variables
	 */
	
	currentStep = [configurationGenerator currentStep];
	rollBackLimit = [[NSUserDefaults standardUserDefaults]
				integerForKey: @"RollBackLimit"];
	limitStep = currentStep - rollBackLimit;
	//We cant go before step -1 (The very first checkpoint)
	if(limitStep < -1)
		limitStep = -1;

	GSPrintf(stdout, @"Beginning roll back of simulation\n");
	NSWarnLog(@"Beginning roll back of simulation\n");
	NSWarnLog(@"Current step: %d. Limit step %d. Last recorded frame %d", 
		currentStep, limitStep, [dataWriter lastFrame]);

	dict = [self dataForLastTrajectoryCheckpoint: limitStep
			dataReader: dataReader];
	restartStep = [[dict objectForKey: @"Step"] intValue]; 
	restartFrame = [[dict objectForKey: @"Frame"] intValue];
	restartCheckpoint = [[dict objectForKey: @"Checkpoint"] intValue];

	NSWarnLog(@"Restarting from production step %d. Frame %d. Trajectory checkpoint %d\n", 
			restartStep, restartFrame, restartCheckpoint);
	
	//Check if any topology changes occured between restartFrame and now
	NSWarnLog(@"Checking for topology changes");
	if((topologyCheckpoint = [dataWriter lastTopologyCheckpoint]) > restartFrame)
	{
		NSWarnLog(@"Topology changed after the restart frame - in frame %d", topologyCheckpoint);
		NSWarnLog(@"Checking who changed");
		//Find who changed
		systemEnum = [[systems fullSystems] objectEnumerator];
		while((system = [systemEnum nextObject]))
		{
			dataSource = [dataReader dataSourceForSystem: system
					inTopologyCheckpoint: topologyCheckpoint];
			if(dataSource != nil)
			{
				NSWarnLog(@"Changed system %@", [system systemName]);
				NSWarnLog(@"Marking for topology roll back");
					[changedSystems addObject: system];
			}	
		}
	}
	else
		NSWarnLog(@"None found");
		
	//Revert all systems to their state at restartFrame
	NSWarnLog(@"\nReturning systems to their state at step %d", restartStep);
	systemEnum = [[systems fullSystems] objectEnumerator];
	while((system = [systemEnum nextObject]))
	{
		NSWarnLog(@"System %@", [system systemName]);
		memento = [dataReader mementoForSystem: system 
				inTrajectoryCheckpoint: restartCheckpoint];
		if([changedSystems containsObject: system])
		{
			NSWarnLog(@"This system was marked for a topology roll back");
			dataSource = [dataReader lastRecordedDataSourceForSystem: system
					inRange: NSMakeRange(0,restartFrame)];
			if(dataSource == nil)
			{
				//We have to retrieve the orginal data source
				initialCollection = [dataReader systemCollection] ;
				dataSource = [[initialCollection systemWithName: [system systemName]] 
						dataSource];
			}
			NSWarnLog(@"Resetting and reloading topology");
			[system setDataSource: dataSource];
			[system reloadData];
			NSWarnLog(@"Complete");
		}	
		NSWarnLog(@"Rolling back dynamic state");
		[system returnToState: memento];
		NSWarnLog(@"Complete");
		//If no velocites were included in the memento reinitialise them now
		captureMask = [[memento valueForMetadataKey: @"MementoMask"]
				intValue];
		if(!(captureMask & AdSystemVelocitiesMemento))
			[system reinitialiseVelocities];
	}

	NSWarnLog(@"\nAll systems rolled back. Deleting data acquired after restart step");
	//Delete all data after restartFrame
	[dataWriter rollBackToFrame: restartFrame];
	[dataReader release];
	[storage release];
	NSWarnLog(@"Complete. Simulation data synched to current state");

	/*
	 * We have to reset the checkpoint related instance variables 
	 * - frameIteration & productionCheckpoints - so they reflect the
	 * rolled back state. This means they must show that the last checkpointed
	 * frame was the restart frame. This is slightly tricky since AdCore
	 * does not close a frame until the next one is open. Therefore we
	 * have to account for the fact that when openFrame() is next called
	 * it will peform frame closing related tasks on the restartFrame.
	 *
	 * To do this we remove all entries from productionCheckpoints that
	 * correspond to frames after the restart frame including the restartFrame
	 * itself. This is because when openFrame() closes the
	 * the restartFrame it will add data on it to productionCheckpoints.
	 * Hence we must remove its entry aswell to avoid duplicates.
	 */
	
	NSWarnLog(@"Updating internal ivars to reflect new state");
	frameIteration = restartFrame;
	index = [productionCheckpoints indexOfObject: dict];
	[productionCheckpoints removeObjectsInRange: NSMakeRange(0, index+1)];

	//Reset the timer
	[[AdMainLoopTimer mainLoopTimer] resetAll];

	//Broadcast a system contents did change for each system to ensure
	//everything is updated to the new coodinates 
	//We set checkpointTopology to NO since we dont need to
	
	NSWarnLog(@"Notifying all objects of change in system contents\n");

	checkpointTopology = NO;
	systemEnum = [[systems fullSystems] objectEnumerator];
	while((system = [systemEnum nextObject]))
	{
		NSWarnLog(@"Broadcasting notification of change for %@", [system systemName]);
		[[NSNotificationCenter defaultCenter]
			postNotificationName: @"AdSystemContentsDidChangeNotification"
			object: system];
		NSWarnLog(@"Done");
	}	
	checkpointTopology = YES;

	//restartStep corresponds to the state we went back to. 
	//Since the state was captured at the end of this step any production
	//loop restarting should start from restartStep + 1
 	NSWarnLog(@"Roll back complete. The next production step will be step %d\n", restartStep + 1);
	GSPrintf(stdout, @"Roll back complete. The next production step will be step %d\n", 
			restartStep + 1);
	return restartStep + 1;
}

@end
