# Analysis drivers.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/prelude"
require "adlint/traits"
require "adlint/message"
require "adlint/analyzer"

module AdLint #:nodoc:
  def setup(command, traits_fpath, verbose)
    self.verbose = verbose
    $stdout.sync = true if verbose?
    at_exit { print_elapsed_time($stdout) if verbose? }

    # NOTE: Psych::SyntaxError is a subclass of the builtin ::SyntaxError.
    #       The rescue clause without error_type is equivalent to
    #       `rescue StandardError'.  So, the rescue clause without error_type
    #       cannot catch Psych::SyntaxError.

    begin
      Traits.instance.read_from(traits_fpath)
      unless Traits.instance.valid?
        $stderr.puts "#{command}: Traits file `#{traits_fpath}' is invalid."
        $stderr.puts
        $stderr.puts "Detailed message is below;"
        Traits.instance.errors.each_with_index do |error, index|
          $stderr.puts "#{index + 1}. #{error}"
        end
        $stderr.puts
        exit 3
      end
    rescue SyntaxError, StandardError => ex
      $stderr.puts "#{command}: Failed to read `#{traits_fpath}'."
      $stderr.puts
      $stderr.puts "Detailed message is below;"
      $stderr.puts ex.message, ex.backtrace
      $stderr.puts
      exit 2
    end

    begin
      MessageCatalog.instance.load
    rescue SyntaxError, StandardError => ex
      $stderr.puts "#{command}: Failed to read the message catalog for " +
        "`#{Traits.instance.of_message.language}'."
      $stderr.puts
      $stderr.puts "Detailed message is below;"
      $stderr.puts ex.message, ex.backtrace
      $stderr.puts
      exit 2
    end
  end
  module_function :setup

  def sma(fpaths, strip_num = 0, output_dpath = nil)
    failed = false
    fpaths.each do |fpath|
      FileUtils.mkdir_p(sma_output_dpath(fpath, strip_num, output_dpath))
      unless SingleModuleAnalyzer.new(fpath, strip_num, output_dpath).run
        failed = true
      end
    end
    exit 4 if failed
  rescue => ex
    $stderr.puts ex.message, ex.backtrace
    $stderr.puts
    exit 2
  end
  module_function :sma

  def cma(fpaths, output_dpath = nil)
    FileUtils.mkdir_p(output_dpath) if output_dpath
    unless CrossModuleAnalyzer.new(fpaths, output_dpath).run
      exit 5
    end
  rescue => ex
    $stderr.puts ex.message, ex.backtrace
    $stderr.puts
    exit 2
  end
  module_function :cma

  def chk(fpaths, strip_num = 0, output_dpath = nil)
    failed = false
    fpaths.each do |fpath|
      FileUtils.mkdir_p(chk_output_dpath(fpath, strip_num, output_dpath))
      unless ConfigurationValidator.new(fpath, strip_num, output_dpath).run
        failed = true
      end
    end
    exit 6 if failed
  rescue => ex
    $stderr.puts ex.message, ex.backtrace
    $stderr.puts
    exit 2
  end
  module_function :chk

  def sma_output_dpath(fpath, strip_num = 0, output_dpath = nil)
    fpath.strip(strip_num).expand_path(output_dpath).dirname
  end
  module_function :sma_output_dpath

  def chk_output_dpath(fpath, strip_num = 0, output_dpath = nil)
    sma_output_dpath(fpath, strip_num, output_dpath)
  end
  module_function :chk_output_dpath

  def sma_met_fpaths(fpaths, strip_num = 0, output_dpath = nil)
    fpaths.map do |fpath|
      if output_dpath
        output_dpath.join(fpath.strip(strip_num)).add_ext(".met.csv")
      else
        fpath.strip(strip_num).add_ext(".met.csv")
      end
    end
  end
  module_function :sma_met_fpaths
end
