# C preprocessor.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/error"
require "adlint/source"
require "adlint/report"
require "adlint/cpp/syntax"
require "adlint/cpp/source"
require "adlint/cpp/macro"
require "adlint/cpp/lexer"
require "adlint/cpp/constexpr"

module AdLint #:nodoc:
module Cpp #:nodoc:

  # == DESCRIPTION
  # C preprocessor language evaluator.
  #
  # Preprocessor executes recursive descent parsing and evaluation at a time.
  class Preprocessor
    include ReportUtil

    def execute(context, source)
      @context = context
      context.push_lexer(create_lexer(context, source))
      preprocessing_file(context)
    end

    extend Pluggable

    def_plugin :on_user_header_included
    def_plugin :on_system_header_included
    def_plugin :on_object_like_macro_defined
    def_plugin :on_function_like_macro_defined
    def_plugin :on_va_function_like_macro_defined
    def_plugin :on_macro_undefined
    def_plugin :on_asm_section_evaled
    def_plugin :on_unknown_pragma_evaled
    def_plugin :on_pp_token_extracted
    def_plugin :on_block_comment_found
    def_plugin :on_line_comment_found
    def_plugin :on_nested_block_comment_found
    def_plugin :on_eof_newline_not_found
    def_plugin :on_unlexable_char_found
    def_plugin :on_cr_at_eol_found
    def_plugin :on_eof_mark_at_eof_found
    def_plugin :on_illformed_defined_op_found
    def_plugin :on_undefined_macro_referred

    private
    def preprocessing_file(context)
      PreprocessingFile.new(context.translation_unit_fpath, group(context))
    end

    def group(context)
      if group_part = group_part(context)
        group = Group.new.push(group_part)
        while group_part = group_part(context)
          group.push(group_part)
        end
        return group
      end
      nil
    end

    def group_part(context)
      if top_token = context.top_token
        case top_token.type
        when :IF, :IFDEF, :IFNDEF
          return if_section(context)
        when :INCLUDE, :INCLUDE_NEXT, :DEFINE, :UNDEF, :LINE, :ERROR, :PRAGMA
          return control_line(context)
        when :ASM
          return asm_section(context)
        when :NULL_DIRECTIVE
          return NullDirective.new(context.next_token)
        when :TEXT_LINE
          text_line = TextLine.new(context.next_token)
          tokens = TextLineNormalizer.normalize(text_line, context)
          if tokens
            context.deferred_text_lines.clear
            tokens.each do |token|
              context.source.add_token(token)
              notify_pp_token_extracted(token)
            end
          else
            context.deferred_text_lines.push(text_line)
          end
          return text_line
        end
      end
      nil
    end

    def if_section(context)
      context.push_branch
      if_group = if_group(context)

      while top_token = context.top_token
        case top_token.type
        when :ELIF
          elif_groups = elif_groups(context)
        when :ELSE
          else_group = else_group(context)
        when :ENDIF
          endif_line = endif_line(context)
          break
        end
      end

      E(:E0004, if_group.location) unless endif_line

      context.pop_branch
      IfSection.new(if_group, elif_groups, else_group, endif_line)
    end

    def if_group(context)
      if keyword = context.top_token
        case keyword.type
        when :IF
          return if_statement(context)
        when :IFDEF
          return ifdef_statement(context)
        when :IFNDEF
          return ifndef_statement(context)
        end
      end
      nil
    end

    def if_statement(context)
      keyword = context.next_token
      unless pp_tokens = pp_tokens(context)
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      expression = ExpressionNormalizer.normalize(pp_tokens, context, self)
      if expression.value == 0
        context.skip_group
      else
        group = group(context)
        context.branch_evaluated = true
      end
      IfStatement.new(keyword, expression, group)
    end

    def ifdef_statement(context)
      keyword = context.next_token
      unless identifier = context.next_token and identifier.type == :IDENTIFIER
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      if macro_defined?(context, identifier)
        group = group(context)
        context.branch_evaluated = true
      else
        context.skip_group
      end
      IfdefStatement.new(keyword, identifier, group)
    end

    def ifndef_statement(context)
      keyword = context.next_token
      unless identifier = context.next_token and identifier.type == :IDENTIFIER
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      if macro_defined?(context, identifier)
        context.skip_group
      else
        group = group(context)
        context.branch_evaluated = true
      end
      IfndefStatement.new(keyword, identifier, group)
    end

    def elif_groups(context)
      if elif_group = elif_group(context)
        elif_groups = ElifGroups.new.push(elif_group)
        while elif_group = elif_group(context)
          elif_groups.push(elif_group)
        end
        return elif_groups
      end
      nil
    end

    def elif_group(context)
      unless top_token = context.top_token and top_token.type == :ELIF
        return nil
      end

      if keyword = context.next_token
        if keyword.type == :ELIF
          unless pp_tokens = pp_tokens(context)
            return nil
          end
          unless new_line = context.next_token and new_line.type == :NEW_LINE
            return nil
          end
          expression = ExpressionNormalizer.normalize(pp_tokens, context, self)
          if context.branch_evaluated? || expression.value == 0
            context.skip_group
          else
            group = group(context)
            context.branch_evaluated = true
          end
          return ElifStatement.new(keyword, expression, group)
        end
      end
      nil
    end

    def else_group(context)
      if keyword = context.next_token
        if keyword.type == :ELSE
          unless new_line = context.next_token and new_line.type == :NEW_LINE
            return nil
          end
          if context.branch_evaluated?
            context.skip_group
          else
            group = group(context)
            context.branch_evaluated = true
          end
          return ElseStatement.new(keyword, group)
        end
      end
      nil
    end

    def endif_line(context)
      if keyword = context.next_token
        if keyword.type == :ENDIF
          unless new_line = context.next_token and new_line.type == :NEW_LINE
            return nil
          end
          return EndifLine.new(keyword)
        end
      end
      nil
    end

    def control_line(context)
      if keyword = context.top_token
        case keyword.type
        when :INCLUDE
          return include_line(context)
        when :INCLUDE_NEXT
          return include_next_line(context)
        when :DEFINE
          return define_line(context)
        when :UNDEF
          return undef_line(context)
        when :LINE
          return line_line(context)
        when :ERROR
          return error_line(context)
        when :PRAGMA
          return pragma_line(context)
        end
      end
      nil
    end

    def include_line(context)
      keyword = context.next_token
      if header_name = context.top_token
        case header_name.type
        when :USR_HEADER_NAME
          return user_include_line(context, keyword)
        when :SYS_HEADER_NAME
          return system_include_line(context, keyword)
        else
          return macro_include_line(context, keyword)
        end
      end
      nil
    end

    def include_next_line(context)
      keyword = context.next_token
      if header_name = context.top_token
        case header_name.type
        when :USR_HEADER_NAME
          return user_include_next_line(context, keyword)
        when :SYS_HEADER_NAME
          return system_include_next_line(context, keyword)
        else
          return macro_include_next_line(context, keyword)
        end
      end
      nil
    end

    def user_include_line(context, keyword)
      header_name = context.next_token
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      user_include_line =
        UserIncludeLine.new(keyword, header_name, context.include_depth)
      include_first_user_header(user_include_line, context)
      user_include_line
    end

    def user_include_next_line(context, keyword)
      header_name = context.next_token
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      user_include_next_line =
        UserIncludeNextLine.new(keyword, header_name, context.include_depth)
      include_next_user_header(user_include_next_line, context)
      user_include_next_line
    end

    def system_include_line(context, keyword)
      header_name = context.next_token
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      system_include_line =
        SystemIncludeLine.new(keyword, header_name, context.include_depth)
      include_first_system_header(system_include_line, context)
      system_include_line
    end

    def system_include_next_line(context, keyword)
      header_name = context.next_token
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      system_include_next_line =
        SystemIncludeNextLine.new(keyword, header_name, context.include_depth)
      include_next_system_header(system_include_next_line, context)
      system_include_next_line
    end

    def macro_include_line(context, keyword)
      unless pp_tokens = pp_tokens(context)
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      PPTokensNormalizer.normalize(pp_tokens, context)
      return nil if pp_tokens.tokens.empty?
      case parameter = pp_tokens.tokens.map { |t| t.value }.join
      when /\A".*"\z/
        user_include_line = UserIncludeLine.new(
          keyword, Token.new(:USR_HEADER_NAME, parameter,
                             pp_tokens.tokens.first.location),
          context.include_depth)
        include_first_user_header(user_include_line, context)
        return user_include_line
      when /\A<.*>\z/
        system_include_line = SystemIncludeLine.new(
          keyword, Token.new(:SYS_HEADER_NAME, parameter,
                             pp_tokens.tokens.first.location),
          context.include_depth)
        include_first_system_header(system_include_line, context)
        return system_include_line
      end
      nil
    end

    def macro_include_next_line(context, keyword)
      unless pp_tokens = pp_tokens(context)
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      PPTokensNormalizer.normalize(pp_tokens, context)
      return nil if pp_tokens.tokens.empty?
      case parameter = pp_tokens.tokens.map { |t| t.value }.join
      when /\A".*"\z/
        user_include_next_line = UserIncludeNextLine.new(
          keyword, Token.new(:USR_HEADER_NAME, parameter,
                             pp_tokens.tokens.first.location),
          context.include_depth)
        include_next_user_header(user_include_next_line, context)
        return user_include_next_line
      when /\A<.*>\z/
        system_include_next_line = SystemIncludeNextLine.new(
          keyword, Token.new(:SYS_HEADER_NAME, parameter,
                             pp_tokens.tokens.first.location),
          context.include_depth)
        include_next_system_header(system_include_next_line, context)
        return system_include_next_line
      end
      nil
    end

    def define_line(context)
      keyword = context.next_token
      unless identifier = context.next_token and identifier.type == :IDENTIFIER
        return nil
      end
      symbol = context.symbol_table.create_new_symbol(MacroName, identifier)

      if paren = context.top_token and paren.type == "("
        context.next_token
        identifier_list = identifier_list(context)
        unless paren_or_ellipsis = context.next_token
          return nil
        end
        case paren_or_ellipsis.type
        when "..."
          ellipsis = paren_or_ellipsis
          if paren = context.top_token and paren.type == ")"
            context.next_token
          else
            return nil
          end
        when ")"
          ellipsis = nil
        else
          return nil
        end
        replacement_list = replacement_list(context)
        unless new_line = context.next_token and new_line.type == :NEW_LINE
          return nil
        end
        if ellipsis
          define_line = VaFunctionLikeDefineLine.new(keyword, identifier,
                                                     identifier_list,
                                                     replacement_list, symbol)
          macro = FunctionLikeMacro.new(define_line)
          notify_va_function_like_macro_defined(define_line, macro)
        else
          define_line = FunctionLikeDefineLine.new(keyword, identifier,
                                                   identifier_list,
                                                   replacement_list, symbol)
          macro = FunctionLikeMacro.new(define_line)
          notify_function_like_macro_defined(define_line, macro)
        end
      else
        replacement_list = replacement_list(context)
        unless new_line = context.next_token and new_line.type == :NEW_LINE
          return nil
        end
        define_line = ObjectLikeDefineLine.new(keyword, identifier,
                                               replacement_list, symbol)
        macro = ObjectLikeMacro.new(define_line)
        notify_object_like_macro_defined(define_line, macro)
      end

      context.macro_table.define(macro)
      define_line
    end

    def undef_line(context)
      keyword = context.next_token
      unless identifier = context.next_token and identifier.type == :IDENTIFIER
        return nil
      end
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end

      undef_line = UndefLine.new(keyword, identifier)
      macro = context.macro_table.lookup(identifier.value)
      # NOTE: Undefining macro may be nil if not defined.
      notify_macro_undefined(undef_line, macro)

      context.macro_table.undef(identifier.value)
      undef_line
    end

    def line_line(context)
      keyword = context.next_token
      pp_tokens = pp_tokens(context)
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end

      # NOTE: The ISO C99 standard saids;
      #
      # 6.10.4 Line control
      #
      # Semantics
      #
      # 5 A preprocessing directive of the form
      #    # line pp-tokens new-line
      #   that does not match one of the two previous forms is permitted.  The
      #   preprocessing tokens after line on the directive are processed just
      #   as in normal text (each identifier currently defined as a macro name
      #   is replaced by its replacement list of preprocessing tokens).  The
      #   directive resulting after all replacements shall match one of the two
      #   previous forms and is then processed as appropriate.
      PPTokensNormalizer.normalize(pp_tokens, context) if pp_tokens

      LineLine.new(keyword, pp_tokens)
    end

    def error_line(context)
      keyword = context.next_token
      pp_tokens = pp_tokens(context)
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      ErrorLine.new(keyword, pp_tokens)
    end

    def pragma_line(context)
      keyword = context.next_token
      pp_tokens = pp_tokens(context)
      unless new_line = context.next_token and new_line.type == :NEW_LINE
        return nil
      end
      pragma_line = PragmaLine.new(keyword, pp_tokens)
      if pp_tokens && pp_tokens.tokens.size == 1 &&
          pp_tokens.tokens.first.value == "once"
        context.once_set.add(keyword.location.fpath)
      else
        notify_unknown_pragma_evaled(pragma_line)
      end
      pragma_line
    end

    def identifier_list(context)
      unless identifier = context.top_token and identifier.type == :IDENTIFIER
        return nil
      end
      identifier_list = IdentifierList.new
      while token = context.next_token
        if token.type == :IDENTIFIER
          identifier_list.push(token)
        end
        if comma = context.top_token and comma.type == ","
          context.next_token
        else
          break
        end
        unless identifier = context.top_token and
            identifier.type == :IDENTIFIER
          break
        end
      end
      identifier_list
    end

    def replacement_list(context)
      pp_tokens(context)
    end

    def pp_tokens(context)
      unless pp_token = context.top_token and pp_token.type == :PP_TOKEN
        return nil
      end
      pp_tokens = PPTokens.new
      while token = context.top_token
        if token.type == :PP_TOKEN
          pp_tokens.push(context.next_token)
        else
          break
        end
      end
      pp_tokens
    end

    def asm_section(context)
      asm_line = asm_line(context)
      context.skip_group
      endasm_line = endasm_line(context)
      asm_section = AsmSection.new(asm_line, endasm_line)
      notify_asm_section_evaled(asm_section)
      asm_section
    end

    def asm_line(context)
      if keyword = context.next_token
        if keyword.type == :ASM
          unless new_line = context.next_token and new_line.type == :NEW_LINE
            return nil
          end
          return AsmLine.new(keyword)
        end
      end
      nil
    end

    def endasm_line(context)
      if keyword = context.next_token
        if keyword.type == :ENDASM
          unless new_line = context.next_token and new_line.type == :NEW_LINE
            return nil
          end
          return EndasmLine.new(keyword)
        end
      end
      nil
    end

    def include_first_user_header(user_include_line, context)
      basename =
        user_include_line.header_name.value.sub(/\A"(.*)"\z/, "\\1")
      current_dpath = user_include_line.location.fpath.dirname
      if fpath = resolve_first_user_header(basename, current_dpath)
        include_user_header(fpath, user_include_line, context)
      else
        E(:E0010, user_include_line.location, basename)
        raise MissingUserHeaderError.new(user_include_line.location, basename,
                                         context.msg_fpath, context.log_fpath)
      end
    end

    def include_next_user_header(user_include_next_line, context)
      basename =
        user_include_next_line.header_name.value.sub(/\A"(.*)"\z/, "\\1")
      current_dpath = user_include_next_line.location.fpath.dirname
      if fpath = resolve_next_user_header(basename, current_dpath)
        include_user_header(fpath, user_include_next_line, context)
      else
        E(:E0010, user_include_next_line.location, basename)
        raise MissingUserHeaderError.new(user_include_next_line.location,
                                         basename,
                                         context.msg_fpath, context.log_fpath)
      end
    end

    def include_user_header(fpath, user_include_line, context)
      unless context.once_set.include?(fpath)
        Analyzer.current.info("including \"#{fpath}\" at " +
                              "#{user_include_line.location.to_s}")
        user_include_line.fpath = fpath
        user_header = UserHeader.new(fpath, user_include_line.location)
        context.push_lexer(create_lexer(context, user_header))
        context.sources.push(user_header)
        notify_user_header_included(user_include_line, user_header)
      end
    end

    def include_first_system_header(system_include_line, context)
      basename =
        system_include_line.header_name.value.sub(/\A<(.*)>\z/, "\\1")
      if fpath = resolve_first_system_header(basename)
        include_system_header(fpath, system_include_line, context)
      else
        E(:E0009, system_include_line.location, basename)
        raise MissingSystemHeaderError.new(system_include_line.location,
                                           basename,
                                           context.msg_fpath,
                                           context.log_fpath)
      end
    end

    def include_next_system_header(system_include_next_line, context)
      basename =
        system_include_next_line.header_name.value.sub(/\A<(.*)>\z/, "\\1")
      if fpath = resolve_next_system_header(basename)
        include_system_header(fpath, system_include_next_line, context)
      else
        E(:E0009, system_include_line.location, basename)
        raise MissingSystemHeaderError.new(system_include_next_line.location,
                                           basename,
                                           context.msg_fpath,
                                           context.log_fpath)
      end
    end

    def include_system_header(fpath, system_include_line, context)
      unless context.once_set.include?(fpath)
        Analyzer.current.info("including <#{fpath}> at " +
                              "#{system_include_line.location.to_s}")
        system_include_line.fpath = fpath
        system_header = SystemHeader.new(fpath, system_include_line.location)
        context.push_lexer(create_lexer(context, system_header))
        context.sources.push(system_header)
        notify_system_header_included(system_include_line, system_header)
      end
    end

    def resolve_first_user_header(basename, current_dpath)
      resolve_user_headers(basename, current_dpath, 1).first
    end

    def resolve_next_user_header(basename, current_dpath)
      resolve_user_headers(basename, current_dpath, 2).last
    end

    def resolve_user_headers(basename, current_dpath, max_num)
      include_paths = [current_dpath]
      include_paths += Traits.instance.of_project.include_path
      include_paths += Traits.instance.of_compiler.include_path

      result = []
      include_paths.each do |dpath|
        fpath = dpath.join(Pathname.new(basename))
        fpath = Pathname.new(fpath.to_s.gsub(/\\\\|\\/, "/"))
        result.push(fpath) if fpath.readable?
        break if result.size == max_num
      end
      result
    end

    def resolve_first_system_header(basename)
      resolve_system_headers(basename, 1).first
    end

    def resolve_next_system_header(basename)
      resolve_system_headers(basename, 2).last
    end

    def resolve_system_headers(basename, max_num)
      include_paths = Traits.instance.of_project.include_path
      include_paths += Traits.instance.of_compiler.include_path

      result = []
      include_paths.each do |dpath|
        fpath = dpath.join(Pathname.new(basename))
        fpath = Pathname.new(fpath.to_s.gsub(/\\\\|\\/, "/"))
        result.push(fpath) if fpath.readable?
        break if result.size == max_num
      end
      result
    end

    def macro_defined?(context, identifier)
      if macro = context.macro_table.lookup(identifier.value)
        macro.define_line.mark_as_referred_by(identifier)
        true
      else
        false
      end
    end

    def create_lexer(context, source)
      Lexer.new(source).tap { |lexer| attach_lexer_plugin(lexer, context) }
    end

    def attach_lexer_plugin(lexer, context)
      lexer.on_block_comment_found += lambda { |comment, location|
        on_block_comment_found.invoke(comment, location)
      }
      lexer.on_line_comment_found += lambda { |comment, location|
        on_line_comment_found.invoke(comment, location)
      }
      lexer.on_nested_block_comment_found += lambda { |location|
        on_nested_block_comment_found.invoke(location)
      }
      lexer.on_unterminated_block_comment += lambda { |location|
        handle_unterminated_block_comment(context, location)
      }
      lexer.on_eof_newline_not_found += lambda { |location|
        on_eof_newline_not_found.invoke(location)
      }
      lexer.on_unlexable_char_found += lambda { |char, location|
        on_unlexable_char_found.invoke(char, location)
      }
      lexer.on_cr_at_eol_found += lambda { |location|
        on_cr_at_eol_found.invoke(location)
      }
      lexer.on_eof_mark_at_eof_found += lambda { |location|
        on_eof_mark_at_eof_found.invoke(location)
      }
    end

    def notify_user_header_included(user_include_line, user_header)
      on_user_header_included.invoke(user_include_line, user_header)
    end

    def notify_system_header_included(system_include_line, system_header)
      on_system_header_included.invoke(system_include_line, system_header)
    end

    def notify_object_like_macro_defined(define_line, macro)
      on_object_like_macro_defined.invoke(define_line, macro)
    end

    def notify_function_like_macro_defined(define_line, macro)
      on_function_like_macro_defined.invoke(define_line, macro)
    end

    def notify_va_function_like_macro_defined(define_line, macro)
      on_va_function_like_macro_defined.invoke(define_line, macro)
    end

    def notify_macro_undefined(undef_line, macro)
      on_macro_undefined.invoke(undef_line, macro)
    end

    def notify_asm_section_evaled(asm_section)
      on_asm_section_evaled.invoke(asm_section)
    end

    def notify_unknown_pragma_evaled(pragma_line)
      on_unknown_pragma_evaled.invoke(pragma_line)
    end

    def notify_pp_token_extracted(pp_token)
      on_pp_token_extracted.invoke(pp_token)
    end

    def notify_illformed_defined_op_found(location)
      on_illformed_defined_op_found.invoke(location)
    end

    def notify_undefined_macro_referred(identifier)
      on_undefined_macro_referred.invoke(identifier)
    end

    def handle_unterminated_block_comment(context, location)
      E(:E0016, location)
      raise UnterminatedCommentError.new(location,
                                         context.msg_fpath, context.log_fpath)
    end

    def report
      @context.report
    end
  end

  class PreprocessContext
    def initialize(phase_context)
      @phase_context = phase_context
      @deferred_text_lines = []
      @lexer_stack = []
      @branch_stack = []
      @once_set = Set.new
    end

    attr_reader :deferred_text_lines
    attr_reader :once_set

    def translation_unit_fpath
      @phase_context[:sources].first.fpath
    end

    def source
      @phase_context[:c_source]
    end

    def sources
      @phase_context[:sources]
    end

    def symbol_table
      @phase_context[:symbol_table]
    end

    def macro_table
      @phase_context[:cpp_macro_table]
    end

    def report
      @phase_context.report
    end

    def msg_fpath
      @phase_context.msg_fpath
    end

    def log_fpath
      @phase_context.log_fpath
    end

    def push_lexer(lexer)
      @lexer_stack.push(lexer)
    end

    def top_token
      return nil if @lexer_stack.empty?
      unless token = @lexer_stack.last.top_token
        @lexer_stack.pop
        top_token
      else
        token
      end
    end

    def next_token
      return nil unless top_token
      @last_token = @lexer_stack.last.next_token
    end

    def skip_group
      until @lexer_stack.last.skip_group
        @lexer_stack.pop
        break if @lexer_stack.empty?
      end
    end

    def push_branch
      @branch_stack.push(false)
    end

    def pop_branch
      @branch_stack.pop
    end

    def branch_evaluated=(evaluated)
      @branch_stack[-1] = evaluated
    end

    def branch_evaluated?
      @branch_stack.last
    end

    def include_depth
      @lexer_stack.size
    end
  end

  module PPTokensNormalizer
    def normalize(pp_tokens, context)
      context.macro_table.replace(pp_tokens.tokens)
      pp_tokens
    end
    module_function :normalize
  end

  module ExpressionNormalizer
    def normalize(pp_tokens, context, preprocessor = nil)
      PPTokensNormalizer.normalize(pp_tokens, context)
      const_expr = ConstantExpression.new(context, pp_tokens.tokens)
      if preprocessor
        const_expr.on_illformed_defined_op_found +=
          preprocessor.method(:notify_illformed_defined_op_found)
        const_expr.on_undefined_macro_referred +=
          preprocessor.method(:notify_undefined_macro_referred)
      end
      const_expr.evaluate
    end
    module_function :normalize
  end

  module TextLineNormalizer
    def normalize(text_line, context)
      pp_tokens = []
      unless context.deferred_text_lines.empty?
        context.deferred_text_lines.each do |deferred_line|
          lexer = TextLineToPPTokensLexer.new(deferred_line)
          pp_tokens += lexer.execute.to_a
        end
      end

      lexer = TextLineToPPTokensLexer.new(text_line)
      pp_tokens += lexer.execute.to_a

      function_like_macro_referred = pp_tokens.any? { |t|
        (macro = context.macro_table.lookup(t.value)) ?
          macro.function_like? : false
      }

      if function_like_macro_referred
        return nil unless complete_macro_reference?(pp_tokens, context)
      end

      context.macro_table.replace(pp_tokens)
      pp_tokens
    end
    module_function :normalize

    def complete_macro_reference?(pp_tokens, context)
      index = 0
      while token = pp_tokens[index]
        index += 1
        macro = context.macro_table.lookup(token.value)
        if macro && macro.function_like?
          next if not_calling_function_like_macro?(pp_tokens, index)
        else
          next
        end

        # NOTE: It's not completed when a new-line appears after the macro
        #       name.
        return false unless pp_tokens[index..-1].any? { |t| t.value == "(" }

        paren_count = 0
        while token = pp_tokens[index]
          case token.value
          when "("
            paren_count += 1
          when ")"
            paren_count -= 1
            break if paren_count == 0
          end
          index += 1
        end

        return false if paren_count > 0
      end
      true
    end
    module_function :complete_macro_reference?

    def not_calling_function_like_macro?(pp_tokens, index)
      while pp_token = pp_tokens[index]
        case
        when pp_token.value == "("
          return false
        when pp_token.type == :NEW_LINE
          index += 1
        else
          return true
        end
      end
      false
    end
    module_function :not_calling_function_like_macro?
  end

end
end
