# Kernel of the abstract interpreter of C language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/monitor"
require "adlint/c/environ"
require "adlint/c/resolver"
require "adlint/c/mediator"
require "adlint/c/syntax"
require "adlint/c/expr"
require "adlint/c/conv"
require "adlint/c/option"
require "adlint/c/util"

module AdLint #:nodoc:
module C #:nodoc:

  class Program
    def initialize(interpreter, translation_unit)
      @interpreter = interpreter
      @translation_unit = translation_unit
    end

    def execute
      @interpreter.notify_translation_unit_started(@translation_unit)
      @translation_unit.accept(ExecutionDriver.new(@interpreter))
      @interpreter.notify_translation_unit_ended(@translation_unit)
    end

    class ExecutionDriver < SyntaxTreeVisitor
      include InterpreterMediator

      def initialize(interpreter)
        @interpreter = interpreter
      end

      def visit_declaration(node) interpret(node) end
      def visit_ansi_function_definition(node) interpret(node) end
      def visit_kandr_function_definition(node) interpret(node) end

      private
      def interpreter; @interpreter end
    end
    private_constant :ExecutionDriver
  end

  class Interpreter
    include InterpreterMediator
    include Conversion
    include InterpreterOptions

    def initialize(type_table)
      type_table = type_table
      @environment = Environment.new(type_table)
      @type_resolver = DynamicTypeResolver.new(type_table, self)

      @sub_interpreters = [
        DeclarationInterpreter.new(self),
        ParameterDefinitionInterpreter.new(self),
        FunctionInterpreter.new(self),
        SwitchStatementInterpreter.new(self),
        StatementInterpreter.new(self),
        ExpressionInterpreter.new(self)
      ]

      @options_stack = []

      # NOTE: Active (executing) function may be nested if the nested
      #       function-definition of the GCC extension is used.
      @active_function_stack = []
    end

    attr_reader :environment
    attr_reader :type_resolver

    extend Pluggable

    def self.def_plugin_and_notifier(event_name, *arg_names)
      class_eval <<-EOS
        def_plugin :on_#{event_name}
        def notify_#{event_name}(#{arg_names.join(",")})
          unless quiet?
            on_#{event_name}.invoke(#{arg_names.join(",")})
          end
        end
      EOS
    end
    private_class_method :def_plugin_and_notifier

    # NOTE: Notified when the interpreter evaluates a variable-declaration.
    def_plugin_and_notifier :variable_declared,
                            :variable_declaration, :variable

    # NOTE: Notified when the interpreter evaluates a variable-definition.
    def_plugin_and_notifier :variable_defined, :variable_definition, :variable

    # NOTE: Notified when the interpreter evaluates an initializer of
    #       variable-definition.
    def_plugin_and_notifier :variable_initialized,
                            :variable_definition, :variable, :init_variable

    # NOTE: Notified when the interpreter evaluates a function-declaration.
    def_plugin_and_notifier :function_declared,
                            :function_declaration, :function

    # NOTE: Notified when the interpreter evaluates a function-definition.
    def_plugin_and_notifier :function_defined, :function_definition, :function

    # NOTE: Notified when the interpreter evaluates a struct-type-declaration.
    def_plugin_and_notifier :struct_declared, :struct_type_declaration

    # NOTE: Notified when the interpreter evaluates a union-type-declaration.
    def_plugin_and_notifier :union_declared, :union_type_declaration

    # NOTE: Notified when the interpreter evaluates a enum-type-declaration.
    def_plugin_and_notifier :enum_declared, :enum_type_declaration

    # NOTE: Notified when the interpreter evaluates a typedef-declaration.
    def_plugin_and_notifier :typedef_declared, :typedef_declaration

    # NOTE: Notified when the interpreter starts execution of a
    #       function-definition.
    def_plugin_and_notifier :function_started, :function_definition, :function

    # NOTE: Notified when the interpreter ends execution of a
    #       function-definition.
    def_plugin_and_notifier :function_ended, :function_definition, :function

    # NOTE: Notified when the interpreter evaluates a parameter-definition at
    #       beginning of execution of a function-definition.
    def_plugin_and_notifier :parameter_defined,
                            :parameter_definition, :variable

    # NOTE: Notified when the interpreter evaluates an expression which results
    #       a named variable.
    def_plugin_and_notifier :variable_referred, :expression, :variable

    # NOTE: Notified when the interpreter evaluates an expression which results
    #       a constant temporary variable.
    def_plugin_and_notifier :constant_referred, :constant_specifier, :variable

    # NOTE: Notified when the interpreter refer to a value of a variable while
    #       evaluating an expression.
    def_plugin_and_notifier :variable_value_referred, :expression, :variable

    # NOTE: Notified when the interpreter overwrite a value of a variable while
    #       evaluating an expression.
    def_plugin_and_notifier :variable_value_updated, :expression, :variable

    # NOTE: Notified when the interpreter refer to a function object while
    #       evaluating an expression.
    def_plugin_and_notifier :function_referred, :expression, :function

    # NOTE: Notified when the interpreter evaluates a sizeof-expression.
    def_plugin_and_notifier :sizeof_expr_evaled,
                            :expression, :operand_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a sizeof-type-expression.
    def_plugin_and_notifier :sizeof_type_expr_evaled,
                            :expression, :type, :result_variable

    # NOTE: Notified when the interpreter evaluates a cast-expression.
    def_plugin_and_notifier :explicit_conv_performed,
                            :expression, :original_variable, :result_variable

    # NOTE: Notified when the interpreter performs an implicit type conversion
    #       while evaluating an expression.
    def_plugin_and_notifier :implicit_conv_performed,
                            :initializer_or_expression, :original_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates an
    #       array-subscript-expression.
    def_plugin_and_notifier :array_subscript_expr_evaled,
                            :array_subscript_expression,
                            :array_or_pointer_variable, :subscript_variable,
                            :array_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a function-call-expression.
    def_plugin_and_notifier :function_call_expr_evaled,
                            :function_call_expression, :function,
                            :arg_variables, :result_variable

    # NOTE: Notified when the interpreter evaluates an
    #       unary-arithmetic-expression.
    def_plugin_and_notifier :unary_arithmetic_expr_evaled,
                            :unary_arithmetic_expression, :operand_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a
    #       multiplicative-expression.
    def_plugin_and_notifier :multiplicative_expr_evaled,
                            :multiplicative_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates an additive-expression.
    def_plugin_and_notifier :additive_expr_evaled,
                            :additive_expression, :lhs_variable, :rhs_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a shift-expression.
    def_plugin_and_notifier :shift_expr_evaled,
                            :shift_expression, :lhs_variable, :rhs_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a relational-expression.
    def_plugin_and_notifier :relational_expr_evaled,
                            :relational_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates an equality-expression.
    def_plugin_and_notifier :equality_expr_evaled,
                            :equality_expression, :lhs_variable, :rhs_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a bitwise and-expression.
    def_plugin_and_notifier :and_expr_evaled,
                            :and_expression, :lhs_variable, :rhs_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates an exclusive-or-expression.
    def_plugin_and_notifier :exclusive_or_expr_evaled,
                            :exclusive_or_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a bitwise
    #       inclusive-or-expression.
    def_plugin_and_notifier :inclusive_or_expr_evaled,
                            :inclusive_or_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a logical-and-expression.
    def_plugin_and_notifier :logical_and_expr_evaled,
                            :logical_and_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a logical-or-expression.
    def_plugin_and_notifier :logical_or_expr_evaled,
                            :logical_or_expression, :lhs_variable,
                            :rhs_variable, :result_variable

    # NOTE: Notified when the interpreter evaluates a conditional-expression.
    def_plugin_and_notifier :conditional_expr_evaled,
                            :conditional_expression, :controlling_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates an address-expression.
    def_plugin_and_notifier :address_expr_evaled,
                            :address_expression, :object, :pointer_variable

    # NOTE: Notified when the interpreter evaluates an indirection-expression.
    def_plugin_and_notifier :indirection_expr_evaled,
                            :indirection_expression, :pointer_variable,
                            :dereferenced_variable

    # NOTE: Notified when the interpreter evaluates a
    #       member-access-by-value-expression or a
    #       member-access-by-pointer-expression.
    def_plugin_and_notifier :member_access_expr_evaled,
                            :member_access_expression, :outer_variable,
                            :member_variable

    # NOTE: Notified when the interpreter evaluates a
    #       prefix-increment-expression.
    def_plugin_and_notifier :prefix_increment_expr_evaled,
                            :prefix_increment_expression, :operand_variable,
                            :original_value

    # NOTE: Notified when the interpreter evaluates a
    #       postfix-increment-expression.
    def_plugin_and_notifier :postfix_increment_expr_evaled,
                            :postfix_increment_expression, :operand_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a
    #       prefix-decrement-expression.
    def_plugin_and_notifier :prefix_decrement_expr_evaled,
                            :prefix_decrement_expression, :operand_variable,
                            :original_value

    # NOTE: Notified when the interpreter evaluates a
    #       postfix-decrement-expression.
    def_plugin_and_notifier :postfix_decrement_expr_evaled,
                            :postfix_decrement_expression, :operand_variable,
                            :result_variable

    # NOTE: Notified when the interpreter evaluates a
    #       simple-assignment-expression or a compound-assignment-expression.
    def_plugin_and_notifier :assignment_expr_evaled,
                            :assignment_expression, :lhs_variable,
                            :rhs_variable

    # NOTE: Notified when the interpreter starts execution of a
    #       switch-statement.
    def_plugin_and_notifier :switch_stmt_started, :switch_statement

    # NOTE: Notified when the interpreter ends execution of a switch-statement.
    def_plugin_and_notifier :switch_stmt_ended, :switch_statement

    # NOTE: Notified when the interpreter starts execution of a
    #       while-statement.
    def_plugin_and_notifier :while_stmt_started, :while_statement

    # NOTE: Notified when the interpreter ends execution of a while-statement.
    def_plugin_and_notifier :while_stmt_ended, :while_statement

    # NOTE: Notified when the interpreter starts execution of a do-statement.
    def_plugin_and_notifier :do_stmt_started, :do_statement

    # NOTE: Notified when the interpreter ends execution of a do-statement.
    def_plugin_and_notifier :do_stmt_ended, :do_statement

    # NOTE: Notified when the interpreter starts execution of a for-statement.
    def_plugin_and_notifier :for_stmt_started, :for_statement

    # NOTE: Notified when the interpreter ends execution of a for-statement.
    def_plugin_and_notifier :for_stmt_ended, :for_statement

    # NOTE: Notified when the interpreter starts execution of a
    #       c99-for-statement.
    def_plugin_and_notifier :c99_for_stmt_started, :c99_for_statement

    # NOTE: Notified when the interpreter ends execution of a
    #       c99-for-statement.
    def_plugin_and_notifier :c99_for_stmt_ended, :c99_for_statement

    # NOTE: Notified when the interpreter evaluates a goto-statement.
    def_plugin_and_notifier :goto_stmt_evaled, :goto_statement, :label_name

    # NOTE: Notified when the interpreter evaluates a return-statement.
    def_plugin_and_notifier :return_stmt_evaled,
                            :return_statement, :result_variable

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the if-statement.
    def_plugin_and_notifier :if_ctrlexpr_evaled, :if_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the if-else-statement.
    def_plugin_and_notifier :if_else_ctrlexpr_evaled,
                            :if_else_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the switch-statement.
    def_plugin_and_notifier :switch_ctrlexpr_evaled,
                            :switch_statement, :ctrlexpr_result

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the case-labeled-statement.
    def_plugin_and_notifier :case_ctrlexpr_evaled,
                            :case_labeled_statement, :ctrlexpr_result

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the while-statement.
    def_plugin_and_notifier :while_ctrlexpr_evaled,
                            :while_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the do-statement.
    def_plugin_and_notifier :do_ctrlexpr_evaled, :do_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the for-statement.
    def_plugin_and_notifier :for_ctrlexpr_evaled,
                            :for_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter evaluates a controlling expression of
    #       the c99-for-statement.
    def_plugin_and_notifier :c99_for_ctrlexpr_evaled,
                            :c99_for_statement, :ctrlexpr_value

    # NOTE: Notified when the interpreter defines a generic-label.
    def_plugin_and_notifier :label_defined, :generic_labeled_statement

    # NOTE: Notified when the interpreter starts execution of a
    #       compound-statement.
    def_plugin_and_notifier :block_started, :compound_statement

    # NOTE: Notified when the interpreter ends execution of a
    #       compound-statement.
    def_plugin_and_notifier :block_ended, :compound_statement

    # NOTE: Notified when the interpreter forks execution paths of a
    #       function-definition.
    def_plugin_and_notifier :branch_started, :branch

    # NOTE: Notified when the interpreter joins execution paths of a
    #       function-definition.
    def_plugin_and_notifier :branch_ended, :branch

    # NOTE: Notified when the interpreter starts execution of a
    #       translation-unit.
    def_plugin_and_notifier :translation_unit_started, :translation_unit

    # NOTE: Notified when the interpreter ends execution of a translation-unit.
    def_plugin_and_notifier :translation_unit_ended, :translation_unit

    # NOTE: Notified when control reaches to a sequence-point.
    def_plugin_and_notifier :sequence_point_reached, :sequence_point

    def execute(node, *options)
      @options_stack.push(current_options + options)
      node.accept(interpreter_for(node))
    ensure
      @options_stack.pop
    end

    def object_to_variable(object)
      case
      when object.function?
        temporary_variable(pointer_type(object.type), pointer_value_of(object))
      when object.type.array?
        temporary_variable(pointer_type(object.type.base_type),
                           pointer_value_of(object))
      else
        object
      end
    end

    def value_of(object)
      if object.type.array? || object.type.function?
        pointer_value_of(object)
      else
        object.value.to_single_value
      end
    end

    def pointer_value_of(object)
      ScalarValue.of(object.binding.memory.address)
    end

    def quiet?
      current_options.include?(QUIET)
    end

    def _quiet=(quiet)
      # NOTE: This method is called only from ControllingExpression.
      if quiet
        current_options.add(QUIET)
      else
        current_options.delete(QUIET)
      end
    end

    def _active_function
      # NOTE: This method is called only from
      #       StatementInterpreter#visit_return_statement.
      # NOTE: To convert returning object, StatementInterpreter must have
      #       knowledge about conversion destination type.
      @active_function_stack.last
    end

    def _enter_function(function_definition)
      # NOTE: This method is called only from FunctionInterpreter.
      @active_function_stack.push(function_definition)
    end

    def _leave_function(function_definition)
      # NOTE: This method is called only from FunctionInterpreter.
      @active_function_stack.pop
    end

    private
    def interpreter_for(node)
      @sub_interpreters.find do |interp|
        node.kind_of?(interp.target_node_class)
      end
    end

    def current_options
      @options_stack.last || Set.new
    end

    def interpreter
      # NOTE: This method is of the requirement for including
      #       InterpreterMediator.
      self
    end
  end

  class SubInterpreter < SyntaxTreeVisitor
    include InterpreterMediator
    include NotifierMediator
    include Conversion
    include MonitorUtil
    include InterpreterOptions

    def initialize(owner, target_node_class)
      @owner = owner
      @target_node_class = target_node_class
    end

    attr_reader :target_node_class

    private
    def interpreter
      @owner
    end
  end

  class DeclarationInterpreter < SubInterpreter
    def initialize(owner)
      super(owner, Declaration)
    end

    def visit_function_declaration(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      node.type.declarations.each do |decl|
        decl.mark_as_referred_by(node.identifier)
      end

      function = declare_function(node)

      notify_function_declared(node, function)

      evaluate_sequence_point(node.init_declarator.declarator)
    end

    def visit_variable_declaration(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      node.type.declarations.each do |decl|
        decl.mark_as_referred_by(node.identifier)
      end

      variable = declare_variable(node)
      notify_variable_declared(node, variable)

      evaluate_sequence_point(node.declarator)
    end

    def visit_variable_definition(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      node.type.declarations.each do |decl|
        decl.mark_as_referred_by(node.identifier)
      end

      if initializer = node.initializer
        init_var, converted = evaluate_initializer(initializer, node.type)
        variable = define_variable(node, converted.value.to_defined_value)

        notify_variable_value_referred(node, init_var)
        notify_variable_defined(node, variable)
        notify_variable_initialized(node, variable, init_var)
      else
        notify_variable_defined(node, define_variable(node))
      end

      evaluate_sequence_point(node.init_declarator.declarator)
    end

    def visit_struct_type_declaration(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      notify_struct_declared(node)
    end

    def visit_union_type_declaration(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      notify_union_declared(node)
    end

    def visit_enum_type_declaration(node)
      checkpoint(node)

      if enumerators = node.enum_specifier.enumerators
        sequence = 0
        enumerators.each do |enumerator|
          if expression = enumerator.expression
            object = interpret(expression)
            if object.variable? && object.value.scalar?
              enumerator.value = object.value.unique_sample
            end
          end
          enumerator.value ||= sequence
          define_enumerator(enumerator)
          sequence = enumerator.value + 1
        end
      end

      notify_enum_declared(node)
    end

    def visit_typedef_declaration(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      node.type.real_type.declarations.each do |decl|
        decl.mark_as_referred_by(node.identifier)
      end

      notify_typedef_declared(node)

      evaluate_sequence_point(node.init_declarator.declarator)

      Analyzer.current.info("user type `#{node.identifier.value}' " +
                            "defined at #{node.location.to_s}.")
    end

    private
    def evaluate_initializer(initializer, variable_type)
      init_interp = InitializerInterpreter.new(interpreter)
      variable, converted = init_interp.execute(initializer, variable_type)

      # NOTE: An implicit conversion has been done by InitializerInterpreter.

      # NOTE: For the case of array variable definition with a
      #       string-literal-specifier as the initializer.
      if variable_type.array? && variable.type.pointer?
        unless array = pointee_of(variable) and array.type.array?
          array = temporary_variable(variable_type)
        end
        deduct_array_length_from_array_variable(variable_type, array)
        variable = converted = array
      end

      return variable, converted
    end

    def deduct_array_length_from_array_variable(variable_type, array_variable)
      variable_type.length ||= array_variable.type.length
    end

    def evaluate_sequence_point(full_declarator)
      if sequence_point = full_declarator.subsequent_sequence_point
        notify_sequence_point_reached(sequence_point)
      end
    end
  end

  class InitializerInterpreter
    include InterpreterMediator
    include NotifierMediator
    include Conversion
    include MonitorUtil

    def initialize(interpreter)
      @interpreter = interpreter
    end

    def execute(initializer, variable_type)
      checkpoint(initializer.location)

      case
      when expression = initializer.expression
        # NOTE: An implicit conversion is already notified in
        #       #evaluate_expression.
        return evaluate_expression(expression, variable_type)
      when initializers = initializer.initializers
        variable = evaluate_initializers(initializers, variable_type)

        if variable.type.same_as?(variable_type)
          converted = variable
        else
          converted =
            do_conversion(variable, variable_type) ||
            temporary_variable(variable_type)
          notify_implicit_conv_performed(initializer, variable, converted)
        end
      else
        variable = converted = temporary_variable(variable_type)
      end

      return variable, converted
    end

    private
    def evaluate_expression(expression, type)
      checkpoint(expression.location)

      object = interpret(expression)

      variable = object_to_variable(object)
      unless variable == object
        notify_implicit_conv_performed(expression, object, variable)
      end

      if variable.type.same_as?(type)
        converted = variable
      else
        converted = do_conversion(variable, type) || temporary_variable(type)
        notify_implicit_conv_performed(expression, variable, converted)
      end

      return variable, converted
    end

    def evaluate_initializers(initializers, variable_type)
      case
      when variable_type.array?
        deduct_array_length_from_initializers(variable_type, initializers)
        member_types = [variable_type.base_type] * variable_type.impl_length
      when variable_type.composite?
        member_types = variable_type.members.map { |member| member.type }
      else
        member_types = [variable_type]
      end

      values = member_types.zip(initializers).map { |type, initializer|
        if initializer
          checkpoint(initializer.location)
          case
          when expression = initializer.expression
            value_of(evaluate_expression(expression, type).last)
          when initializers = initializer.initializers
            value_of(evaluate_initializers(initializers, type))
          else
            type.undefined_value
          end
        else
          type.undefined_value
        end
      }

      case
      when variable_type.array?
        temporary_variable(variable_type, ArrayValue.new(values))
      when variable_type.composite?
        temporary_variable(variable_type, CompositeValue.new(values))
      else
        temporary_variable(variable_type, values.first)
      end
    end

    def deduct_array_length_from_initializers(variable_type, initializers)
      variable_type.length ||= initializers.size
    end

    def interpreter
      # NOTE: This is private attr_reader for InterpreterMediator.
      #       This attribute is read via a normal method to suppress
      #       `private attribute?' warning.
      @interpreter
    end
  end

  class ParameterDefinitionInterpreter < SubInterpreter
    def initialize(owner)
      super(owner, ParameterDefinition)
    end

    def visit_parameter_definition(node)
      checkpoint(node.location)

      resolve_unresolved_type(node)

      node.type.declarations.each do |decl|
        if identifier = node.identifier
          decl.mark_as_referred_by(identifier)
        else
          decl.mark_as_referred_by(node.head_token)
        end
      end

      if node.identifier
        variable = define_variable(node.to_variable_definition,
                                   node.type.parameter_value)
        notify_parameter_defined(node, variable)
      end
    end
  end

  class FunctionInterpreter < SubInterpreter
    def initialize(owner)
      super(owner, FunctionDefinition)
    end

    def visit_ansi_function_definition(node)
      interpret_function(node)
    end

    def visit_kandr_function_definition(node)
      interpret_function(node)
    end

    private
    def interpret_function(node)
      checkpoint(node.location)

      reset_environment
      resolve_unresolved_type(node)

      if function = function_named(node.identifier.value)
        function.declarations_and_definitions.each do |decl_or_def|
          decl_or_def.mark_as_referred_by(node.identifier)
        end
        function.declarations_and_definitions.push(node)
      else
        function = define_explicit_function(node)
      end

      notify_function_defined(node, function)

      begin
        interpreter._enter_function(node)
        scoped_eval do
          notify_function_started(node, function)
          notify_block_started(node.function_body)

          node.parameter_definitions.each { |pdef| interpret(pdef) }

          BreakEvent.catch do
            node.function_body.block_items.each { |item| interpret(item) }
          end

          notify_block_ended(node.function_body)
          notify_function_ended(node, function)
        end
      ensure
        interpreter._leave_function(node)
      end
    end
  end

  class StatementInterpreter < SubInterpreter
    include BranchOptions
    include BranchGroupOptions
    include SyntaxNodeCollector

    def initialize(owner)
      super(owner, Statement)
    end

    def visit_generic_labeled_statement(node)
      checkpoint(node.location)

      node.executed = true
      notify_label_defined(node)

      uninitialize_block_local_variables(node)
      interpret(node.statement)
    end

    def visit_case_labeled_statement(node)
      checkpoint(node.location)

      node.executed = true
      ctrlexpr = node.expression
      ctrlexpr_result = object_to_variable(interpret(ctrlexpr, QUIET))
      notify_case_ctrlexpr_evaled(node, ctrlexpr_result)

      interpret(node.statement)
    end

    def visit_default_labeled_statement(node)
      checkpoint(node.location)

      node.executed = true
      interpret(node.statement)
    end

    def visit_compound_statement(node)
      checkpoint(node.location)

      node.executed = true
      scoped_eval do
        begin
          notify_block_started(node)
          node.block_items.each { |block_item| interpret(block_item) }
        ensure
          notify_block_ended(node)
        end
      end
    end

    def visit_expression_statement(node)
      checkpoint(node.location)

      node.executed = true
      interpret(node.expression) if node.expression
    end

    def visit_if_statement(node)
      checkpoint(node.location)

      node.executed = true
      ctrlexpr = node.expression
      ctrlexpr_value = value_of(interpret(ctrlexpr, QUIET))
      notify_if_ctrlexpr_evaled(node, ctrlexpr_value)

      case
      when ctrlexpr_value.must_be_true?
        branched_eval(ctrlexpr, NARROWING, FINAL, COMPLETE) do
          interpret(node.statement)
        end
      when ctrlexpr_value.may_be_true?
        branched_eval(ctrlexpr, NARROWING, FINAL) do
          interpret(node.statement)
        end
      else
        # NOTE: To warn about the controlling expression and to end the current
        #       branch group of else-if sequence.
        branched_eval(nil, NARROWING, FINAL) do
          interpret(ctrlexpr)
        end
      end
    end

    def visit_if_else_statement(node)
      checkpoint(node.location)

      node.executed = true
      ctrlexpr = node.expression
      ctrlexpr_value = value_of(interpret(ctrlexpr, QUIET))
      notify_if_else_ctrlexpr_evaled(node, ctrlexpr_value)

      case
      when ctrlexpr_value.must_be_true?
        branched_eval(ctrlexpr, NARROWING, FINAL, COMPLETE) do
          interpret(node.then_statement)
        end
        return
      when ctrlexpr_value.may_be_true?
        branched_eval(ctrlexpr, NARROWING) do
          interpret(node.then_statement)
        end
      else
        # NOTE: To warn about the controlling expression.
        interpret(ctrlexpr)
      end

      case node.else_statement
      when IfStatement, IfElseStatement
        interpret(node.else_statement)
      else
        branched_eval(nil, NARROWING, FINAL, COMPLETE) do
          interpret(node.else_statement)
        end
      end
    end

    def visit_while_statement(node)
      checkpoint(node.location)

      node.executed = true
      notify_while_stmt_started(node)

      widen_varying_variable_value_domain(node)
      effective_ctrlexpr = deduct_effective_controlling_expression(node)

      ctrlexpr = node.expression
      ctrlexpr_value = value_of(interpret(ctrlexpr))
      notify_while_ctrlexpr_evaled(node, ctrlexpr_value)

      case
      when ctrlexpr_value.must_be_true?
        branched_eval(effective_ctrlexpr,
                      NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
          interpret(node.statement)
        end
      when ctrlexpr_value.may_be_true?
        branched_eval(effective_ctrlexpr,
                      NARROWING, FINAL, IMPLICIT_CONDITION) do
          interpret(node.statement)
        end
      end
    ensure
      notify_while_stmt_ended(node)
    end

    def visit_do_statement(node)
      checkpoint(node.location)

      node.executed = true
      notify_do_stmt_started(node)

      widen_varying_variable_value_domain(node)
      effective_ctrlexpr = deduct_effective_controlling_expression(node)

      branched_eval(effective_ctrlexpr,
                    NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
        interpret(node.statement)
      end

      ctrlexpr_value = value_of(interpret(node.expression))
      notify_do_ctrlexpr_evaled(node, ctrlexpr_value)
    ensure
      notify_do_stmt_ended(node)
    end

    def visit_for_statement(node)
      checkpoint(node.location)

      node.executed = true
      notify_for_stmt_started(node)

      node.initial_statement.accept(self)

      widen_varying_variable_value_domain(node)
      effective_ctrlexpr = deduct_effective_controlling_expression(node)

      node.condition_statement.executed = true
      if ctrlexpr = node.condition_statement.expression
        ctrlexpr_value = value_of(interpret(ctrlexpr))
        notify_for_ctrlexpr_evaled(node, ctrlexpr_value)

        case
        when ctrlexpr_value.must_be_true?
          branched_eval(effective_ctrlexpr,
                        NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
            interpret(node.body_statement)
            interpret(node.expression) if node.expression
            # NOTE: To avoid that value of the controlling variable is marked
            #       as updated at end of the for-statement.  Value of the
            #       controlling variable is referred by the controlling
            #       expression at the last iteration.
            # FIXME: This re-interpretation of the controlling expression may
            #        causes duplicative warning messages.
            interpret(ctrlexpr)
          end
        when ctrlexpr_value.may_be_true?
          branched_eval(effective_ctrlexpr,
                        NARROWING, FINAL, IMPLICIT_CONDITION) do
            interpret(node.body_statement)
            interpret(node.expression) if node.expression
            # NOTE: To avoid that value of the controlling variable is marked
            #       as updated at end of the for-statement.  Value of the
            #       controlling variable is referred by the controlling
            #       expression at the last iteration.
            # FIXME: This re-interpretation of the controlling expression may
            #        causes duplicative warning messages.
            interpret(ctrlexpr)
          end
        end
      else
        branched_eval(effective_ctrlexpr,
                      NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
          interpret(node.body_statement)
          interpret(node.expression) if node.expression
        end
      end
    ensure
      notify_for_stmt_ended(node)
    end

    def visit_c99_for_statement(node)
      checkpoint(node.location)

      scoped_eval do
        interpret(node.declaration)

        node.executed = true
        notify_c99_for_stmt_started(node)

        widen_varying_variable_value_domain(node)
        effective_ctrlexpr = deduct_effective_controlling_expression(node)

        node.condition_statement.executed = true
        if ctrlexpr = node.condition_statement.expression
          ctrlexpr_value = value_of(interpret(ctrlexpr))
          notify_c99_for_ctrlexpr_evaled(node, ctrlexpr_value)

          case
          when ctrlexpr_value.must_be_true?
            branched_eval(effective_ctrlexpr,
                          NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
              interpret(node.body_statement)
              interpret(node.expression) if node.expression
              # NOTE: To avoid that value of the controlling variable is marked
              #       as updated at end of the c99-for-statement.  Value of the
              #       controlling variable is referred by the controlling
              #       expression at the last iteration.
              # FIXME: This re-interpretation of the controlling expression may
              #        causes duplicative warning messages.
              interpret(ctrlexpr)
            end
          when ctrlexpr_value.may_be_true?
            branched_eval(effective_ctrlexpr,
                          NARROWING, FINAL, IMPLICIT_CONDITION) do
              interpret(node.body_statement)
              interpret(node.expression) if node.expression
              # NOTE: To avoid that value of the controlling variable is marked
              #       as updated at end of the c99-for-statement.  Value of the
              #       controlling variable is referred by the controlling
              #       expression at the last iteration.
              # FIXME: This re-interpretation of the controlling expression may
              #        causes duplicative warning messages.
              interpret(ctrlexpr)
            end
          end
        else
          branched_eval(effective_ctrlexpr,
                        NARROWING, FINAL, IMPLICIT_CONDITION, COMPLETE) do
            interpret(node.body_statement)
            interpret(node.expression) if node.expression
          end
        end
      end
    ensure
      notify_c99_for_stmt_ended(node)
    end

    def visit_goto_statement(node)
      checkpoint(node.location)

      # TODO: Must implement goto semantics.
      node.executed = true
      notify_goto_stmt_evaled(node, node.identifier.value)
    end

    def visit_continue_statement(node)
      checkpoint(node.location)

      node.executed = true
      BreakEvent.of_continue.throw
    end

    def visit_break_statement(node)
      checkpoint(node.location)

      node.executed = true
      BreakEvent.of_break.throw
    end

    def visit_return_statement(node)
      checkpoint(node.location)

      node.executed = true

      unless node.expression
        notify_return_stmt_evaled(node, nil)
        BreakEvent.of_return.throw
      end

      object = interpret(node.expression)
      variable = object_to_variable(object)
      unless variable == object
        notify_implicit_conv_performed(node.expression, object, variable)
      end

      notify_variable_value_referred(node.expression, variable)

      if active_function = interpreter._active_function and
          return_type = active_function.type.return_type
        if variable.type.same_as?(return_type)
          converted = variable
        else
          converted =
            do_conversion(variable, return_type) ||
            temporary_variable(return_type)
          notify_implicit_conv_performed(node.expression, variable, converted)
        end
      else
        converted = variable
      end

      notify_return_stmt_evaled(node, variable)
      BreakEvent.of_return.throw
    end

    private
    def uninitialize_block_local_variables(generic_labeled_statement)
      related_goto_statements = generic_labeled_statement.referrers
      return if related_goto_statements.empty?

      local_variables.each do |variable|
        variable_definition = variable.declarations_and_definitions.first

        anterior_gotos = related_goto_statements.select { |goto|
          goto.location.line_no < variable_definition.location.line_no
        }

        unless anterior_gotos.empty?
          variable.value.enter_versioning_group
          variable.value.begin_versioning
          variable.uninitialize!
          variable.value.end_versioning
          variable.value.leave_versioning_group(true)
        end
      end
    end

    def widen_varying_variable_value_domain(iteration_statement)
      varying_vars = {}
      iteration_statement.varying_variable_names.each do |name|
        if variable = variable_named(name)
          varying_vars[variable] = variable.value.dup
          variable.widen_value_domain!(:==, variable.type.arbitrary_value)
        end
      end

      varying_vars.each do |variable, orig_value|
        case deduct_variable_varying_path(variable, iteration_statement)
        when :increase
          variable.narrow_value_domain!(:>=, orig_value)
        when :decrease
          variable.narrow_value_domain!(:<=, orig_value)
        end
      end
    end

    def deduct_variable_varying_path(variable, iteration_statement)
      histogram = iteration_statement.varying_expressions.map { |expr|
        case expr
        when SimpleAssignmentExpression
          deduct_ctrl_var_path_by_simple_assignment_expr(variable, expr)
        when CompoundAssignmentExpression
          deduct_ctrl_var_path_by_compound_assignment_expr(variable, expr)
        when PrefixIncrementExpression, PostfixIncrementExpression
          expr.operand.identifier.value == variable.name ? :increase : nil
        when PrefixDecrementExpression, PostfixDecrementExpression
          expr.operand.identifier.value == variable.name ? :decrease : nil
        else
          nil
        end
      }.compact.each_with_object(Hash.new(0)) { |dir, hash| hash[dir] += 1 }

      if histogram.empty?
        nil
      else
        histogram[:decrease] <= histogram[:increase] ? :increase : :decrease
      end
    end

    def deduct_ctrl_var_path_by_simple_assignment_expr(variable, expr)
      return nil unless expr.lhs_operand.identifier.value == variable.name

      additive_exprs = collect_additive_expressions(expr.rhs_operand)
      histogram = additive_exprs.map { |additive_expr|
        if additive_expr.lhs_operand.kind_of?(ObjectSpecifier)
          lhs_name = additive_expr.lhs_operand.identifier.value
        end

        if additive_expr.rhs_operand.kind_of?(ObjectSpecifier)
          rhs_name = additive_expr.rhs_operand.identifier.value
        end

        unless lhs_name == variable.name || rhs_name == variable.name
          next nil
        end

        case additive_expr.operator.type
        when "+"
          :increase
        when "-"
          :decrease
        else
          nil
        end
      }.compact.each_with_object(Hash.new(0)) { |dir, hash| hash[dir] += 1 }

      if histogram.empty?
        nil
      else
        histogram[:decrease] <= histogram[:increase] ? :increase : :decrease
      end
    end

    def deduct_ctrl_var_path_by_compound_assignment_expr(variable, expr)
      return nil unless expr.lhs_operand.identifier.value == variable.name

      case expr.operator.type
      when "+="
        :increase
      when "-="
        :decrease
      else
        nil
      end
    end

    def deduct_effective_controlling_expression(iteration_statement)
      if effective_ctrlexpr = iteration_statement.deduct_controlling_expression
        # TODO: If the controlling expression is `i == 2 || i == 5', this
        #       expression should not be used.
        if effective_ctrlexpr.kind_of?(EqualityExpression) &&
            effective_ctrlexpr.operator.type == "=="
          effective_ctrlexpr = nil
        end
      end
      effective_ctrlexpr
    end
  end

  class SwitchStatementInterpreter < SubInterpreter
    include BranchOptions
    include BranchGroupOptions

    def initialize(owner)
      super(owner, SwitchStatement)
    end

    def visit_switch_statement(node)
      checkpoint(node.location)

      node.executed = true
      notify_switch_stmt_started(node)

      ctrlexpr = node.expression
      ctrlexpr_result = object_to_variable(interpret(ctrlexpr))
      notify_switch_ctrlexpr_evaled(node, ctrlexpr_result)

      execute_switch_body(ctrlexpr_result, node.statement)

      notify_switch_stmt_ended(node)
    end

    private
    def execute_switch_body(variable, node)
      checkpoint(node.location)

      node.executed = true
      scoped_eval do
        begin
          notify_block_started(node)
          execute_switch_branches(variable, node.block_items)
        ensure
          notify_block_ended(node)
        end
      end
    end

    def execute_switch_branches(variable, block_items)
      base_options = [SMOTHER_BREAK, IMPLICIT_CONDITION, NARROWING]
      base_options.push(COMPLETE) if complete?(block_items)

      index = 0
      while block_item = block_items[index]
        while block_item.kind_of?(GenericLabeledStatement)
          block_item.executed = true
          notify_label_defined(block_item)
          block_item = block_item.statement
        end

        case block_item
        when CaseLabeledStatement, DefaultLabeledStatement
          options = base_options.dup
          options.push(FINAL) if final_branch?(block_items, index)
          index = execute_branch(block_item, block_items, index + 1, options)
          break unless index
        else
          interpret(block_item)
          index += 1
        end
      end
    end

    def execute_branch(labeled_statement, block_items, index, branch_options)
      ctrlexpr = labeled_statement.normalized_expression
      ctrlexpr_value = value_of(interpret(ctrlexpr, QUIET))

      case
      when ctrlexpr_value.must_be_true?
        branch_options.push(FINAL, COMPLETE)
      when ctrlexpr_value.must_be_false?
        # NOTE: To end the current branch group of switch-statement if this
        #       case-clause is the final one.
        branched_eval(ctrlexpr, *branch_options) {}
        return seek_next_branch(block_items, index)
      end

      branched_eval(ctrlexpr, *branch_options) do |branch|
        interpret(labeled_statement)
        while block_item = block_items[index]
          while block_item.kind_of?(GenericLabeledStatement)
            block_item.executed = true
            notify_label_defined(block_item)
            block_item = block_item.statement
          end

          case block_item
          when CaseLabeledStatement, DefaultLabeledStatement
            # NOTE: Fall through!
            prepare_fall_through(branch, branch_options, block_item)
            branch_options.push(FINAL) if final_branch?(block_items, index)
            branch.add_options(*branch_options)
          end
          interpret(block_item)
          index += 1
        end
        # NOTE: To simulate implicit breaking of the last case-clause.
        BreakEvent.of_break.throw
      end

      branch_options.include?(FINAL) ?
        nil : seek_next_branch(block_items, index)
    end

    def prepare_fall_through(branch, branch_options, labeled_statement)
      value_domain_manip = nil

      branch.restart_versioning do
        ctrlexpr = labeled_statement.normalized_expression
        ctrlexpr_value = value_of(interpret(ctrlexpr, QUIET))

        case
        when ctrlexpr_value.must_be_true?
          branch_options.push(FINAL, COMPLETE)
        when ctrlexpr_value.must_be_false?
          return
        end

        value_domain_manip =
          branch.controlling_expression.ensure_true_by_widening(ctrlexpr)
      end

      value_domain_manip.commit!
    end

    def final_branch?(block_items, index)
      index += 1
      while block_item = block_items[index]
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end

        case block_item
        when CaseLabeledStatement, DefaultLabeledStatement
          return false
        else
          index += 1
        end
      end
      true
    end

    def complete?(block_items)
      index = 0
      while block_item = block_items[index]
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end

        case block_item
        when DefaultLabeledStatement
          return true
        else
          index += 1
        end
      end
      false
    end

    def seek_next_branch(block_items, index)
      while block_item = block_items[index]
        while block_item.kind_of?(GenericLabeledStatement)
          notify_label_defined(block_item)
          block_item = block_item.statement
        end

        case block_item
        when CaseLabeledStatement, DefaultLabeledStatement
          break
        else
          index += 1
        end
      end
      index
    end
  end

  class ExpressionInterpreter < SubInterpreter
    include ExpressionEvaluator
    include BranchOptions
    include BranchGroupOptions

    def initialize(owner)
      super(owner, Expression)
    end

    def self.def_eval_with_sequence_point(method_name)
      class_eval <<-EOS
        def #{method_name}(node)
          super
        ensure
          if sequence_point = node.subsequent_sequence_point
            notify_sequence_point_reached(sequence_point)
          end
        end
      EOS
    end
    private_class_method :def_eval_with_sequence_point

    def_eval_with_sequence_point :visit_error_expression
    def_eval_with_sequence_point :visit_object_specifier
    def_eval_with_sequence_point :visit_constant_specifier
    def_eval_with_sequence_point :visit_string_literal_specifier
    def_eval_with_sequence_point :visit_null_constant_specifier
    def_eval_with_sequence_point :visit_grouped_expression
    def_eval_with_sequence_point :visit_array_subscript_expression
    def_eval_with_sequence_point :visit_function_call_expression
    def_eval_with_sequence_point :visit_member_access_by_value_expression
    def_eval_with_sequence_point :visit_member_access_by_pointer_expression
    def_eval_with_sequence_point :visit_bit_access_by_value_expression
    def_eval_with_sequence_point :visit_bit_access_by_pointer_expression
    def_eval_with_sequence_point :visit_postfix_increment_expression
    def_eval_with_sequence_point :visit_postfix_decrement_expression
    def_eval_with_sequence_point :visit_compound_literal_expression
    def_eval_with_sequence_point :visit_prefix_increment_expression
    def_eval_with_sequence_point :visit_prefix_decrement_expression
    def_eval_with_sequence_point :visit_address_expression
    def_eval_with_sequence_point :visit_indirection_expression
    def_eval_with_sequence_point :visit_unary_arithmetic_expression
    def_eval_with_sequence_point :visit_sizeof_expression
    def_eval_with_sequence_point :visit_sizeof_type_expression
    def_eval_with_sequence_point :visit_alignof_expression
    def_eval_with_sequence_point :visit_alignof_type_expression
    def_eval_with_sequence_point :visit_cast_expression
    def_eval_with_sequence_point :visit_multiplicative_expression
    def_eval_with_sequence_point :visit_additive_expression
    def_eval_with_sequence_point :visit_shift_expression
    def_eval_with_sequence_point :visit_relational_expression
    def_eval_with_sequence_point :visit_equality_expression
    def_eval_with_sequence_point :visit_and_expression
    def_eval_with_sequence_point :visit_exclusive_or_expression
    def_eval_with_sequence_point :visit_inclusive_or_expression
    def_eval_with_sequence_point :visit_logical_and_expression
    def_eval_with_sequence_point :visit_logical_or_expression
    def_eval_with_sequence_point :visit_simple_assignment_expression
    def_eval_with_sequence_point :visit_compound_assignment_expression
    def_eval_with_sequence_point :visit_comma_separated_expression

    def visit_conditional_expression(node)
      checkpoint(node.location)

      ctrlexpr = node.condition
      ctrlexpr_variable = interpret(ctrlexpr, QUIET)
      ctrlexpr_value = value_of(ctrlexpr_variable)

      then_var = nil
      if ctrlexpr_value.may_be_true?
        branched_eval(ctrlexpr, NARROWING) do
          then_var = object_to_variable(interpret(node.then_expression))
        end
      end

      else_var = nil
      if ctrlexpr_value.may_be_false?
        branched_eval(nil, NARROWING, FINAL, COMPLETE) do
          else_var = object_to_variable(interpret(node.else_expression))
        end
      else
        branched_eval(nil, NARROWING, FINAL, COMPLETE) {}
      end

      case
      when then_var && else_var
        result_val = then_var.value.single_value_unified_with(else_var.value)
        result = temporary_variable(then_var.type, result_val)
      when then_var
        result = then_var
      when else_var
        result = else_var
      else
        # FIXME: Nevertheless, the then-expression is not reachable, the branch
        #        execution check may fail in evaluation of the else branch.
        result = temporary_variable
      end

      notify_conditional_expr_evaled(node, ctrlexpr_variable, result)
      result
    ensure
      if sequence_point = node.subsequent_sequence_point
        notify_sequence_point_reached(sequence_point)
      end
    end
  end

end
end
