# C expression evaluator.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/c/mediator"
require "adlint/c/const"
require "adlint/c/conv"

module AdLint #:nodoc:
module C #:nodoc:

  module ExpressionUtil
    # NOTE: Host class of this module must include InterpreterMediator,
    #       NotifierMediator and Conversion.

    def execute_object_specifier(node)
      if variable = variable_named(node.identifier.value)
        variable.declarations_and_definitions.each do |decl_or_def|
          decl_or_def.mark_as_referred_by(node.identifier)
        end
        _notify_object_referred(node, variable)
        # NOTE: Array object will be converted into its start address by the
        #       outer expression.  So, it is correct to return an array object
        #       itself.
        return variable
      end

      if function = function_named(node.identifier.value)
        function.declarations_and_definitions.each do |decl_or_def|
          decl_or_def.mark_as_referred_by(node.identifier)
        end
        _notify_object_referred(node, function)
        return function
      end

      if enumerator = enumerator_named(node.identifier.value)
        enumerator.mark_as_referred_by(node.identifier)
        return temporary_variable(enumerator.type,
                                  ScalarValue.of(enumerator.value))
      end

      define_implicit_function(node.identifier.value)
    end

    def execute_array_subscript_expression(node, object, subscript)
      unless object.variable? && (object.type.array? || object.type.pointer?)
        return temporary_variable
      end

      array_or_pointer = object
      result_type = array_or_pointer.type.unqualify.base_type

      case
      when array_or_pointer.type.array?
        array = array_or_pointer
        # NOTE: An array-subscript-expression with an array object only refers
        #       the array object, never refer the value of the array object.
      when array_or_pointer.type.pointer?
        if pointee = pointee_of(array_or_pointer) and pointee.type.array?
          array = pointee
        end
        # NOTE: An array-subscript-expression with a pointer object do refers
        #       the value of the pointer object.
        _notify_variable_value_referred(node, array_or_pointer)
      end

      unless subscript.variable? && subscript.value.scalar? &&
          subscript.value.exist?
        return temporary_variable(result_type)
      end
      _notify_variable_value_referred(node, subscript)

      # NOTE: An array subscript converted to `int' implicitly.
      if subscript.type.same_as?(int_type)
        int_subscript = subscript
      else
        if int_subscript = do_conversion(subscript, int_type)
          notify_implicit_conv_performed(node.array_subscript,
                                         subscript, int_subscript)
        else
          return temporary_variable(result_type)
        end
      end

      # FIXME: Domain of the subscript may have multiple values.
      subscript_value = int_subscript.value.unique_sample
      if array and inner_variable = array.inner_variable_at(subscript_value)
        _notify_object_referred(node, inner_variable)
        if inner_variable.type.same_as?(result_type)
          result = inner_variable
        end
      end
      result ||= temporary_variable(result_type)

      notify_array_subscript_expr_evaled(node, array_or_pointer,
                                         int_subscript, array, result)
      result
    end

    def execute_function_call_expression(node, object, args)
      if object.function?
        function = object
      else
        if object.type.pointer? && object.type.unqualify.base_type.function?
          if pointee = pointee_of(object) and pointee.function?
            function = pointee
          else
            function_type = object.type.unqualify.base_type
            function = define_anonymous_function(function_type)
          end
        else
          return temporary_variable
        end
      end
      _notify_variable_value_referred(node, object)

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.2.2 Function calls
      #
      # Semantics
      #
      # 10 The order of evaluation of the function designator, the actual
      #    arguments, and subexpressions within the actual arguments is
      #    unspecified, but there is a sequence point before the actual call.
      notify_sequence_point_reached(SequencePoint.new(node)) unless args.empty?

      result = function.call(interpreter, node, args)
      unless function.builtin?
        arg_variables = args.map { |arg, expr| object_to_variable(arg) }
        notify_function_call_expr_evaled(node, function, arg_variables, result)
      end

      result
    end

    def execute_member_access_by_value_expression(node, object)
      unless object.variable? && object.type.composite?
        return temporary_variable
      end
      variable = object

      member_variable = variable.inner_variable_named(node.identifier.value)
      # NOTE: A member-access-by-value-expression only refers the composite
      #       object, never refer the value of the composite object.

      # NOTE: `member_variable' is nil when this expression represents the
      #       direct member access extension.
      notify_member_access_expr_evaled(node, variable, member_variable)

      if member_variable
        _notify_object_referred(node, member_variable)
        member_variable
      else
        temporary_variable
      end
    end

    def execute_member_access_by_pointer_expression(node, object)
      unqualified_type = object.type.unqualify

      unless object.variable? && unqualified_type.pointer? &&
          unqualified_type.base_type.composite?
        return temporary_variable
      end
      pointer = object

      if pointee = pointee_of(pointer)
        if pointee.type.array?
          if first_element = pointee.inner_variable_at(0)
            pointee = first_element
          else
            pointee = temporary_variable(object.type.unqualify.base_type)
          end
        end
      end
      # NOTE: A member-access-by-pointer-expression do refers the value of the
      #       pointer object.
      _notify_variable_value_referred(node, pointer)

      if pointee && pointee.type.composite?
        member_variable = pointee.inner_variable_named(node.identifier.value)
      else
        member = unqualified_type.base_type.member_named(node.identifier.value)
        member_variable = temporary_variable(member.type) if member
      end

      # NOTE: `member_variable' is nil when this expression represents the
      #       direct member access extension.
      notify_member_access_expr_evaled(node, pointer, member_variable)

      if member_variable
        _notify_object_referred(node, member_variable)
        member_variable
      else
        temporary_variable
      end
    end

    def execute_postfix_increment_expression(node, object)
      return temporary_variable unless object.variable? && object.type.scalar?
      variable = object

      result = temporary_variable(variable.type, variable.value.dup)

      # NOTE: Value of the variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, variable)

      if variable.value.scalar?
        variable.assign!(variable.value + ScalarValue.of(1))
        _notify_variable_value_updated(node, variable)
      end

      notify_postfix_increment_expr_evaled(node, variable, result)

      result
    end

    def execute_postfix_decrement_expression(node, object)
      return temporary_variable unless object.variable? && object.type.scalar?
      variable = object

      result = temporary_variable(variable.type, variable.value.dup)

      # NOTE: Value of the variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warnings detections.
      # _notify_variable_value_referred(node, variable)

      if variable.value.scalar?
        variable.assign!(variable.value - ScalarValue.of(1))
        _notify_variable_value_updated(node, variable)
      end

      notify_postfix_decrement_expr_evaled(node, variable, result)

      result
    end

    def execute_prefix_increment_expression(node, object)
      return temporary_variable unless object.variable? && object.type.scalar?
      variable = object

      original_value = variable.value.dup

      # NOTE: Value of the variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warnings detections.
      # _notify_variable_value_referred(node, variable)

      if variable.value.scalar?
        variable.assign!(variable.value + ScalarValue.of(1))
        _notify_variable_value_updated(node, variable)
      end

      notify_prefix_increment_expr_evaled(node, variable, original_value)

      variable
    end

    def execute_prefix_decrement_expression(node, object)
      return temporary_variable unless object.variable? && object.type.scalar?
      variable = object

      original_value = variable.value.dup

      # NOTE: Value of the variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, variable)

      if variable.value.scalar?
        variable.assign!(variable.value - ScalarValue.of(1))
        _notify_variable_value_updated(node, variable)
      end

      notify_prefix_decrement_expr_evaled(node, variable, original_value)

      variable
    end

    def execute_address_expression(node, object)
      # NOTE: An address-expression does not read the value of the object.  But
      #       value reference should be notified to emphasize global variable
      #       cross-references.
      _notify_variable_value_referred(node, object)

      pointer = temporary_variable(pointer_type(object.type),
                                   pointer_value_of(object))

      notify_address_expr_evaled(node, object, pointer)
      pointer
    end

    def execute_indirection_expression(node, object)
      return temporary_variable unless object.variable? && object.type.pointer?
      pointer = object

      unqualified_type = pointer.type.unqualify
      pointee = pointee_of(pointer)
      _notify_variable_value_referred(node, pointer)

      case
      when pointee
        _notify_object_referred(node, pointee)
        if pointee.type.array?
          if first_element = pointee.inner_variable_at(0)
            pointee = first_element
          else
            pointee = temporary_variable(unqualified_type.base_type)
          end
        end

        unless unqualified_type.base_type.same_as?(pointee.type)
          pointee =
            do_conversion(pointee, unqualified_type.base_type) ||
            temporary_variable(unqualified_type.base_type)
        end
      when unqualified_type.base_type.function?
        pointee = define_anonymous_function(unqualified_type.base_type)
      else
        pointee = temporary_variable(unqualified_type.base_type)
      end

      notify_indirection_expr_evaled(node, pointer, pointee)

      pointee
    end

    def execute_unary_arithmetic_expression(node, object)
      variable = object_to_variable(object)
      return temporary_variable unless variable.type.scalar?

      unless variable == object
        notify_implicit_conv_performed(node.operand, object, variable)
      end

      case node.operator.type
      when "+"
        result = temporary_variable(variable.type, +variable.value)
      when "-"
        result = temporary_variable(variable.type, -variable.value)
      when "~"
        result = temporary_variable(variable.type, ~variable.value)
      when "!"
        result = temporary_variable(int_type, !variable.value)
      else
        # NOTREACHED
      end
      _notify_variable_value_referred(node, variable)

      notify_unary_arithmetic_expr_evaled(node, variable, result)

      result
    end

    def execute_cast_expression(node, object)
      resolve_unresolved_type(node.type_name)

      variable = object_to_variable(object)
      converted =
        do_conversion(variable, node.type_name.type) ||
        temporary_variable(node.type_name.type)

      notify_explicit_conv_performed(node, variable, converted)

      converted
    end

    def execute_multiplicative_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      case node.operator.type
      when "*"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value * rhs_converted.value)
      when "/"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        # NOTE: "Div by 0" semantics is implemented in value-value arithmetic.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value / rhs_converted.value)
      when "%"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        # NOTE: "Div by 0" semantics is implemented in value-value arithmetic.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value % rhs_converted.value)
      else
        # NOTREACHED
      end
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_multiplicative_expr_evaled(node, lhs_variable, rhs_variable,
                                        result)
      result
    end

    def execute_additive_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      case node.operator.type
      when "+"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value + rhs_converted.value)
      when "-"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value - rhs_converted.value)
      else
        # NOTREACHED
      end
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_additive_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_shift_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.7 Bitwise shift operators
      #
      # 3 The integer promotions are performed on each of the operands.  The
      #   type of the result is that of the promoted left operand.  If the
      #   value of the right operand is negative or is greater than or equal to
      #   the width of the promoted left operand, the behavior is undefined.

      lhs_converted = do_integer_promotion(lhs_variable)
      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      rhs_converted = do_integer_promotion(rhs_variable)
      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      case node.operator.type
      when "<<"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value << rhs_converted.value)
      when ">>"
        # NOTE: Domain of the arithmetic result value will be restricted by
        #       min-max of the variable type.
        result = temporary_variable(lhs_converted.type,
                                    lhs_converted.value >> rhs_converted.value)
      else
        # NOTREACHED
      end
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_shift_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_relational_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      unless lhs_variable.type.scalar?
        return temporary_variable(int_type, ScalarValue.of_arbitrary)
      end

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      unless rhs_variable.type.scalar?
        return temporary_variable(int_type, ScalarValue.of_arbitrary)
      end

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      case node.operator.type
      when "<"
        result = temporary_variable(int_type,
                                    lhs_converted.value < rhs_converted.value)
      when ">"
        result = temporary_variable(int_type,
                                    lhs_converted.value > rhs_converted.value)
      when "<="
        result = temporary_variable(int_type,
                                    lhs_converted.value <= rhs_converted.value)
      when ">="
        result = temporary_variable(int_type,
                                    lhs_converted.value >= rhs_converted.value)
      else
        # NOTREACHED
        result = temporary_variable(int_type, ScalarValue.of_arbitrary)
      end
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_relational_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_equality_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      unless lhs_variable.type.scalar?
        return temporary_variable(int_type, ScalarValue.of_arbitrary)
      end

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      unless rhs_variable.type.scalar?
        return temporary_variable(int_type, ScalarValue.of_arbitrary)
      end

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      case node.operator.type
      when "=="
        result = temporary_variable(int_type,
                                    lhs_converted.value == rhs_converted.value)
      when "!="
        result = temporary_variable(int_type,
                                    lhs_converted.value != rhs_converted.value)
      else
        # NOTREACHED
        result = temporary_variable(int_type, ScalarValue.of_arbitrary)
      end
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_equality_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_and_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      result = temporary_variable(lhs_converted.type,
                                  lhs_converted.value & rhs_converted.value)
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_and_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_exclusive_or_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      result = temporary_variable(lhs_converted.type,
                                  lhs_converted.value ^ rhs_converted.value)
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_exclusive_or_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_inclusive_or_expression(node, lhs_object, rhs_object)
      lhs_variable = object_to_variable(lhs_object)
      return temporary_variable unless lhs_variable.type.scalar?

      unless lhs_variable == lhs_object
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_object, lhs_variable)
      end

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      result = temporary_variable(lhs_converted.type,
                                  lhs_converted.value | rhs_converted.value)
      _notify_variable_value_referred(node, lhs_variable)
      _notify_variable_value_referred(node, rhs_variable)

      notify_inclusive_or_expr_evaled(node, lhs_variable, rhs_variable, result)

      result
    end

    def execute_simple_assignment_expression(node, lhs_object, rhs_object)
      return lhs_object unless lhs_object.variable?
      lhs_variable = lhs_object

      rhs_variable = object_to_variable(rhs_object)
      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      if rhs_variable.type.same_as?(lhs_variable.type)
        rhs_converted = rhs_variable
      else
        rhs_converted =
          do_conversion(rhs_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      # NOTE: Even if rhs_object is a NamedVariable, new value will be
      #       instantiated in value-coercing.
      #       So, value-aliasing never occurs.
      lhs_variable.assign!(rhs_converted.value.to_defined_value)
      _notify_variable_value_referred(node, rhs_variable)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, rhs_variable)

      lhs_variable
    end

    def execute_compound_assignment_expression(node, lhs_object, rhs_object)
      unless lhs_object.variable? && lhs_object.type.scalar?
        return temporary_variable
      end
      lhs_variable = lhs_object

      rhs_variable = object_to_variable(rhs_object)
      return temporary_variable unless rhs_variable.type.scalar?

      unless rhs_variable == rhs_object
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_object, rhs_variable)
      end

      case node.operator.type
      when "*="
        execute_mul_then_assign_expression(node, lhs_variable, rhs_variable)
      when "/="
        execute_div_then_assign_expression(node, lhs_variable, rhs_variable)
      when "%="
        execute_mod_then_assign_expression(node, lhs_variable, rhs_variable)
      when "+="
        execute_add_then_assign_expression(node, lhs_variable, rhs_variable)
      when "-="
        execute_sub_then_assign_expression(node, lhs_variable, rhs_variable)
      when "<<="
        execute_shl_then_assign_expression(node, lhs_variable, rhs_variable)
      when ">>="
        execute_shr_then_assign_expression(node, lhs_variable, rhs_variable)
      when "&="
        execute_and_then_assign_expression(node, lhs_variable, rhs_variable)
      when "^="
        execute_xor_then_assign_expression(node, lhs_variable, rhs_variable)
      when "|="
        execute_ior_then_assign_expression(node, lhs_variable, rhs_variable)
      end

      lhs_variable
    end

    def execute_mul_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value * rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_multiplicative_expr_evaled(node, lhs_variable, rhs_variable,
                                        result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_div_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        # NOTE: "Div by 0" semantics is implemented in value-value arithmetic.
        result_value = lhs_converted.value / rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_multiplicative_expr_evaled(node, lhs_variable, rhs_variable,
                                        result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_mod_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        # NOTE: "Div by 0" semantics is implemented in value-value arithmetic.
        result_value = lhs_converted.value % rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_multiplicative_expr_evaled(node, lhs_variable, rhs_variable,
                                        result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_add_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value + rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_additive_expr_evaled(node, lhs_variable, rhs_variable,
                                  result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_sub_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value - rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_additive_expr_evaled(node, lhs_variable, rhs_variable,
                                  result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_shl_then_assign_expression(node, lhs_variable, rhs_variable)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.7 Bitwise shift operators
      #
      # 3 The integer promotions are performed on each of the operands.  The
      #   type of the result is that of the promoted left operand.  If the
      #   value of the right operand is negative or is greater than or equal to
      #   the width of the promoted left operand, the behavior is undefined.

      lhs_converted = do_integer_promotion(lhs_variable)
      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      rhs_converted = do_integer_promotion(rhs_variable)
      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value << rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_shift_expr_evaled(node, lhs_converted, rhs_converted,
                               result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_shr_then_assign_expression(node, lhs_variable, rhs_variable)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.5.7 Bitwise shift operators
      #
      # 3 The integer promotions are performed on each of the operands.  The
      #   type of the result is that of the promoted left operand.  If the
      #   value of the right operand is negative or is greater than or equal to
      #   the width of the promoted left operand, the behavior is undefined.

      lhs_converted = do_integer_promotion(lhs_variable)
      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      rhs_converted = do_integer_promotion(rhs_variable)
      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value >> rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_shift_expr_evaled(node, lhs_converted, rhs_converted,
                               result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_and_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value & rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_and_expr_evaled(node, lhs_variable, rhs_variable, result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_xor_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value ^ rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_exclusive_or_expr_evaled(node, lhs_variable, rhs_variable,
                                      result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def execute_ior_then_assign_expression(node, lhs_variable, rhs_variable)
      lhs_converted, rhs_converted =
        do_usual_arithmetic_conversion(lhs_variable, rhs_variable)

      unless lhs_variable == lhs_converted
        notify_implicit_conv_performed(node.lhs_operand,
                                       lhs_variable, lhs_converted)
      end

      unless rhs_variable == rhs_converted
        notify_implicit_conv_performed(node.rhs_operand,
                                       rhs_variable, rhs_converted)
      end

      if lhs_converted.value.scalar? && rhs_converted.value.scalar?
        result_value = lhs_converted.value | rhs_converted.value
      else
        result_value = lhs_converted.value
      end

      # NOTE: Value of the lhs_variable is referred at this point.  But value
      #       reference should not be notified not to confuse sequence-point
      #       warning detections.
      # _notify_variable_value_referred(node, lhs_variable)

      _notify_variable_value_referred(node, rhs_variable)

      result_variable = temporary_variable(lhs_converted.type, result_value)

      notify_inclusive_or_expr_evaled(node, lhs_variable, rhs_variable,
                                      result_variable)

      if result_variable.type.same_as?(lhs_variable.type)
        result_converted = result_variable
      else
        result_converted =
          do_conversion(result_variable, lhs_variable.type) ||
          temporary_variable(lhs_variable.type)
        notify_implicit_conv_performed(node.lhs_operand,
                                       result_variable, result_converted)
      end

      # NOTE: Domain of the arithmetic result value will be restricted by
      #       min-max of the variable type.
      lhs_variable.assign!(result_converted.value)
      _notify_variable_value_updated(node, lhs_variable)

      notify_assignment_expr_evaled(node, lhs_variable, result_converted)
    end

    def _notify_object_referred(node, object)
      case object
      when Variable
        interpreter.notify_variable_referred(node, object)
      when Function
        interpreter.notify_function_referred(node, object)
      end
    end

    def _notify_variable_value_referred(node, object)
      if object.variable?
        interpreter.notify_variable_value_referred(node, object)
      end

      # NOTE: When a value of the inner-variable of array or composite object
      #       is referred, notification of the outer variable's value has
      #       already been done in sub expressions.
    end

    def _notify_variable_value_updated(node, object)
      if object.variable?
        interpreter.notify_variable_value_updated(node, object)
      end

      if object.kind_of?(InnerVariable)
        # NOTE: When a value of the inner-variable of array or composite object
        #       is updated, the outer variable's value should also be notified
        #       to be updated.
        _notify_variable_value_updated(node, object.owner)
      end
    end

    def _char_array_value(string_literal)
      chars = string_literal.chars.map { |ch| ScalarValue.of(ch.ord) }
      ArrayValue.new(chars + [ScalarValue.of("\0".ord)])
    end

    def _wchar_array_value(string_literal)
      chars = string_literal.chars.map { |ch| ScalarValue.of(ch.ord) }
      ArrayValue.new(chars + [ScalarValue.of("\0".ord)])
    end
  end

  module ExpressionEvaluator
    # NOTE: Host class of this module must include InterpreterMediator and
    #       MonitorUtil.

    include NotifierMediator
    include ConstantEvaluator
    include Conversion
    include ExpressionUtil

    def visit_error_expression(node)
      checkpoint(node.location)

      temporary_variable
    end

    def visit_object_specifier(node)
      checkpoint(node.location)

      execute_object_specifier(node)
    end

    def visit_constant_specifier(node)
      checkpoint(node.location)

      if variable = eval_constant(node)
        notify_constant_referred(node, variable)
        variable
      else
        temporary_variable
      end
    end

    def visit_string_literal_specifier(node)
      checkpoint(node.location)

      case node.literal.value
      when /\A"(.*)"\z/
        temporary_variable(array_type(char_type, $1.length + 1),
                           _char_array_value($1))
      when /\AL"(.*)"\z/i
        temporary_variable(array_type(wchar_type, $1.length + 1),
                           _wchar_array_value($1))
      else
        temporary_variable(array_type(char_type))
      end
    end

    def visit_null_constant_specifier(node)
      checkpoint(node.location)

      # TODO: NULL may not be 0 on some environments.
      #       Representation of NULL should be configurable?
      temporary_variable(pointer_type(void_type), ScalarValue.of(0))
    end

    def visit_grouped_expression(node)
      checkpoint(node.location)

      node.expression.accept(self)
    end

    def visit_array_subscript_expression(node)
      checkpoint(node.location)

      execute_array_subscript_expression(
        node, node.expression.accept(self), node.array_subscript.accept(self))
    end

    def visit_function_call_expression(node)
      checkpoint(node.location)

      object = node.expression.accept(self)
      args = node.argument_expressions.map { |expr| [expr.accept(self), expr] }

      execute_function_call_expression(node, object, args)
    end

    def visit_member_access_by_value_expression(node)
      checkpoint(node.location)

      object = node.expression.accept(self)

      execute_member_access_by_value_expression(node, object)
    end

    def visit_member_access_by_pointer_expression(node)
      checkpoint(node.location)

      object = node.expression.accept(self)

      execute_member_access_by_pointer_expression(node, object)
    end

    def visit_bit_access_by_value_expression(node)
      checkpoint(node.location)

      # TODO: Should support the GCC extension.
      temporary_variable
    end

    def visit_bit_access_by_pointer_expression(node)
      checkpoint(node.location)

      # TODO: Should support the GCC extension.
      temporary_variable
    end

    def visit_postfix_increment_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_postfix_increment_expression(node, object)
    end

    def visit_postfix_decrement_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_postfix_decrement_expression(node, object)
    end

    def visit_compound_literal_expression(node)
      checkpoint(node.location)

      # TODO: Should support C99 features.
      temporary_variable(node.type_name.type)
    end

    def visit_prefix_increment_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_prefix_increment_expression(node, object)
    end

    def visit_prefix_decrement_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_prefix_decrement_expression(node, object)
    end

    def visit_address_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_address_expression(node, object)
    end

    def visit_indirection_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_indirection_expression(node, object)
    end

    def visit_unary_arithmetic_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)

      execute_unary_arithmetic_expression(node, object)
    end

    def visit_sizeof_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)
      if object.variable?
        variable = object
        result = temporary_variable(
          int_type, ScalarValue.of(variable.type.aligned_byte_size))
        notify_sizeof_expr_evaled(node, variable, result)
      else
        result = temporary_variable(int_type)
      end

      result
    end

    def visit_sizeof_type_expression(node)
      checkpoint(node.location)

      resolve_unresolved_type(node.operand)

      result = temporary_variable(
        int_type, ScalarValue.of(node.operand.type.aligned_byte_size))
      notify_sizeof_type_expr_evaled(node, node.operand.type, result)

      result
    end

    def visit_alignof_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)
      if object.variable?
        variable = object
        temporary_variable(int_type,
                           ScalarValue.of(variable.type.byte_alignment))
      else
        temporary_variable(int_type)
      end
    end

    def visit_alignof_type_expression(node)
      checkpoint(node.location)

      resolve_unresolved_type(node.operand)

      temporary_variable(int_type,
                         ScalarValue.of(node.operand.type.aligned_byte_size))
    end

    def visit_cast_expression(node)
      checkpoint(node.location)

      object = node.operand.accept(self)
      execute_cast_expression(node, object)
    end

    def visit_multiplicative_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_multiplicative_expression(node, lhs_object, rhs_object)
    end

    def visit_additive_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_additive_expression(node, lhs_object, rhs_object)
    end

    def visit_shift_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_shift_expression(node, lhs_object, rhs_object)
    end

    def visit_relational_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_relational_expression(node, lhs_object, rhs_object)
    end

    def visit_equality_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_equality_expression(node, lhs_object, rhs_object)
    end

    def visit_and_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_and_expression(node, lhs_object, rhs_object)
    end

    def visit_exclusive_or_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_exclusive_or_expression(node, lhs_object, rhs_object)
    end

    def visit_inclusive_or_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_inclusive_or_expression(node, lhs_object, rhs_object)
    end

    def visit_logical_and_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      return temporary_variable(int_type) unless lhs_object.variable?

      lhs_variable = lhs_object
      if lhs_variable.value.scalar? && lhs_variable.value.must_be_false?
        # NOTE: Doing the short-circuit evaluation.
        return temporary_variable(int_type, ScalarValue.of_false)
      end

      rhs_object = node.rhs_operand.accept(self)
      return temporary_variable(int_type) unless rhs_object.variable?

      rhs_variable = rhs_object

      if lhs_variable.value.scalar? && rhs_variable.value.scalar?
        # NOTE: No usual-arithmetic-conversion.
        result = temporary_variable(
          int_type, lhs_variable.value.logical_and(rhs_variable.value))
      else
        result = temporary_variable(int_type)
      end

      notify_logical_and_expr_evaled(node, lhs_variable, rhs_variable, result)
      result
    end

    def visit_logical_or_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      return temporary_variable(int_type) unless lhs_object.variable?

      lhs_variable = lhs_object
      if lhs_variable.value.scalar? && lhs_variable.value.must_be_true?
        # NOTE: Doing the short-circuit evaluation.
        return temporary_variable(int_type, ScalarValue.of_true)
      end

      rhs_object = node.rhs_operand.accept(self)
      return temporary_variable(int_type) unless rhs_object.variable?

      rhs_variable = rhs_object

      if lhs_variable.value.scalar? && rhs_variable.value.scalar?
        # NOTE: No usual-arithmetic-conversion.
        result = temporary_variable(
          int_type, lhs_variable.value.logical_or(rhs_variable.value))
      else
        result = temporary_variable(int_type)
      end

      notify_logical_or_expr_evaled(node, lhs_variable, rhs_variable, result)
      result
    end

    def visit_simple_assignment_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_simple_assignment_expression(node, lhs_object, rhs_object)
    end

    def visit_compound_assignment_expression(node)
      checkpoint(node.location)

      lhs_object = node.lhs_operand.accept(self)
      rhs_object = node.rhs_operand.accept(self)

      execute_compound_assignment_expression(node, lhs_object, rhs_object)
    end

    def visit_comma_separated_expression(node)
      checkpoint(node.location)

      node.expressions.map { |expression| expression.accept(self) }.last
    end
  end

end
end
