/*--- command.h ----------------------------------------------------------------
Copyright (C) 2004, 2005, 2006 Sylvain Fourmanoit <syfou@users.sourceforge.net>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.   
------------------------------------------------------------------------------*/
/* Command interpreter related header
 */

/*----------------------------------------------------------------------------*/
#ifndef HAVE_COMMAND_H
#define HAVE_COMMAND_H

#ifndef HAVE_CONFIG_H
#error Autogenerated config.h should be used.
#endif

/*----------------------------------------------------------------------------*/
#include "config.h"			/* Autoconf header */

#ifdef HAVE_STDIO_H
#include <stdio.h>			/* Standard IO routines  */
#endif


#ifdef HAVE_STDARG_H			/* Variadic functions */
#include <stdarg.h>
#endif

#include <readline/readline.h>          /* GNU readline: greatly simplify
					   input capture */

#ifdef HAVE_READLINE_HISTORY_H		/* GNU history support */
#include <readline/history.h>
#endif

#ifdef HAVE_SYS_TIME_H			/* timeval struct, fd_set typedef */
#include <sys/time.h>
#endif

#ifdef HAVE_SYS_TYPES_H			/* select() and associated functions */
#include <sys/types.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_ICONV_H			/* charset conversion */
#include <iconv.h>
#endif

#ifndef X_DISPLAY_MISSING		/* mask defines */
#include <X11/X.h>                      
#endif

#include "types.h"			/* Various typedef */
#include "error.h"			/* Error wrapper */
#include "vector.h"                     /* Vector API */

#include "event.h"                      /* Used by command_replayer() */

/*----------------------------------------------------------------------------*/
typedef struct s_command {
  int    interactive;		/* Interactive input flag */
  int    recording;             /* Recording input flag */
  int    replay_pos;            /* Current replay position */
  int    replay_stop;		/* Stop replay position */
  int    replay_abort;          /* Specifically abord replaying */
  int    replay_abort_on_events;/* How to react to events */
  struct timeval replay_time;   /* Start time for replay */
  int    rank;			/* Command rank */
  int    ready;			/* Command is ready to be processed */
  int    message_out;           /* Message have been delivered */
  char * line;			/* Dynamic line */
  char * message;               /* Dynamic message */
#ifdef HAVE_ICONV_H
  char * from_page;		/* From codepage name */
  iconv_t cd;			/* Conversion descriptor */
#endif
} t_command;

typedef struct s_COMMAND {
  char * name;			/* User printable name of the function. */
  rl_compentry_func_t 
     * generator; 		/* Generator function */
  char *doc;			/* Documentation for this function.  */
} COMMAND;

#ifndef X_DISPLAY_MISSING
typedef struct s_XWINDOW_EVENT {
  char *name;			/* User printable name of the event */
  long mask;			/* Associated mask */
} X_WINDOW_EVENT;
#endif

/* Note: this typedef has to be in sync AT ALL TIMES with the
         contents of variable COMMANDS: this is why we use
         automatic generation performed by lister.sh */
typedef enum e_command {
#include "command_enum.h"
} command_enum;

/*----------------------------------------------------------------------------*/
extern const COMMAND          COMMANDS[];
extern const char           * DIRECTIONS[];
extern const char           * OPERATIONS[];
extern const char           * LOAD_ERRORS[];
extern const char           * RGBA_TABLES[];
extern const char           * WINDOW_MANAGER[];

#ifndef X_DISPLAY_MISSING
extern const X_WINDOW_EVENT   X_WINDOW_EVENTS[];
#endif

extern t_command command;

/*----------------------------------------------------------------------------*/
/* History convenience function: it exists in GNU readline >= 5.0,
   but it is a backport for older libraries */
#ifdef HAVE_READLINE_HISTORY_H
#ifndef HAVE_FREE_HISTORY_ENTRY
histdata_t free_history_entry (HIST_ENTRY*);
#endif
#endif

/*----------------------------------------------------------------------------*/
/* Command output formatting functions */
void command_ok(char *, ...);
void command_error(char *, ...);
int command_printf(const char *, ...);

/*----------------------------------------------------------------------------*/
/* String duplicator. Returns duplicated string or NULL
*/
char * dupstr(const char *);

/*----------------------------------------------------------------------------*/
/* String duplicator up to n characters: result always zero terminated 
   with n>0. Returns duplicated string or NULL
*/
char * dupnstr(const char *, int);

/*----------------------------------------------------------------------------*/
/* String duplicator that also take care of conversion to utf8 if applicable  */
char * dupstr_utf8(const char *);

/*----------------------------------------------------------------------------*/
/* Return actual size of history list*/
#ifdef HAVE_READLINE_HISTORY_H
int history_size(void);
#endif

/*----------------------------------------------------------------------------*/
/* Returns 1 if a string is blank, 0 otherwise.
 */
int blank_line(const char *);

/*----------------------------------------------------------------------------*/
/* Send the number of second passed since the beginning of current replay
 */
double command_gate_chronometer(void);

/*----------------------------------------------------------------------------*/
/* Initialisation routine of the command interpreter. 
   Return 1 in case of success, 0 otherwise.
*/
int command_interpreter_reset(void);

/*----------------------------------------------------------------------------*/
/* Command replayer function, indirect mode counterpart of 'command_interpreter'
   Fills the same structures than the latter.
*/
int command_replayer();

/*----------------------------------------------------------------------------*/
/* Command Interpreter function. If there is at least one character, pooling
   will continue until the input buffer is flushed.
   Return 1 if an error occured, 0 otherwise.
*/
int command_interpreter(void);

/*----------------------------------------------------------------------------*/
/* Command arguments separator function. Identify the command and return
   pointer to beginning of nth character, if found.
*/
char * command_subsplitter(char *, int);

/*----------------------------------------------------------------------------*/
/* Command arguments separator function. Identify the command and split
   its parameters into individual fields, sent back into a vector.
*/
vector * command_splitter(const char *, command_enum *);

/*----------------------------------------------------------------------------*/
/* A generic generator function, to be reused with many textual generator 
*/
char * generic_generator(const char *, int, char **, int);

/*----------------------------------------------------------------------------*/
/* A generic index generator function, to be reused with many textual generator 
*/
char * generic_index_generator(const char *, int, int);

/*----------------------------------------------------------------------------*/
/* A generic index generator function, to be reused with many textual generator,
   include a -1 index.
*/
char * generic_index_generator_with_null(const char *, int, int);

/*----------------------------------------------------------------------------*/
#endif
