/* -*- C++ -*- adept/taglist.cpp
   written by Peter Rockai <me@mornfall.net> */

#include <qlabel.h>
#include <qtimer.h>
#include <kdebug.h>
#include <qdragobject.h>
#include <qevent.h>
#include <klocale.h>

#include <apt-front/cache/component/tags.h>
#include <adept/taglist.h>
#include <adept/utils.h>

using namespace adept;

TagLabel::TagLabel( Tag t, TagList *l, QWidget *p, const char *n )
    : QHBox( p, n ), m_tag( t ), m_list( l )
{
    if ( t == Tag() ) {
        m_description = new QLabel( QString( " " ) + i18n( "[none]" ), this );
    } else {
        m_remove = new QLabel( this );
        m_remove->setPixmap( SmallIcon( u8( "cancel" ) ) );
        m_remove->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        m_description = new QLabel(  QString( " [" ) + t.fullname( "n/a" ) + "] "
                                     + t.shortDescription( "n/a" ), this );
    }
}

void TagLabel::mouseReleaseEvent( QMouseEvent *e ) {
    if ( e->button() == Qt::LeftButton &&
         dynamic_cast< QLabel * >( childAt( e->pos() ) ) == m_remove )
        m_list->setTags( m_list->tags() - m_tag );
}

TagList::TagList( QWidget *p, const char *n )
    : QVBox( p, n )
{
    m_name = new QLabel( this );
    m_tagBox = new QVBox( this );
    m_tagBox->setFrameShape( QFrame::Panel );
    m_tagBox->setFrameShadow( QFrame::Sunken );
    m_updateScheduled = false;
    setAcceptDrops( true );
    scheduleUpdateList();
    m_tagSpacer = new QSpacerItem( 0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding );
}

void TagList::setTags( Tag::Set t )
{
    m_tags = t;
    scheduleUpdateList();
    emit tagsChanged( m_tags );
}

void TagList::addTag( Tag t )
{
    if ( t == Tag() )
        return;
    m_tags += t;
    scheduleUpdateList();
    emit tagsChanged( m_tags );
}

void TagList::setName( QString n )
{
    m_name->setText( n );
}

void TagList::scheduleUpdateList()
{
    if (! m_updateScheduled) {
        kdDebug() << "TagList: scheduling update" << endl;
        QTimer::singleShot( 0, this, SLOT( updateList() ) );
        m_updateScheduled = true;
    }
}

void TagList::updateList()
{
    kdDebug() << "TagList (" + m_name->text() + "): updating list" << endl;
    clearList();
    if ( m_tags.empty() ) {
        appendLabel( new TagLabel( Tag(), this, m_tagBox ) );
    } else {
        for ( Tag::Set::iterator i = m_tags.begin(); i != m_tags.end(); ++i ) {
            appendLabel( new TagLabel( *i, this, m_tagBox ) );
        }
    }
    m_tagBox->layout()->addItem( m_tagSpacer );
    update();
    parentWidget()->adjustSize();
    m_updateScheduled = false;
}

void TagList::appendLabel( TagLabel *l )
{
    m_list.push_back( l );
    l->show();
}

void TagList::mouseMoveEvent( QMouseEvent *e ) {
    TagLabel *child = dynamic_cast< TagLabel * >( childAt( e->pos() )->parentWidget() );
    if ( !child )
        return;
    QDragObject *d = new QTextDrag( child->tag().fullname( "" ), this );
    d->dragCopy();
}

void TagList::dragEnterEvent( QDragEnterEvent *e ) {
    kdDebug() << "TagList::dragEnterEvent" << endl;
    e->accept( QTextDrag::canDecode( e ) );
}

void TagList::dropEvent( QDropEvent* e ) {
    QString tag;
    kdDebug() << "TagList: drop event" << endl;
    QTextDrag::decode( e, tag );
    try {
        addTag( cache::Global::get().tags().tagByName(
                    static_cast< const char * >( tag.local8Bit() ) ) );
    } catch (...) {} // not a tag, ignore
    scheduleUpdateList();
}

void TagList::clearList()
{
    for (List::iterator i = m_list.begin(); i != m_list.end(); ++i ) {
        delete *i;
    }
    m_list.clear();
    m_tagBox->layout()->removeItem( m_tagSpacer );
}
