/* ------------------------------------------------------------------------
 * $Id: SolidBuilder.cc,v 1.2 2001/08/02 14:22:11 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-08-01 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Math.hh"
#include "Solid/SolidBuilder.hh"

// -- Code Segment

void computeNormals(SolidMesh *s, bool smooth)
{
    std::vector<Vector3D> face_normals;
    
    // Clear the normal list
    s->normals.clear();

    // Compute face normals
    for (unsigned int i = 0; i < s->faces.size(); i++) {
	
	const TriangleFace &t = s->faces[i];
	
	// Compute two vectors in the triangle's plane
	Vector3D v1 = s->vertices[t.v[1]] - s->vertices[t.v[0]];
	Vector3D v2 = s->vertices[t.v[2]] - s->vertices[t.v[0]];

	// Now, compute the normal and normalize it
	Vector3D n = Vector3D::cross(v1, v2);
	n.normalize();
	
	// Store the normal
	face_normals.push_back(n);
    }
    
    // If we don't want a smooth surface, we use the face normals 
    if (smooth == false) {
	
	// Copy normal vector
	s->normals = face_normals;
	
	// Set the correct indices into the face list
	for (unsigned int i = 0; i < s->faces.size(); i++)
	    s->faces[i].n = TriangleIndex(i, i, i);
    }
    // Otherwise, we need to compute the vertex normals
    else {

	// Step through all faces and compute vertex normals
	for (unsigned int i = 0; i < s->faces.size(); i++) {

	    Vector3D vector_sum[3];
	    
	    // Retrieve the current face
	    TriangleIndex curr_face = s->faces[i].v;

	    // Search for faces that share at least two of the
	    // vertices in the current face.
	    for (unsigned int j = 0; j < s->faces.size(); j++) {

		// Retrieve the current face
		TriangleIndex new_face = s->faces[j].v;
		
		// See how many of vertices we share
		int sum = 0;
		bool adjacent[3] = { false, false, false };
		for (int k = 0; k < 3; k++)
		    for (int l = 0; l < 3; l++) 
			if (curr_face[k] == new_face[l]) {
			    sum++;
			    adjacent[k] = true;
			}
		
		// Make sure we have more than one vertex in the new face
		if (sum < 2) continue;

		// Add the face normal to the affected faces
		for (int k = 0; k < 3; k++)
		    if (adjacent[k]) vector_sum[k] += face_normals[j];
	    }
	    
	    // Now compute vertex normals for this face
	    for (int k = 0; k < 3; k++) {
		
		// Normalize the accumulated normal and store it
		vector_sum[k].normalize();
		s->normals.push_back(vector_sum[k]);
		
		// Set the correct index (at the end of the normal list)
		s->faces[i].n[k] = s->normals.size() - 1;
	    }
	}
    }
}

void createHorzCircle(std::vector<Vector3D> &vertices, float radius, 
		      int segments, float y)
{
    // Compute deltas
    float rad_delta = 2 * Math::pi / segments;
    
    // Create the bottom vertices
    for (int n = 0; n < segments; n++) {
	
	float radians = n * rad_delta;
	
	// Compute radial coordinates
	float x = radius * cos(radians);
	float z = radius * sin(radians);
	
	// Add vertices
	vertices.push_back(Vector3D(x, y, z));
    }
}

SolidMesh *SolidBuilder::buildCuboid()
{
    float vertexData[] = {
	-0.5, 0.5, -0.5, 0.5, 0.5, -0.5, -0.5, -0.5, -0.5,
	0.5, -0.5, -0.5, -0.5, 0.5, 0.5, 0.5, 0.5, 0.5,
	-0.5, -0.5, 0.5, 0.5, -0.5, 0.5
    };
    float texCoordData[] = { 0.0, 0.0, 1.0, 0.0, 1.0, 1.0, 0.0, 1.0 };
    int vertexIndexData[] = {
	0, 1, 2, 1, 3, 2, 1, 5, 7, 1, 7, 3, 5, 4, 6, 5, 6, 7,
	0, 6, 4, 0, 2, 6, 0, 4, 5, 0, 5, 1, 7, 6, 2, 2, 3, 7 
    };
    int texCoordIndexData[] = {
	2, 3, 1, 3, 0, 1, 2, 3, 0, 2, 0, 1, 2, 3, 0, 2, 0, 1,
	3, 1, 2, 3, 0, 1, 3, 0, 1, 3, 1, 2, 2, 3, 0, 0, 1, 2
    };

    // Calculate number of elements
    int vertexNumber = sizeof(vertexData) / sizeof(float);
    int texCoordNumber = sizeof(texCoordData) / sizeof(float);
    int vertexIndexNumber = sizeof(vertexIndexData) / sizeof(int);
    int texCoordIndexNumber = sizeof(texCoordIndexData) / sizeof(int);
    
    // Build lists out of the arrays
    std::vector<float> vertexList(vertexData, vertexData + vertexNumber);
    std::vector<float> texCoordList(texCoordData,
				    texCoordData + texCoordNumber);
    std::vector<int> vertexIndexList(vertexIndexData,
				     vertexIndexData + vertexIndexNumber);
    std::vector<int> texCoordIndexList(texCoordIndexData, texCoordIndexData +
				       texCoordIndexNumber);
    
    // Now, build the solid and return it
    return buildSolidMesh(vertexList, texCoordList, vertexIndexList,
			  texCoordIndexList);
}

SolidMesh *SolidBuilder::buildPlane()
{
    float vertexData[] = {
	-0.5, 0.0, -0.5, -0.5, 0.0, 0.5, 0.5, 0.0, 0.5, 0.5, 0.0, -0.5
    };
    float texCoordData[] = { 0.0, 0.0, 1.0, 0.0, 1.0, 1.0, 0.0, 1.0 };
    int vertexIndexData[] = { 0, 1, 2, 0, 2, 3 };
    int texCoordIndexData[] = {	0, 2, 3, 0, 1, 2 };
    
    // Calculate number of elements
    int vertexNumber = sizeof(vertexData) / sizeof(float);
    int texCoordNumber = sizeof(texCoordData) / sizeof(float);
    int vertexIndexNumber = sizeof(vertexIndexData) / sizeof(int);
    int texCoordIndexNumber = sizeof(texCoordIndexData) / sizeof(int);
    
    // Build lists out of the arrays
    std::vector<float> vertexList(vertexData, vertexData + vertexNumber);
    std::vector<float> texCoordList(texCoordData,
				    texCoordData + texCoordNumber);
    std::vector<int> vertexIndexList(vertexIndexData,
				     vertexIndexData + vertexIndexNumber);
    std::vector<int> texCoordIndexList(texCoordIndexData, texCoordIndexData +
				       texCoordIndexNumber);
    
    // Now, build the solid and return it
    return buildSolidMesh(vertexList, texCoordList, vertexIndexList,
			  texCoordIndexList);
}

SolidMesh *SolidBuilder::buildCylinder(int segments)
{
    // Sanity check
    if (segments < 3) segments = 3;
    
    // Create the solid mesh container
    SolidMesh *s = new SolidMesh();

    // Create the top and bottom vertices (as circles)
    createHorzCircle(s->vertices, 1.0, segments, 0.5);
    createHorzCircle(s->vertices, 1.0, segments, -0.5);
    
    // Add the two center vertices
    s->vertices.push_back(Vector3D(0, 0.5, 0));
    s->vertices.push_back(Vector3D(0, -0.5, 0));

    // Add center texture coordinate
    s->texCoords.push_back(Vector2D(0, 0));

    // Build faces for the bottom and top
    for (int n = 0; n < segments; n++) {
	TriangleFace t;

	t.v[0] = 2 * segments;
	t.v[1] = (n + 1) % segments;
	t.v[2] = n;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);
	
	t.v[0] = 2 * segments + 1;
	t.v[1] = n + segments;
	t.v[2] = (n + 1) % segments + segments;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);
    }
    
    // Build the faces for the "mantle"
    for (int n = 0; n < segments; n++) {
	TriangleFace t;
	
	t.v[0] = n;
	t.v[1] = (n + 1) % segments;
	t.v[2] = n + segments;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);

	t.v[0] = n + segments;
	t.v[1] = (n + 1) % segments;
	t.v[2] = (n + 1) % segments + segments;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);
    }
    
    // Compute normals
    computeNormals(s, false);
    
    return s;
}
    
SolidMesh *SolidBuilder::buildSphere(int latSegments, int longSegments)
{
    // Sanity check
    if (latSegments < 3) latSegments = 3;
    if (longSegments < 3) longSegments = 3;
    
    // Create the solid mesh container
    SolidMesh *s = new SolidMesh();

    // Create the latitudinal circle
    int num_circles = latSegments - 2;
    float y_delta = 2.0 / (latSegments - 1);
    for (int n = 0; n < num_circles; n++) {
	float y = (n + 1) * y_delta - 1.0f;
	float radius = sqrt(1.0 - y * y);
	createHorzCircle(s->vertices, radius, longSegments, y);
    }
    
    // Add the two center vertices
    s->vertices.push_back(Vector3D(0, -1.0, 0));
    s->vertices.push_back(Vector3D(0, 1.0, 0));

    // Add center texture coordinate
    s->texCoords.push_back(Vector2D(0, 0));

    // Build faces for the bottom and top caps
    for (int n = 0; n < longSegments; n++) {
	TriangleFace t;
	
	t.v[0] = num_circles * longSegments;
	t.v[1] = n;
	t.v[2] = (n + 1) % longSegments;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);

	t.v[0] = num_circles * longSegments + 1;
	t.v[1] = (n + 1) % longSegments + (num_circles - 1) * longSegments;
	t.v[2] = n + (num_circles - 1) * longSegments;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);
    }
    
    // Build the faces for the rest of the sphere
    for (int n = 0; n < num_circles - 1; n++) {

	// Compute an index offset
	int offset = n * longSegments;

	// Iterate around the whole perimeter
	for (int m = 0; m < longSegments; m++) {
	    TriangleFace t;
	    
	    t.v[0] = m + offset;
	    t.v[1] = m + longSegments + offset;
	    t.v[2] = (m + 1) % longSegments + offset;
	    t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	    s->faces.push_back(t);
	    
	    t.v[0] = m + longSegments + offset;
	    t.v[1] = (m + 1) % longSegments + longSegments + offset;
	    t.v[2] = (m + 1) % longSegments + offset;
	    t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	    s->faces.push_back(t);
	}
    }

    // Compute normals
    computeNormals(s, false);
    
    return s;    
}

SolidMesh *SolidBuilder::buildCone(int segments)
{
    // Sanity check
    if (segments < 3) segments = 3;

    // Create the solid mesh container
    SolidMesh *s = new SolidMesh();
    
    // Create the bottom vertices (as a circle)
    createHorzCircle(s->vertices, 1.0, segments, 0);

    // Add the two center vertices
    s->vertices.push_back(Vector3D(0, 0, 0));
    s->vertices.push_back(Vector3D(0, 1, 0));

    // Add center texture coordinate
    s->texCoords.push_back(Vector2D(0, 0));
    
    // Build faces for the bottom
    for (int n = 0; n < segments; n++) {
	TriangleFace t;
	t.v[0] = segments; t.v[1] = n; t.v[2] = (n + 1) % segments;
	//t.n[0] = 0; t.n[1] = n + 2; t.n[2] = n + 1 + 2;
	//t.tc[0] = 0; t.tc[1] = n + 1; t.tc[2] = n + 1 + 1;
	t.tc[0] = 0; t.tc[1] = 0; t.tc[2] = 0;
	s->faces.push_back(t);
    }

    // Build faces for the "mantle"
    for (int n = 0; n < segments; n++) {
	TriangleFace t;
	t.v[0] = segments + 1; t.v[2] = n; t.v[1] = (n + 1) % segments;
	//t.n[0] = 1; t.n[2] = n + 2; t.n[1] = n + 1 + 2;
	//t.tc[0] = 0; t.tc[2] = n + 1; t.tc[1] = n + 1 + 1;
	t.tc[0] = 0; t.tc[2] = 0; t.tc[1] = 0;
	s->faces.push_back(t);
    }
    
    // Compute normals
    computeNormals(s, true);
    
    return s;
}

SolidMesh *SolidBuilder::buildTorus(int slices, int segments)
{
    return 0;
}

SolidMesh *SolidBuilder::buildSolidMesh(const std::vector<float> &v_data,
					const std::vector<float> &tc_data,
					const std::vector<int> &vi_data,
					const std::vector<int> &tci_data)
{
    // Create the solid mesh container
    SolidMesh *s = new SolidMesh();
    
    // Define the vertex data
    for (unsigned int i = 0; i < v_data.size(); i += 3)
	s->vertices.push_back(Vector3D(v_data[i], v_data[i + 1],
				       v_data[i + 2]));

    // Don't forget the texture coordinate data!
    for (unsigned int i = 0; i < tc_data.size(); i += 2)
	s->texCoords.push_back(Vector2D(tc_data[i], tc_data[i + 1]));
    
    // Finally, the face data
    for (unsigned int i = 0; i < vi_data.size(); i += 3) {
	
 	TriangleFace t;
	
	// Define the indices
	t.v = TriangleIndex(vi_data[i], vi_data[i + 1], vi_data[i + 2]);
	t.tc = TriangleIndex(tci_data[i], tci_data[i + 1], tci_data[i + 2]);
	
	// Add it to the list
	s->faces.push_back(t);
    }

    // Compute normals for the solid
    computeNormals(s, false);

    return s;
}
