/* ------------------------------------------------------------------------
 * $Id: SolidEvaluator.hh,v 1.7 2001/08/27 15:10:26 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-23 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _SolidEvaluator_hh_
#define _SolidEvaluator_hh_

// -- System Includes
#include <stack>
#include <vector>

// -- 3Dwm Includes
#include "Nobel/CORBA.hh"
#include "Nobel/Appearance.hh"
#include "Nobel/SolidVisitor.hh"

// -- Forward Declarations
class BSPTree;
class SolidCache;

// -- Struct Declarations

/**
 * Triangle mesh geometry shape struct. Contains a triangle mesh and
 * an appearance to go with it (i.e. the material).
 **/
struct TriMeshShape {

    /// Triangle mesh pointer
    Nobel::TriangleMesh *mesh;

    /// Appearance reference
    Nobel::Appearance_var appearance;

    /**
     * Constructor.
     **/
    TriMeshShape() : mesh(0), appearance(Nobel::Appearance::_nil()) { }

    /**
     * Constructor.
     *
     * @param _mesh pointer to triangle mesh to use.
     * @param _app appearance reference.
     **/
    TriMeshShape(Nobel::TriangleMesh *_mesh, Nobel::Appearance_ptr _app)
	: mesh(_mesh), appearance(_app) { }
};

// -- Class Declarations

/**
 * Solid tree evaluator. Traverses a solid tree and builds up an
 * internal representation of it that can be rendered.
 **/
class SolidEvaluator : public virtual POA_Nobel::Solid::Visitor,
		       public virtual PortableServer::RefCountServantBase {
public:
    
    /**
     * Constructor.
     *
     * @param cache pointer to solid cache to use.
     **/ 
    SolidEvaluator(SolidCache *cache);
    
    /**
     * Destructor. 
     **/
    virtual ~SolidEvaluator() { }

    /**
     * Evaluate a given solid tree. The evaluator will traverse the
     * tree hierarchy and build the triangle mesh it represents.
     *
     * @return pointer to triangle mesh representing the solid tree.
     **/
    std::vector<TriMeshShape> evaluate(Nobel::Solid::Node_ptr tree);

    // IDL functions
    virtual void visitComplement(Nobel::Solid::Unary_ptr node);
    virtual void visitUnion(Nobel::Solid::Binary_ptr node);
    virtual void visitIntersection(Nobel::Solid::Binary_ptr node);
    virtual void visitSubtraction(Nobel::Solid::Binary_ptr node);
    virtual void visitAppearance(Nobel::Solid::Appearance_ptr node);
    virtual void visitTransform(Nobel::Solid::Transform_ptr node);
    virtual void visitGeometry(Nobel::Solid::Geometry_ptr node);
    virtual void visitPrimitive(Nobel::Solid::Primitive_ptr node);
private:

    /**
     * Build a list of triangle meshes for different materials given a
     * BSP tree.
     *
     * @param t pointer to BSP tree to use.
     * @return list of meshes and their associated appearances.
     **/
    std::vector<TriMeshShape> buildMeshFromBSP(BSPTree *t) const;
    
    /**
     * Build a BSP tree from a triangle mesh. 
     *
     * @param mesh pointer to triangle mesh.
     * @return pointer to BSP tree representing mesh.
     **/
    BSPTree *buildBSPFromMesh(Nobel::TriangleMesh *mesh);

    /**
     * Retrieve an appearance reference given a material index.
     *
     * @param material index for the desired appearance.
     * @return reference to the desired appearance object.
     **/
    Nobel::Appearance_ptr getAppearance(int material) const;
    
    /**
     * Retrieve the current material index.
     *
     * @return current material index (-1 if none).
     **/
    int getCurrentMaterial() const {
	if (_material_stack.empty()) return -1;
	return _material_stack.top();
    }

    /**
     * Push a material index onto the material stack. 
     *
     * @param material material index to push onto the material stack.
     **/
    void pushMaterial(int material) {
	_material_stack.push(material);
    }

    /**
     * Pop the topmost (current) material index off the material
     * stack. This will expose the following material index as the new
     * current material.
     **/
    void popMaterial() {
	_material_stack.pop();
    }

    /// Tree stack item type
    struct TreeItem {
	BSPTree *tree;
	bool changed;
	TreeItem(BSPTree *t = 0, bool c = false) : tree(t), changed(c) { }
    };
    
    /**
     * Push a BSP tree onto the evaluation context stack.
     *
     * @param tree pointer to BSP tree to push.
     * @param changed flag whether the tree has changed or not.
     **/
    void pushTree(BSPTree *tree, bool changed = false) {
	_tree_stack.push(TreeItem(tree, changed));
    }

    /**
     * Retrieve the topmost BSP tree from the evaluation context
     * stack. The tree will be popped off the stack.
     *
     * @return the topmost tree item on the stack.
     **/
    TreeItem popTree() {
	TreeItem tree = _tree_stack.top();
	_tree_stack.pop();
	return tree;
    }

    std::stack<TreeItem> _tree_stack;
    std::stack<int> _material_stack;
    std::vector<Nobel::Appearance_var> _appearances;
    Nobel::Solid::Node_var _current_tree;
    SolidCache *_cache;
    bool _appChanged;
};

#endif /* SolidEvaluator.hh */
