# -*- coding: utf-8 -*-

# Copyright (c) 2008 - 2010 Lukas Hetzenecker <LuHe@gmx.at>

import pickle
from PyQt4.QtCore import *
from PyQt4.QtGui import *
from PyQt4.QtSql import *
import ui.ui_wizard_helloPage
import ui.ui_wizard_phonePage
import ui.ui_wizard_installPage
import ui.ui_wizard_runPage
import ui.ui_wizard_minimizePage
import ui.ui_wizard_databasePage
import ui.resource_rc
import lib.obex_wrapper
from lib.classes import *

try:
   # PyBluez module for Linux and Windows
   import bluetooth
except ImportError:
   # Lightblue for Mac OS X
   import lightblue

LINUX= "qt_x11_wait_for_window_manager" in dir()

class Wizard(QWizard):
    def __init__(self,  parent,  main):
        super(Wizard,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        self.helper = main.helper

        self.__device = None

        self.helloPage = helloPage(self,  main)
        self.phonePage = phonePage(self,  main)
        self.installPage = installPage(self,  main)
        self.runPage = runPage(self,  main)
        self.minimizePage = minimizePage(self,  main)
        self.databasePage = databasePage(self,  main)

        self.addPage(self.helloPage)
        self.addPage(self.phonePage)
        self.addPage(self.installPage)
        self.addPage(self.runPage)
        self.addPage(self.minimizePage)
        self.addPage(self.databasePage)

        self.connect(self,  SIGNAL("finished(int)"),  self.finished)

        self.setWindowTitle(self.tr("Series 60 Remote - Settings"))
        self.setWindowIcon(QIcon(":/phone"))

        self.log.info(QString("Show wizard"))

        self.show()

    def __str__(self):
        return "\"Wizard\""

    def setDevice(self,  device):
        self.__device = device

    def device(self):
        return self.__device

    def finished(self,  id):
        if id == 1:
            self.settings.setSetting("general/firstStart",  False)
        else:
            self.main.app.quit()

    def closeEvent(self,  event):
        self.main.app.quit()
        event.accept()

class helloPage(ui.ui_wizard_helloPage.Ui_helloPage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(helloPage,  self).__init__(parent)
        self.setupUi(self)

    def validatePage(self):
        return True

class phonePage(ui.ui_wizard_phonePage.Ui_phonePage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(phonePage,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        
        self.scanStartedItem = None

        self.setupUi(self)

        self.connect(self.connection,  SIGNAL("scanStarted"), self.scanStarted)
        self.connect(self.connection,  SIGNAL("foundDevice"), self.scanFoundDevice)
        self.connect(self.connection,  SIGNAL("scanFailed"), self.scanFailed)
        self.connect(self.connection,  SIGNAL("scanCompleted"), self.scanCompleted)
        self.connect(self.rescanButton,  SIGNAL("clicked()"),  self.scanStart)

        self.scanStart()

    def scanStart(self):
        self.connection.scan()

    def scanStarted(self):
        self.log.info(QString("Device scan started..."))
        self.devicesTree.clear()
        self.scanStartedItem = QTreeWidgetItem(self.devicesTree)
        self.scanStartedItem.setText(0, self.trUtf8("Scanning..."))
        self.scanStartedItem.setIcon(0,  QIcon(":/wait"))
        self.scanStartedItem.setFirstColumnSpanned(True)

        self.rescanButton.setEnabled(False)

    def scanFoundDevice(self,  addr,  name):
        self.log.info(QString("Found device '%1' with MAC %2").arg(name,  addr))
        item = QTreeWidgetItem(self.devicesTree)
        item.setText(0, name)
        item.setText(1, addr)

    def scanFailed(self,  message):
        self.log.error(QString("Device scan failed with error: %1").arg(message))
        self.devicesTree.clear()
        scanFailedItem = QTreeWidgetItem(self.devicesTree)
        scanFailedItem.setText(0, self.tr("Scanning failed: %1").arg(message))
        scanFailedItem.setToolTip(0, self.tr("Scanning failed: %1").arg(message))
        scanFailedItem.setIcon(0,  QIcon(":/dialog-close"))
        scanFailedItem.setFirstColumnSpanned(True)

        self.rescanButton.setEnabled(True)

    def scanCompleted(self):
        self.log.info(QString("Device scan completed"))
        self.devicesTree.takeTopLevelItem(0)
        self.scanStartedItem= None
        
        if self.devicesTree.topLevelItemCount() == 0:
            item = QTreeWidgetItem(self.devicesTree)
            item.setText(0, self.tr("No devices found"))
            item.setFirstColumnSpanned(True)

        self.devicesTree.resizeColumnToContents(0)
        self.devicesTree.resizeColumnToContents(1)

        self.rescanButton.setEnabled(True)

    def validatePage(self):
        try:
            device = self.devicesTree.currentItem()
            name = unicode(device.text(0))
            addr = str(device.text(1))
        except:
            name = ""
            addr = ""

        if not (addr and name):
            self.log.warning(QString("No device selected"))
            ret = QMessageBox.question(self, self.tr("No devices found"),
            self.trUtf8("""You didn't select a mobile phone.
If you continue, you have to select it afterwards in the settings."""),
                QMessageBox.StandardButtons(\
                    QMessageBox.Ignore | \
                    QMessageBox.Retry),
                QMessageBox.Retry)

            if ret == QMessageBox.Retry:
                return False
            else:
                self.connection.scanStop()
                return True

        self.log.info(QString("Selected device %1 with mac %2").arg(name, addr))

        device = Device()
        device.setName(name)
        device.setBluetoothAddress(addr)

        self.settings.setSetting("bluetooth/lastName",  name)

        self.parent.setDevice(device)

        self.connection.scanStop()
        return True

class installPage(ui.ui_wizard_installPage.Ui_installPage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(installPage,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings
        self.helper = main.helper

        self.setupUi(self)

    def initializePage(self):
        device = self.parent.device()

        if lib.obex_wrapper.FOUND_OBEX:
            self.obexStack.setCurrentWidget(self.obexFoundWidget)
            self.log.info(QString("OBEX support was found, trying to send installation file to device"))
        else:
            self.obexStack.setCurrentWidget(self.obexNotFoundWidget)
            self.log.info(QString("OBEX support was not found"))
            if LINUX:
                self.operatingSystemStack.setCurrentWidget(self.linuxWidget)
            else:
                self.operatingSystemStack.setCurrentWidget(self.windowsWidget)

        self.connect(self.sendApplicationButton,  SIGNAL("clicked()"),  
                                lambda : self.helper.sendFile(self,  device, self.main.applicationSis,  self.main.applicationSisRegexp))
        self.connect(self.sendPythonButton,  SIGNAL("clicked()"),  
                                lambda : self.helper.sendFile(self, device, self.main.pythonSis,  self.main.pythonSisRegexp))
        self.connect(self.openFolderButton,  SIGNAL("clicked()"),  self.helper.openFolder)
    
    def validatePage(self):
        return True

class runPage(ui.ui_wizard_runPage.Ui_runPage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(runPage,  self).__init__(parent)
        self.setupUi(self)

    def validatePage(self):
        return True

class minimizePage(ui.ui_wizard_minimizePage.Ui_minimizePage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(minimizePage,  self).__init__(parent)
        self.setupUi(self)

    def validatePage(self):
        return True

class databasePage(ui.ui_wizard_databasePage.Ui_databasePage,  QWizardPage):
    def __init__(self,  parent,  main):
        super(databasePage,  self).__init__(parent)

        self.parent = parent
        self.main = main

        self.log = main.log
        self.connection = main.connection
        self.database = main.database
        self.settings = main.settings

        self.setupUi(self)

        self.sqliteDriverNotFoundLabel.setHidden(QSqlDatabase.drivers().contains("QSQLITE"))
        self.mysqlDriverNotFoundLabel.setHidden(QSqlDatabase.drivers().contains("QMYSQL"))

    def validatePage(self):
        type = str(self.typeBox.currentText()).lower()
        data = dict()

        if type == "sqlite":
           file = str(QFileInfo(self.settings.fileName()).absoluteDir().absolutePath()) + "/messages.db"
           data["filename"] = file
        elif type == "mysql":
            host = str(self.hostLine.text())
            port = int(self.portBox.value())
            user = str(self.userLine.text())
            pw = str(self.passLine.text())
            db = str(self.databaseLine.text())

            if not (host and port and user and pw and db):
                QMessageBox.critical(self,  self.trUtf8("Mysql connection fields incomplete!"), self.trUtf8("You didn't fill in all required fields. If you are ubsure please use SQLite."))
                return False

            data["host"] = host
            data["port"] = port
            data["user"] = user
            data["pass"] = pw
            data["database"] = db

        self.log.info(QString("Trying to connect to database"))
        ret = self.database.openDatabase(type,  data)

        if ret == False:
            # Connection wasn't succesful
            # Show error message
            err = self.database.error()

            message = QMessageBox.critical(self,
            self.tr("Connection failed!"),
            self.trUtf8("Connection failed with error:\n%1\n").arg(err),
            QMessageBox.StandardButtons(\
                    QMessageBox.Ignore | \
                    QMessageBox.Retry),
            QMessageBox.Retry)

            if message == QMessageBox.Retry:
                return False

        else:
            # Connection was successful
            # Write connection data to settings
            self.settings.beginGroup("database")
            self.settings.setSetting("type",  type)
            if type == "sqlite":
                self.settings.beginGroup("sqlite")
                self.settings.setSetting("filename",  file)
                self.settings.endGroup()
            elif type == "mysql":
                self.settings.beginGroup("mysql")
                self.settings.setSetting("host", host)
                self.settings.setSetting("port", port)
                self.settings.setSetting("user", user)
                self.settings.setSetting("pass", pw)
                self.settings.setSetting("database", db)
                self.settings.endGroup()
            self.settings.endGroup()

            if self.parent.device():
                self.database.deviceAdd(self.parent.device())

        return True
