// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc/los.cpp,v 1.1.1.1 2001/07/23 07:25:39 xli18 Exp $
//

 
#include "gc_for_orp.h"
 
#include "los.h"
#include "object_layout.h"
#include "remembered_set.h"
#include "descendents.h"
#include "gc_hooks.h"
#include "generation.h"
#include "gc_consts.h"
 
#include "gc_globals.h"
#include "gc_plan.h"
#include "gc_debug.h"
 
#include "gc_consts.h"
#include "gc_perf.h"
#include "los_container.h"
#include "root_set_enum.h"
#include "orp_threads.h"
#include "orp_synch.h"
//
// This is a dummy class created in
// gc_interface.cpp, which has objects
// of size 12 bytes(for IA32), used for double 
// aligning objects when needed.
//
 
// extern Partial_Reveal_Class  *p_alignment_filler_class;
extern Partial_Reveal_VTable *p_alignment_filler_vtable;
 
#ifdef SPACE_DEMOGRAPHICS
void record_for_space_demographics(unsigned obj_size)
{
    obj_size += sizeof(Object_Gc_Header);

    if (obj_size > 8000) {
        orp_cout << "Big object." << endl;
    }
    if (obj_size < DEMOGRAPHICS_ARRAY_SIZE) {
        space_demographics_size[obj_size] += 1;
    } else {
        cout << "OBJECT OF SIZE " << obj_size << " GREATER THAN SPACE DEMOGRAPHICS LIMIT " << endl;
    }
}
#endif

//
// This is a hack for Milind which reclaims every
// 'N' allocations (specified on the command line
// using an undocumented "-rf" argument).
// NOTE THIS CURRENTLY ONLY WORKS FOR GC_FIXED_V1
//
void forced_reclaim_heap()
{
    p_gc->reclaim_heap(0, true);
}

#ifdef GC_STATS
//
// The LOS is being give a chance to set up the instrumentation
// to do things like space accounting, just before a GC hits.
//
void 
Large_Object_Store::prepare_for_collection() 
{
    _stat_marked_space    = 0;
    _stat_swept_free      = 0;
    _stat_swept_live      = 0;
    _stat_swept_reclaimed = 0;
    _stat_free_space_not_recovered = 0;
}
#else
void 
Large_Object_Store::prepare_for_collection() 
{
}
#endif // GC_STATS



Large_Object_Store::Large_Object_Store(Gc_Interface  *p_gc_interface,
                                       Gc_Fast_Hooks *p_gc_hooks,
                                       Gc_Plan       *p_gc_plan,
									   Generation    *p_container,
		                               Card_Table    *p_card_table,
					                   Block_Store   *p_bs,
                                       LOS_Container *p_los_container)
					   : Gc_Space(0, 
                                  p_gc_hooks,
                                  p_gc_plan,
					              p_container, 
								  NULL, 
					              p_card_table, 
								  p_bs)
{
	_los_block_size_bytes = p_gc_plan->los_block_size_bytes();
	//
	// Get a contiguous set of blocks of the specified
	// size for use in LOS. NOTE we set the generation to YOS_GEN_NUMBER
	// signifying that the LOS is in young space.
	// YOS_GEN_NUMBER is assumed to be 0.
    //
	_p_free_list = 
		_p_base  = 
		_p_block_store->p_get_new_super_block(this,
		                                      _los_block_size_bytes,
											  YOS_GEN_NUMBER, 
                                              false);
    _p_start_search_here = _p_free_list;

	//
	// Initialize the free list to just have one giant block. We call
    // initialize_free_block, rather than reinitialize_free_block,
    // since the latter also sets all the words to a distinct pattern
    // if the right debugging level is set, and can prove quite slow.
	//

	initialize_free_block(_p_free_list, 
                          _los_block_size_bytes,
						  NULL);

	_p_ceiling =
		(void *)((Byte *)_p_base +
				 _los_block_size_bytes);


	//
	// The following will point to the object in the LOS
	// at the highest address. This is useful to have when
	// linearly scanning the LOS, so we don't fall off the
	// edge of the last allocated object.
	//
	_p_last_object_gc_header = NULL;
	_p_last_object           = NULL;
	//
	// A couple of variables for keeping statistics of the LOS.
	//
	_smallest_object_size    = 9999;
	_largest_object_size     = 0;
#ifdef GC_STATS
    _stat_marked_space       = 0;
    _stat_swept_live         = 0;
    _stat_swept_reclaimed    = 0;
    _stat_swept_free         = 0;
    _stat_free_space_not_recovered = 0;
#endif // GC_STATS
	//
	// Save the pointer to the container GC interface
	// for future use. This is used for informing the
	// GC Interface when the LOS is out of space.
	//
	_p_gc_interface = p_gc_interface;

    // Remember the large object store container
    _p_los_container = p_los_container;
	//
	// Turn the low space warning off by default. This will be
	// set when a first-fit fails, then checked during the
	// allocation after the subsequent collection, to see if
	// the heap needs extending.
	//
	_los_space_may_be_critically_low = false;

    // #ifdef GC_MARK_FLAGS implements non-resident mark bits.

    // Sapphire needs to move the mark bit out of the header where it overloads the
	// lock bit. We will assume that the smallest object in fixed object space (LOS)
	// is at least 4 words (16 bytes) long (_min_size_bytes).
	// This means that we need mark_flag_array_size_bits is _los_block_size_bytes/_min_size_bytes flags
	// Just to keep the logic simple use an array of bytes (char) instead of bits.
	// If this ends up being too much space for production code then let a single bit stand for
	// all the objects in some large area.

	// PERFORMANCE PERFORMANCE PERFORMANCE
	// Ask someone what the best bit vector package is and use it instead of byte arrays.

    assert (MARK_AREA_BYTES == (1 << MARK_SHIFT));

	long int mark_flag_array_size = _los_block_size_bytes/MARK_AREA_BYTES;
	_mark_flags = new char[mark_flag_array_size];
	// clear the flags, they will be recleared as they are scanned.
	memset(_mark_flags, 0, mark_flag_array_size);
	assert (_mark_flags);

    // #endif // GC_MARK_FLAGS

    // Life starts out with the lock free.
    this_los_lock = (ORP_thread *)NULL;

    set_generation (p_container);
}

     // #ifdef GC_MARK_FLAGS implements non resident mark bits.
void Large_Object_Store::clear_mark_flags ()
{
	long int mark_flag_array_size = _los_block_size_bytes/MARK_AREA_BYTES;
	// clear the flags, they will be recleared as they are scanned.
	memset(_mark_flags, 0, mark_flag_array_size);
}
     // #endif // GC_MARK_FLAGS
//
// When the Large Object Store gets an evict call, it marks and scans 
// all the descendents of the object, but does not move that object.
// Currently unreachable objects are not reclaimed from this store.
//
Java_java_lang_Object *
Large_Object_Store::p_evict_object(Java_java_lang_Object **pp_obj_ref, bool doing_mos_collection)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
	Java_java_lang_Object *p_obj = *pp_obj_ref;
	//
	// If the object is marked, this means that it has already 
	// been visited. The forwarded bit and the mark bit are the same.
    //               BUT DO NOT DEPEND ON THIS IN THE CODE.
	//
	if (is_object_marked(p_obj)) {
		return p_obj;
	}
    //
    // A mark bit in the header signals that this object has been scanned.
    // After the stop-the-world collection, the cleanup routine should
    // sequentially scan all live objects and turn the mark bit off.
    //
	set_object_marked(p_obj);
#ifdef GC_STATS
    //
    // Update the statistics.
    //
    _stat_marked_space += get_real_object_size_bytes(p_obj);
#endif

    //
	// Use the generation of the object to dictate the object 
	// scanning policy.
	// 
	Generation *p_gen = p_global_bs->p_get_object_generation(p_obj);
	p_gen->cheney_scan_execute_policy(p_obj, doing_mos_collection);

	return p_obj;
}

//
// Mrl_Gc_Vxx calls us to clean up after a stop-the-world collection.
// This is our chance to clear the mark bits in all the objects.
//
void 
Large_Object_Store::cleanup()
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif

#ifdef CLASS_DEMOGRAPHICS
    _walk_los(walk_report_dead_object);
#endif
    prepare_for_collection();
    _linear_sweep_heap_and_reclaim();
#ifdef GC_STATS
	if (stats_gc) {
        int the_size = (int)_p_ceiling - (int)_p_base;
		cout << "LOS Statistics:" << endl;
        cout << the_size << " total size (Ceiling - Base) " << endl;
		cout << _stat_marked_space << " size of objects marked" <<endl;
		cout << _stat_swept_live << " size of live objects swept" <<endl;
		cout << _stat_swept_reclaimed << " size of objects that died" <<endl;
		cout << _stat_swept_free << " size of free space encountered" <<endl;
		cout << "Free space (died + free) = " << _stat_swept_reclaimed + _stat_swept_free << endl;
		cout << "Used space = " << _stat_swept_live << endl;
		cout << "Total space (dead + free + marked) = ";
		cout << _stat_swept_reclaimed + _stat_swept_free + _stat_swept_live << endl;
	}
    //
    // These will get reset when the LOS gets a message to
    // prepare for GC at the start of the next cycle.
    //
#endif // GC_STATS
}

//
// Start a first fit search with this free block.
//
Java_java_lang_Object *
Large_Object_Store::_first_fit_with_start(unsigned int real_size_bytes, 
                                          void *start_here,
                                          bool double_alignment_required                                    
                                          )
{
	assert(p_container);
//    orp_cout << "start_here is " << start_here << endl;
    // Skip the first block and start with the second so we have a previous block
    // to link to if we need to.
    void *p_free;  // = get_free_block_link(start_here);
	void *p_back;
	Java_java_lang_Object *p_return;

    if ((start_here == NULL) || (start_here == _p_free_list)) {
        p_free = _p_free_list;
        p_back = _p_free_list;
    } else {
        p_back = start_here;
        p_free = get_free_block_link (start_here);
    }

//    orp_cout << "search starts here " << p_free << endl;

    while (p_free != NULL) {     
        unsigned available_chunk_size = get_free_block_size(p_free);
        unsigned remaining_chunk_size = available_chunk_size - real_size_bytes;

        unsigned int effective_size_bytes = real_size_bytes;
        bool unaligned_address = false;

#ifdef POINTER64
        // Assert that we are always moving along 8 byte aligned areas for IA64
        assert(!(((POINTER_SIZE_INT)p_free) & 0x7));
#endif
        if (double_alignment_required == true) {
#ifdef POINTER64
            // In IA64 there is not special double alignment required.
            assert(0);
#endif
            if (is_not_double_aligned(p_free)) {
                effective_size_bytes += sizeof(Partial_Reveal_JavaArray) + OBJECT_HEADER_SIZE;
                remaining_chunk_size -= sizeof(Partial_Reveal_JavaArray) + OBJECT_HEADER_SIZE;
                unaligned_address = true;
            }
        }
        if (available_chunk_size == effective_size_bytes) { // we have a perfect fit

            p_return = (Java_java_lang_Object *)p_free;
			if (p_back == p_free) {
				_p_free_list = get_free_block_link(p_free); 
                _p_start_search_here = _p_free_list; // Start using _p_free_list;
            } else {
				void *p_next = get_free_block_link(p_free); 
				set_free_block_link(p_back, p_next);
                assert (p_back != NULL);
                _p_start_search_here = p_back;
			}

            //
            // Craft a dummy object at the front if unaligned
            //

            // Bug me to get this right.            
            if (unaligned_address) {
#ifdef POINTER64
                assert(0);
#endif
                // Fill it with a 12 byte header(for IA32) for an empty array of bytes.
                void *p_dummy_object = p_return;
                assert (p_alignment_filler_vtable);
                p_return = (Java_java_lang_Object *)((char *)p_return + 
                                sizeof(Partial_Reveal_JavaArray) + OBJECT_HEADER_SIZE);
                initialize_byte_array (p_dummy_object,
                                       0,
                                       p_alignment_filler_vtable);
 
            } 

            return p_return;
        }
        // This builds the free chunk list at the end of the block.
        // See below for code that returns allocated space at the
        // end of the block. Use #if 0 to build below.

        if ((effective_size_bytes + 3 * sizeof(void *)) < available_chunk_size) {
            real_size_bytes = effective_size_bytes;

            p_return = (Java_java_lang_Object *)p_free;
            if (p_back == p_free) { 
                // In the case where the object is only 8 bytes we need to
                // get p_next immediately before we destroy it setting up
                // the next free block link.
				void *p_next = get_free_block_link(p_free);
				_p_free_list = (void *)((char *)p_free + real_size_bytes);
				set_free_block_header(_p_free_list);
				set_free_block_size(_p_free_list, remaining_chunk_size);
                assert ((POINTER_SIZE_INT)p_next != FREE_BLOCK_HEADER);
				set_free_block_link(_p_free_list, p_next);
                _p_start_search_here = _p_free_list;              
			} else {
				void *p_next = get_free_block_link(p_free);
				void *p_remaining = (void *)((char *)p_free + real_size_bytes);
				set_free_block_header(p_remaining);
				set_free_block_size(p_remaining, remaining_chunk_size);
				set_free_block_link(p_remaining,p_next); 
				set_free_block_link(p_back, p_remaining);
                _p_start_search_here = p_back;
//                orp_cout << "_p_start_search_here " << _p_start_search_here << endl;
            }


            //
            // Craft a dummy object at the front if unaligned
            //
            if (unaligned_address) {
#ifdef POINTER64
                assert(0);
#endif
                // Fill it with a 12 byte header(if IA32) for an empty array of bytes.
                void *p_dummy_object = p_return;
                assert (p_alignment_filler_vtable);
                p_return = (Java_java_lang_Object *)((char *)p_return + 
                                sizeof(Partial_Reveal_JavaArray) + OBJECT_HEADER_SIZE);
                initialize_byte_array (p_dummy_object,
                                       0,
                                       p_alignment_filler_vtable);
  
            } 

			return p_return;
        } else {
			p_back = p_free;
			p_free = get_free_block_link(p_free);
		}
//
// RLH tried a sequence which returns the allocated
// space from the end of the block instead of from the start of the
// block.
// It slowed things down a little but deserves to be revisited.
//
// This is possible due to some sort of cache issue.
//
	}
	return NULL;
} // first_fit_with_start

//
// When asked to allocate space, the LOS first attempts
// a best fit. If this fails, it attempts a first fit.
//
Java_java_lang_Object *
Large_Object_Store::_first_fit(unsigned int real_size_bytes,
                               bool double_alignment_required                         
                               )
{
	assert(p_container);
    // Start where we left off and search to the end.
    Java_java_lang_Object *p_return = _first_fit_with_start (real_size_bytes, 
                                            _p_start_search_here,
                                            double_alignment_required                                      
                                            );
    if (p_return == NULL) { // We failed try using _p_free_list
        _p_start_search_here = _p_free_list;
#if (GC_DEBUG>3)
        verify_space();
#endif
        p_return = _first_fit_with_start (real_size_bytes, 
                                          _p_free_list,
                                          double_alignment_required                                    
                                          );
        if (p_return == NULL) { // potential GC could make _p_start_search_here
                                // point into middle of coalisced free block so
                                // force _first_fit_with_start to reset it.
            _p_start_search_here = NULL;
        }
    }   // This could have failed also but if so just return p_return.

    return p_return;
}
   

//
// An object is being allocated from the fixed Large Object Space. Check to
// see if it will fit into this store. If not just return null since it may
// be able to fit into some other Large_Object_Store.
//
// If return_null_on_fail is true we need to return without waiting for locks
// so the JITed code can be moved to a GC safepoint.
//  
Java_java_lang_Object *
Large_Object_Store::gc_pinned_malloc(unsigned size, 
									 Partial_Reveal_VTable *p_vtable,
									 bool return_null_on_fail,
                                     bool double_alignment_required                               
                                     )
{
    //
    // The size can't be zero, since there is a requisite VTable pointer
    // word that is the part of every object, and the size supplied
    // to us in this routine should include that VTable pointer word.
    //
    assert(size!=0);
    assert(p_container);
    
#ifdef SPACE_DEMOGRAPHICS
    record_for_space_demographics(size);
#endif
    
#ifdef CLASS_DEMOGRAPHICS
    record_for_class_demographics(p_vtable);
#endif
    
    int attempt = 0;
    
    Partial_Reveal_Class *p_class = p_vtable->clss;
    // bool padding = false;
#if (GC_DEBUG>3)
    unsigned int real_size_bytes = add_gc_space_overhead(size);
#else
    unsigned int real_size_bytes = size;
#endif
    Java_java_lang_Object *p_return;
    
    // Make sure we have the lock for this LOS.
    assert (p_TLS_orpthread == get_lock_owner());

#if 0
#ifndef GC_SAPPHIRE // 1 // Grab the gc lock.    
    if (return_null_on_fail) {
        bool stat = p_gc_lock->_lock_enum_or_null (return_null_on_fail);
        if ((stat == false)) {
            return NULL;
        }
    } else {
        orp_enable_gc();
       	p_gc_lock->_lock_enum_or_null (return_null_on_fail);
        orp_disable_gc();
    }
#endif
#endif // 0
#ifdef RECLAIM_EVERY_ALLOCATION
    extern unsigned debug_reclaim_frequency;
    extern unsigned debug_reclaim_count;
    //
    // This is a debugging extension for the O2 project:
    // At every N allocations do a collection.
    // NOTE THIS CURRENTLY ONLY WORKS FOR GC_FIXED_V1
    //
    if (debug_reclaim_frequency>0) {
        if (debug_reclaim_count>=debug_reclaim_frequency) {
            if (orp_initialized) {
                if (return_null_on_fail) { 
                    assert(0);
#if 0
#ifndef GC_SAPPHIRE // 1 // Grab the gc lock.    
                    p_gc_lock->_unlock_enum_or_null(); // Release the lock....
#endif
#endif // 0
                    return NULL;
                } else {
                    debug_reclaim_count = 1;
                    forced_reclaim_heap();
                }
            }
        } else {
            debug_reclaim_count += 1;
        }
    }
#endif
    
#if (GC_DEBUG>3)
    if (orp_initialized) {
        // verify_space();
    }
#endif
    
    // We hold the gc *global* lock, try to get some space.
    p_return = _first_fit(real_size_bytes, double_alignment_required);
    
    if (p_return) {
        
        // OK, it worked - return it to the caller.
        p_return = initialize_object(p_return,
            real_size_bytes,
            p_vtable); 
        gc_trace (p_return, "Object created in pinned (los.cpp ln 575) area."); 
        //
        // If the address is the highest address object,
        // update our bookkeeping variable. This is used
        // for ensuring that we don't fall of the end
        // when doing a linear scan.
        // 
        if (p_return > _p_last_object) {
            _p_last_object = 
                (Java_java_lang_Object *)p_return;
        }
        
        _update_allocation_stats(p_class, real_size_bytes);
        //
        // Since we successfully allocated an object, clearly
        // the low space warning can be turned off.
        //
        _los_space_may_be_critically_low = false;
#if 0
#ifndef GC_SAPPHIRE // 1 // Grab the gc lock.    
        //unlock();
        p_gc_lock->_unlock_enum_or_null();
#endif
#endif

        // LeaveCriticalSection(&_lock);
        assert (p_return->vt);
        return p_return;
    } else {
        //
        // Bail out if the caller wants to avoid a GC
        //
#if 0
#ifndef GC_SAPPHIRE // 1 // Grab the gc lock.    
        p_gc_lock->_unlock_enum_or_null();
#endif
#endif
        return NULL;
    }
    
    assert (0); // Remove after debug.
}
 
//
// This is a special case version of gc_pinned_malloc, which is used
// at bootstrap time to create objects which don't have classes
// at that time. (The classes come later, and the VTable pointers
// are appropriately patched.) Need to build a common routine
// out of this and the previous one.
//
Java_java_lang_Object *
Large_Object_Store::gc_pinned_malloc_noclass(unsigned size)
{
#if (GC_DEBUG>0)
    //
    // The size can't be zero, since there is a requisite VTable pointer
    // word that is the part of every object, and the size supplied
    // to us in this routine should include that VTable pointer word.
    //
    assert(size!=0);
            // Classes loaded before java.lang.Class must use this API, because the
        // vtable for java.lang.Class hasn't been constructed yet.
        // Currently only three classes should be allocated through this call:
        // java.lang.Object, java.io.Serializable and java.lang.Class.
 
    assert ((size % GC_OBJECT_ALIGNMENT) == 0);

#if (GC_DEBUG>2)
	assert(p_container);
#endif
#endif

#ifdef SPACE_DEMOGRAPHICS
    record_for_space_demographics(size);
#endif

	int attempt = 0;
//	bool padding = false;

#if (GC_DEBUG>3)
    unsigned int real_size_bytes = add_gc_space_overhead(size);
#else
    unsigned int real_size_bytes = size;
#endif
	Java_java_lang_Object *p_return;

	p_gc_lock->_lock_enum();                     // vvvvvvvvvvvvv
#ifdef RECLAIM_EVERY_ALLOCATION
    extern unsigned debug_reclaim_frequency;
    extern unsigned debug_reclaim_count;
	//
	// This is a debugging extension for the O2 project:
	// At every N allocations do a collection.
    // NOTE THIS CURRENTLY ONLY WORKS FOR GC_FIXED_V1
	//
    if (debug_reclaim_frequency>0) {
        if (debug_reclaim_count>=debug_reclaim_frequency) {
            debug_reclaim_count = 1;
	        if (orp_initialized) {
                forced_reclaim_heap();
            }
        } else {
            debug_reclaim_count += 1;
        }
    }
#endif

#if (GC_DEBUG>3)
    if (orp_initialized) {
	    verify_space();
    }
#endif
	//
	// Try twice. If the first time fails, a collection will
	// be triggered, and under normal conditions the second
	// attempt should succeed.
	//
	while (attempt++ < 2) {

        // First fit seems faster. A bibop is used by some since
        // it gives a constant time access. Someday we probable should 
        // implement bibop, particularly if sapphire isn't a big win and
        // we need a concurrent collector.
#if 0
		if ((p_return = _best_fit(real_size_bytes)) != NULL) {
			p_return = initialize_object(p_return,
									real_size_bytes,
									NULL);
		} else 
#endif
        if ((p_return = _first_fit(real_size_bytes,
                                   false     
                                   )) != NULL) {
			p_return = initialize_object(p_return,
									real_size_bytes,
									NULL);
		}
		//
		// OK, it worked - return it to the caller.
		//
		if (p_return) {
			//
			// If the address is the highest address object,
			// update our bookkeeping variable. This is used
			// for ensuring that we don't fall of the end
			// when doing a linear scan.
			// 
			if (p_return > _p_last_object) {
				_p_last_object = 
					(Java_java_lang_Object *)p_return;
			}
			//
			// Since we successfully allocated an object, clearly
			// the low space warning can be turned off.
			//
			_los_space_may_be_critically_low = false;

			//_update_allocation_stats(p_class, real_size_bytes);
			p_gc_lock->_unlock_enum();                            // ^^^^^^^
//            LeaveCriticalSection(&_lock);

			return p_return;
		} else {
			//
			// Failed to allocate the specified size.
			//
			if (_los_space_may_be_critically_low) {

				//
				// Looks like the most recent allocation failed
				// to provide sufficiently free space.
				//
				_p_los_container->extend_los(real_size_bytes);

			} else {
				//
				// Turn the low space warning on.
				//
				_los_space_may_be_critically_low = true;
				_notify_large_object_store_exhausted(real_size_bytes);
			}
		}
	}
	//
	// We are out of space even after a reclamation.
	// Looks like there is no hope.
	//
	_notify_large_object_store_exhausted_error(real_size_bytes);
	p_gc_lock->_unlock_enum();                                      // ^^^^
    // LeaveCriticalSection(&_lock);
	//unlock();

	return NULL;
}

#if (GC_DEBUG>3)
void 
Large_Object_Store::inspect(unsigned int level) 
{
	assert(p_container);

	cout << "Inspecting the Large Object Store" << endl;
	cout << "   " << _resident_object_count << " objects" << endl;
	cout << "   " << _resident_occupied_bytes << " bytes" << endl;
	cout << "   " << ((Byte *)_p_ceiling - (Byte *)_p_free) << " bytes free" << endl;
}
#endif // _DEBUG

//
// We have reached the end of a minor collection, and our
// containing generation is telling us to clean up. We use
// this opportunity to do a linear scan of the LOS in order
// to detect dead objects and do the appropriate thing (run
// finalizers and add these to the free lists.)
//
// Delete, Roy and Rick disagree on this code, Rick needs to code review it.
// this is Ricks version that we will be using for now.

void 
Large_Object_Store::_linear_sweep_heap_and_reclaim()
{
//    orp_cout << " We need to grab the lock for this LOS and then reclaim rebuild ";
//    orp_cout << " the free list.  Who should grab the lock???" << endl;
	assert(p_container);
    //
    // Initialize the free list to NULL.
    //
    _p_free_list = NULL;
    _p_last_free = NULL;
    _last_object = true; // We can't coalesce since we start with no previous objects.
    //
    // Start scanning at the base of the LOS.
    //
    void *p_thing = _p_base;
    //
    // Keep a count of the number of objects scanned.
    //
    _number_scanned_objects = 0;
    //
    // Reset this. It will subsequently record the last
    // encountered live object. This will be used if we
    // discover at the end that the last object is now dead.
    //
    _p_last_live_object = (Java_java_lang_Object *)get_object_from_gc_header((Object_Gc_Header *)p_thing);
#ifdef _DEBUG
    void *back_one = p_thing;
    void *back_two = back_one;
    void *back_three = back_two;
    do {
        back_three = back_two;
        back_two = back_one;
        back_one = p_thing;
	    p_thing = _sweep_thing(p_thing);
        _number_scanned_objects++;
    } while (p_thing < (void *)_p_ceiling);
#else
    
    do {
	    p_thing = _sweep_thing(p_thing);
    } while (p_thing < (void *)_p_ceiling);
#endif //_DEBUG
// old    } while (p_thing < (void *)_p_last_object); WHY isn't this _p_ceiling?????????
#if (GC_DEBUG>3)
    if (orp_initialized) {
	    verify_space();
    }
#endif
    
    _p_start_search_here = NULL;
    assert (p_thing == _p_ceiling);
    return;
    //
    // Finally link up the space past the last object
    //
    if ((p_thing < _p_ceiling) && (is_free_block(p_thing))) {
            assert(is_free_block(p_thing));
#ifdef GC_STATS
            _stat_swept_free += get_free_block_size(p_thing);
#endif // GC_STATS
            if (_p_last_free == NULL) {
                assert (_p_free_list == NULL);
                // We found no free large objects in this collection but there
                // is a single free block left at the end of the space.
                // This can happen if we haven't yet allocated all of los and
                // a collection is caused by the small block collector.
                _p_last_free = p_thing;
                _p_free_list = p_thing;
                gc_trace (p_thing, "Linking onto free list los.cpp ln.904");
                set_free_block_link(p_thing, NULL);
            } else {
                if (_last_object) {
                    // Last object didn't get reclaimed so link up this last block

                    gc_trace (p_thing, "Linking onto free list los.cpp ln.910");
                    set_free_block_link(_p_last_free, p_thing);
                    set_free_block_link(p_thing, NULL);
                } else {
                    // We can Coalesce.
                    // Double check logic.
                    assert (( (POINTER_SIZE_INT)_p_last_free + 
                              (POINTER_SIZE_INT)get_free_block_size(_p_last_free) ) ==
                            (POINTER_SIZE_INT)p_thing);
                    gc_trace (p_thing, "Linking coalesce onto free list los.cpp ln.920");
                    _coalesce_free_blocks (get_free_block_size (p_thing),
                        _p_last_free,
                        p_thing);
                }
            }
    }
    //
    // Reset the pointer so that it is consistent.
    // This is to protect against a situation where the other LOS
    // initiates a GC resulting in this LOS's _p_start_search_here
    // getting blocks coalesced around it, resulting in an
    // invalid pointer.
    //
    _p_start_search_here = NULL;
}


void
Large_Object_Store::dump_stats()
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
    unsigned largest_block = 0;
    unsigned number_of_free_blocks = 0;
    void *p_thing = _p_base;

    while (p_thing < (void *)_p_ceiling) {
	    if (is_free_block(p_thing)) {
	        unsigned int free_block_size = get_free_block_size(p_thing);
            assert(free_block_size>0);
            //cout << " FREE " << free_block_size << " at " << p_thing << endl;
	        p_thing = (void *)((char *)p_thing + free_block_size);

            if (free_block_size > largest_block) {
                largest_block = free_block_size;
            }
            number_of_free_blocks += 1;
	    } else	if (is_object_gc_header(p_thing)) {

	        Object_Gc_Header *p_gc_hdr = (Object_Gc_Header *)p_thing;
            Java_java_lang_Object *p_obj   = get_object_from_gc_header(p_gc_hdr);

	        unsigned real_size_bytes = get_real_object_size_bytes(p_obj);
            assert(real_size_bytes>0);
            //cout << " OBJECT " << real_size_bytes << " at " << p_thing << endl;
            p_thing = 
                (void *)((char *)p_thing + real_size_bytes);
        } else {
            assert(0);
            orp_exit(1);
        }
    }

    cout << "Dumping State: " << number_of_free_blocks << " free. ";
    cout << largest_block << " largest" << endl;

}

//
// We just ran out of LOS. Notify the GC Interface.
//
void
Large_Object_Store::_notify_large_object_store_exhausted(unsigned int size)
{
	//
	// (Finally!) Notify enclosing GC Interface that the LOS is out of space
	//
    assert (0);
// 	_p_gc_interface->notify_out_of_space(999); // NOW CALLED BY CONTAINER LOGIC.
}

//
// No more LOS, even after a collection.
//
void
Large_Object_Store::_notify_large_object_store_exhausted_error(unsigned int size)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
	cout << "Error: Large Object Space Exhausted when allocating ";
	cout << size << " bytes" << endl;

//    dump_stats();

	orp_exit(1);
}

void
Large_Object_Store::_reclaim_object(Object_Gc_Header *p_gc_hdr)
{
    assert(p_container);
	unsigned int obj_size = get_real_object_size_bytes(p_gc_hdr);
	void *p_obj = (void *)p_gc_hdr;

    assert (verify_is_object (p_gc_hdr));

    if (_p_free_list==NULL) {
        // BUG BUG BUG.
        // We have a problem when we are allocating small 8 byte objects in the
        // fixed object space. We can't free them since the free block header requires
        // 12 bytes. Bummer. Since this is only used for debugging we will simply not reclaim them.
        if (obj_size < (3 * sizeof (void *))) {
            assert (obj_size != 0);
            _last_object = true; // Treat this as a live object.
#ifdef GC_STATS
            _stat_free_space_not_recovered += obj_size;
#endif
            return; // Leave object alone.
        }
        //
        // This is the first free slot.
        //
        reinitialize_free_block(p_obj, obj_size, NULL);
        _p_last_free = p_obj;
        _p_free_list = _p_last_free;
        _last_object = false;
    } else {
        //
        // Need to update the free list, perhaps coalese.
        //
        if (!_last_object) {
            //
            // We just found two adjacent free spaces.
            // _p_last_free doesn't change.
            //
            _coalesce_free_blocks (obj_size,
                        _p_last_free,
                        p_obj);
        } else {
            //
            // Nope, this free space has an object between it and
            // the previous free space.
            //
            // BUG BUG BUG.
            // We have a problem when we are allocating small 8 byte objects in the
            // fixed object space. We can't free them since the free block header requires
            // 12 bytes. Bummer. Since this is only used for debugging we will 
            // simply not reclaim them, but we do coalesce them..
            if (obj_size < (3 * sizeof (void *))) {            
                assert (obj_size != 0);  
                _last_object = true; // Treat this as a live object.
                // Unfortunately this means it will only coalesce with 
                // an object that is previously 
#ifdef GC_STATS
                _stat_free_space_not_recovered += obj_size;
#endif // GC_STATS
                return;
            } else {
                _update_free_list(p_obj, obj_size);
                _p_last_free = p_obj;
         
            }
        }
    }
    _last_object = false;

	return;
}


void *
Large_Object_Store::_skip_free_block(void *p_thing)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
	unsigned int free_block_size = get_free_block_size(p_thing);

	return (void *)((char *)p_thing + free_block_size);
}


void *
Large_Object_Store::_skip_free_block_and_update_free_list(void *p_thing)
{
	unsigned int free_block_size = get_free_block_size(p_thing);

    assert (free_block_size != 0);
    assert (free_block_size >= 3*sizeof(POINTER_SIZE_INT));
	assert(p_container);
#ifdef GC_STATS
    _stat_swept_free += free_block_size;
#endif // GC_STATS 
    if (_p_free_list==NULL) {
        //
        // This is the first free block encountered so far.
        //
        _p_free_list = p_thing;
        _last_object = false;
        _p_last_free = _p_free_list;
    } else {
        if (_last_object) {
            //
            // There was an object between two free spaces.
            // Update the previous one to point to this one.
            //
            _update_free_list(p_thing, free_block_size);
            _p_last_free = p_thing;
        } else {
            //
            // We just found two adjacent free blocks.
            // _p_last_free doesn't change.
            //
            _coalesce_free_blocks(free_block_size,
                _p_last_free,
                p_thing);
        }
    }
    //
    // Set the state to help scan the next thing.
    //
    _last_object = false;

	return (void *)((char *)p_thing + free_block_size);
}

//
// Link the newly-found free block with the earlier free block
// that is pointed to by _p_last_free.  There is one or more
// objects between these two.
//
void 
Large_Object_Store::_update_free_list(void *p_block,
                                      unsigned int free_block_size)
{

    assert (free_block_size >= (3 * sizeof (void *)));
    assert(_p_last_free);
	assert(p_container);

    //
    // Initialize the size field for this new block.
    //
    reinitialize_free_block(p_block, free_block_size, NULL);
    //
    // Link the previous block to point to this one.
    //
    set_free_block_link(_p_last_free, p_block);
    //
    // Bump up the last free pointer.
    //
    _p_last_free = p_block;
}   

//
// Link two adjacent blocks together, one passed in as an argument,
// the other pointer to by _p_last_free. Note that _p_last_free is
// always lower in memory.
//
void
Large_Object_Store::_coalesce_free_blocks(unsigned int free_block_size,
                                          void *previous_free_block,
                                          void *p_this_free_block)
{
    assert (_p_last_free == previous_free_block);
    assert(_p_last_free);
	assert(p_container);
    // Double check logic.
    assert (( (POINTER_SIZE_INT)_p_last_free + 
              (POINTER_SIZE_INT)get_free_block_size(_p_last_free) ) ==
            (POINTER_SIZE_INT)p_this_free_block);

    unsigned int current_size = get_free_block_size(_p_last_free);
    unsigned int new_size     = current_size + free_block_size;
    //
    // Create a new larger sized block:
    //
    reinitialize_free_block(_p_last_free, new_size, NULL); 
    //
    // _p_last_free does not change.
    //
}

void *
Large_Object_Store::_skip_java_object(void *p_thing)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
	Object_Gc_Header *p_gc_hdr = (Object_Gc_Header *)p_thing;
    Java_java_lang_Object *p_obj   = get_object_from_gc_header(p_gc_hdr);

	int real_size_bytes = get_real_object_size_bytes(p_obj);

    void *p_next = 
        (void *)((char *)p_thing + real_size_bytes);

    return p_next;
}
// p_thing is a pointer to an object header to get object one must 
// skip over the header.
void *
Large_Object_Store::_sweep_java_object(void *p_thing)
{
	assert(p_container);
	Object_Gc_Header *p_gc_hdr = (Object_Gc_Header *)p_thing;
    Java_java_lang_Object *p_obj = get_object_from_gc_header(p_gc_hdr);

	int real_size_bytes = get_real_object_size_bytes(p_obj);

    void *p_next = 
        (void *)((char *)p_thing + real_size_bytes);

	if (is_object_marked(p_obj)) {
        gc_trace (p_obj, "Object is marked and not reclaimed los ln. 1225");
		//
		// Record this most recent live object, which may
		// later be a candidate for the last object, in case
		// the current last object is dead.
		//
		_p_last_live_object = p_obj;
		//
		// Clear the object, since we don't need it marked
		// any more.
		//
		set_object_unmarked(p_obj);	
        //
        // Set the state to signal that an object was encountered.
        //
        _last_object = true;
#ifdef GC_STATS
        //
        // Update statistics.
        //
        _stat_swept_live += real_size_bytes;

#endif // GC_STATS
	} else {
        gc_trace (p_obj, "Object is not marked and is reclaimed los ln. 1225");
		//
		// Object is not marked, hence it is toast.
		// Check if we are reclaiming the last object.
		//
		if (p_obj == _p_last_object) {
			//
			// Pass the torch onto the last live object.
			//
			_p_last_object = _p_last_live_object;
		}
		_reclaim_object(p_gc_hdr); // An object might not be reclaimed if it is too small
                                   // in which case _last_object will come back true. So it 
                                   // should not be set in this routine.
#ifdef GC_STATS
        //
        // Update statistics.
        //
        _stat_swept_reclaimed += real_size_bytes;
#endif // GC_STATS
	}
	return p_next;
}

//
// The generation may not be available when the first LOS
// is created. Therefore it may need to be patched later.
// Earlier we created LOSs first because they were large,
// monolithic beasts. Now that we have non-contiguous,
// variable sized loses that grow on demand, we don't need
// to create them first. Need to refine this.
//
void 
Large_Object_Store::set_generation(Generation *p_gen) 
{
	p_container = p_gen;
}

void *
Large_Object_Store::_sweep_thing(void *p_thing)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif

	void *p_next_thing_to_sweep = NULL;

	if (is_free_block(p_thing)) {
		p_next_thing_to_sweep = _skip_free_block_and_update_free_list(p_thing);

	} else	if (is_object_gc_header(p_thing)) {
		p_next_thing_to_sweep = _sweep_java_object(p_thing);
	}
    else {
        assert (0);
    }

	return p_next_thing_to_sweep;
}
 
void 
Large_Object_Store::_update_allocation_stats(Partial_Reveal_Class *p_class, 
							                 unsigned int real_size_bytes)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif

	_resident_object_count++;
	_resident_occupied_bytes += real_size_bytes;
 
	if (real_size_bytes < _smallest_object_size)
		_smallest_object_size = real_size_bytes;

	if (real_size_bytes > _largest_object_size)
		_largest_object_size = real_size_bytes;
}

 
bool 
Large_Object_Store::_verify_object_is_in_large_object_store(Java_java_lang_Object *p_obj)
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif
	Gc_Space *p_container = 
		_p_block_store->p_get_object_container(p_obj);

	if (p_container == this) {
			return true;
	}
	return false;
}


#if (GC_DEBUG>3)
bool 
Large_Object_Store::verify_space()
{
	assert(p_container);

    if (_p_free_list==_p_base) {
        //
        // The LOS is empty.
        //
        return true;
    }
	//
	// Do a linear scan starting from the base of the heap.
	//
	void *p_scan = (Object_Gc_Header *)_p_base;

	//while (p_scan <= _p_last_object_gc_header) {
    while(p_scan < _p_ceiling) { // scan the last free block too
		//
		// See if the thing at this offset is a java object or
		// a free block.
		//
		if (is_object_gc_header(p_scan)) {
			//
			// If it is a java object, verify it:
			//
			//p_scan = p_scan_forward_over_object(p_scan);
            p_scan = _p_verify_and_skip_over_object(p_scan);
		} else {
			//
			// Otherwise it is a free block. Verify and skip over.
			//

            p_scan = _p_verify_and_skip_over_free_block(p_scan);

		}
	}
	return true;
}
#endif // _DEBUG

#if (GC_DEBUG>3)
//
// The verification aspect of the following routine kicks in when 
// we write a distinct pattern into the free block when initializing
// it. THen in this routine we can check for that pattern.
//
void *
Large_Object_Store::_p_verify_and_skip_over_free_block(void *p_scan)
{
	unsigned int free_block_size = 
		get_free_block_size(p_scan);

	void *p_next_free_block = get_free_block_link(p_scan);

    if ((p_next_free_block!=NULL)&&(is_object_gc_header(p_next_free_block))) {
        cerr << "Detected LOS integrity problem when scanning" << endl;
        orp_exit(1);
    }
	//
	// Locate the next slot past this free block.
	//
	void *p_return = (Object_Gc_Header *)((char *)p_scan + free_block_size);

    return p_return;
}
#endif // GC_DEBUG>3

#if (GC_DEBUG>3)
void *
Large_Object_Store::_p_verify_and_skip_over_object(void *p_scan)
{
	Object_Gc_Header *p_gc_hdr = (Object_Gc_Header *)p_scan;
    Java_java_lang_Object *p_obj   = get_object_from_gc_header(p_gc_hdr);

	int real_size_bytes = get_real_object_size_bytes(p_obj);

    void *p_next = 
        (void *)((char *)p_scan + real_size_bytes);

	if (is_object_marked(p_obj)) {
		//
        // During verification this would constitute an error, sice
        // the scan happens just before, or just after, a collection.
        // The ORP guarantees that the busy bit is not set, and the
        // GC guarantees that the mark bit is not set at these times.
        //
        cerr << "Detected LOS integrity problem when scanning:";
        cerr << "object " << p_obj << " was marked " << endl;
        orp_exit(1);
    } 

	return p_next;
}
#endif // GC_DEBUG>3

bool
Large_Object_Store::_walk_los(bool (*func)(Object_Gc_Header *))
{
#if (GC_DEBUG>2)
	assert(p_container);
#endif

    void *p_thing = _p_base;

    while (p_thing <= _p_last_object) {
        if (is_free_block(p_thing)) {
            p_thing = _skip_free_block(p_thing);
        } else {
            func((Object_Gc_Header *)p_thing);
            p_thing = _skip_java_object(p_thing);
        }
    }
    return true;
}

ORP_thread *Large_Object_Store::get_lock_owner()
{
    return this_los_lock;
}   

bool Large_Object_Store::try_lock ()
{
    if (InterlockedCompareExchangePointer (
        (PVOID *)&this_los_lock,
        (PVOID)p_TLS_orpthread, 
        (PVOID)NULL) == 
        (PVOID)NULL) {
        return true; // success
    } else {
        return false;
    }
}

void Large_Object_Store::release_lock()
{
    assert (p_TLS_orpthread == this_los_lock);
    this_los_lock = NULL;
}

//
// Scan to see if there is room for this block.
//
bool Large_Object_Store::is_block_available(unsigned int size)
{
    void *p_free = _p_free_list;  
    while (p_free != NULL) {     
        unsigned available_chunk_size = get_free_block_size(p_free);
        if (available_chunk_size > size) {
            return true;
        }
		p_free = get_free_block_link(p_free);
	}
	return false;
} // is_block_available
