// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/method_lookup.cpp,v 1.4 2001/12/07 00:16:00 xli18 Exp $
//


#include "platform.h"
#include <assert.h>
#ifdef ORP_VTUNE_SUPPORT
#include "..\ia32_o1_jit\iJITProf.h"
#include "orp_vtune.h"
#endif
#include "nogc.h"
#include "orp_stats.h"

#include "method_lookup.h"


#define EIP_CACHE_SIZE 512

Method_Lookup_Table methods;



Method_Lookup_Table::Method_Lookup_Table()
{
    _next_entry = 0;
    _capacity   = 0;
    _table      = 0;
    _cache      =
        (JIT_Specific_Info **)gc_malloc_fixed_data_C(EIP_CACHE_SIZE * sizeof(JIT_Specific_Info *));
    memset(_cache, 0, EIP_CACHE_SIZE * sizeof(JIT_Specific_Info *));
    reallocate(511);
} //Method_Lookup_Table::Method_Lookup_Table



void Method_Lookup_Table::reallocate(unsigned new_capacity)
{
    JIT_Specific_Info **new_table =
        (JIT_Specific_Info **)gc_malloc_fixed_data_C(new_capacity * sizeof(JIT_Specific_Info *));
    assert(new_table);
    assert(_next_entry <= _capacity);
    assert(_next_entry < new_capacity);
    memcpy(new_table, _table, _next_entry * sizeof(JIT_Specific_Info *));
    if(_table) {
        gc_free_fixed_data_C(_table);
    }
    _table = new_table;
    _capacity = new_capacity;
} //Method_Lookup_Table::reallocate



void Method_Lookup_Table::add(JIT_Specific_Info *m)
{
    p_meth_addr_table_lock->_lock();


#ifdef ORP_VTUNE_SUPPORT
    //:M Do not need to do this step, since we have JIT VTune_Support. --wy.
//    vtune_notify_load_finished(m);
#endif

    void *code_block_addr = m->get_code_block_addr();
    unsigned code_block_size = m->get_code_block_size();

#ifdef _TRACE
    orp_cout << "Adding: "
             << m->get_method()->get_class()->name->bytes << "."
             << m->get_method()->get_name()->bytes
             << m->get_method()->get_descriptor()
             << " [" << code_block_addr << ".."
             << (void *)(((unsigned)code_block_addr) + code_block_size)
             << "]" << endl;
#endif

    // If the table is full, allocate more memory.
    if(_next_entry >= _capacity) {
        reallocate(2 * _capacity);
    }

    unsigned idx = find_index(code_block_addr);

    for(unsigned i = _next_entry; i > idx; i--) {
        _table[i] = _table[i-1];
    }
    _table[idx] = m;
    _next_entry++;
#if 0 // def _DEBUG
    verify();
    Method *meth = Meth_Addr_Table::find_method(code_block_addr);
    assert(meth);
#ifdef _TRACE
    dump();
#endif
#endif

    p_meth_addr_table_lock->_unlock();
} //Method_Lookup_Table::add



unsigned Method_Lookup_Table::find_index(void *addr)
{
    unsigned L = 0, R = _next_entry;
    JIT_Specific_Info **table = _table;
    
    while(L < R) {
        unsigned M = (L + R) / 2;
        JIT_Specific_Info *m = _table[M];
        void *code_block_addr = m->get_code_block_addr();
        unsigned code_block_size = m->get_code_block_size();
        void *code_end_addr = (void *)((char *)code_block_addr + code_block_size);

        if(addr < code_block_addr) {
            R = M;
        } else if(addr >= code_end_addr) {
            L = M + 1;
        } else {
            return M;
        }
    }
    return L;
} //Method_Lookup_Table::find_index


#define USE_METHOD_LOOKUP_CACHE


JIT_Specific_Info *Method_Lookup_Table::find(void *addr)
{
#ifdef USE_METHOD_LOOKUP_CACHE
    // First try the cache.  There's no need for a lock.
    unsigned cache_idx = ((unsigned)addr) % EIP_CACHE_SIZE;
    JIT_Specific_Info *guess = _cache[cache_idx];
    if(guess) {
        void *guess_start = guess->get_code_block_addr();
        void *guess_end = ((char *)guess->get_code_block_addr()) + guess->get_code_block_size();
        if(addr >= guess_start && addr < guess_end) {
#ifdef ORP_STATS
            orp_stats_total.num_method_lookup_cache_hit++;
#endif
            return guess;
        }
    }
#endif

#ifdef ORP_STATS
    orp_stats_total.num_method_lookup_cache_miss++;
#endif

    p_meth_addr_table_lock->_lock();

    unsigned L = 0, R = _next_entry;
    JIT_Specific_Info **table = _table;
    
    while(L < R) {
        unsigned M = (L + R) / 2;
        JIT_Specific_Info *m = _table[M];
        void *code_block_addr = m->get_code_block_addr();
        unsigned code_block_size = m->get_code_block_size();
        void *code_end_addr = (void *)((char *)code_block_addr + code_block_size);

        if(addr < code_block_addr) {
            R = M;
        } else if(addr > code_end_addr) {
            // Should this be (addr >= code_end_addr)?
            L = M + 1;
        } else {
            p_meth_addr_table_lock->_unlock();
#ifdef USE_METHOD_LOOKUP_CACHE
            _cache[cache_idx] = m;
#endif
            return m;
        }
    }

    p_meth_addr_table_lock->_unlock();

    return 0;
} //Method_Lookup_Table::find



JIT_Specific_Info *Method_Lookup_Table::find_deadlock_free(void *addr)
{
    bool ok = p_meth_addr_table_lock->_lock_or_null();             // vvv
    if(ok) {
        // We acquired the lock.  Can use the fast lookup.
        JIT_Specific_Info *m = find(addr);
        p_meth_addr_table_lock->_unlock_or_null();                 // ^^^
        return m;
    } else {
        // We failed to acquire the lock.  Use slow linear search.
        // The linear search is safe even is someone else is adding a method
        // because of the way the table is modified:
        // 1. If neecssary, the table is reallocated.  This operation is
        //    atomic, so we never see a partially initialized table.
        // 2. Space is made for the new method by shifting all methods
        //    with higher addresses by 1.  The shift is done from the right,
        //    so we never lose an element in the linear search from the left.
        //    We could see the same method twice, but this is safe.
        for(unsigned i = 0; i < _next_entry; i++) {
            JIT_Specific_Info *m = _table[i];
            void *code_block_addr = m->get_code_block_addr();
            unsigned code_block_size = m->get_code_block_size();
            void *code_end_addr = (void *)((char *)code_block_addr + code_block_size);
            if((addr >= code_block_addr) && (addr <= code_end_addr)) {
                return m;
            }
        }
        return 0;
    }
} //Method_Lookup_Table::find_deadlock_free



void Method_Lookup_Table::unload_all()
{
    for(unsigned i = 0; i < _next_entry; i++) {
        JIT_Specific_Info *m = _table[i];
#ifdef ORP_VTUNE_SUPPORT
        vtune_notify_unload_start(m);
#endif
    }
} //Method_Lookup_Table::unload_all



#ifdef ORP_STATS

void Method_Lookup_Table::print_stats()
{
    unsigned code_block_size     = 0;
    unsigned rw_data_block_size  = 0;
    unsigned jit_info_block_size = 0;
    unsigned i;
    int num;

    if(orp_print_total_stats_level > 1) {
        num = 0;
        printf("Methods throwing exceptions:\n");
        for(i = 0; i < _next_entry; i++) {
            JIT_Specific_Info *m = _table[i];
            if(m->num_throws) {
                num++;
                printf("%8llu: %s.%s%s\n",
                       m->num_throws,
                       m->get_method()->get_class()->name->bytes,
                       m->get_method()->get_name()->bytes,
                       m->get_method()->get_descriptor());
            }
        }
        printf("(A total of %d methods)\n", num);

        num = 0;
        printf("Methods catching exceptions:\n");
        for(i = 0; i < _next_entry; i++) {
            JIT_Specific_Info *m = _table[i];
            code_block_size     += m->_code_block_size;
            rw_data_block_size  += m->_rw_data_block_size;
            jit_info_block_size += m->_jit_info_block_size;
            if(m->num_catches) {
                num++;
                printf("%8llu: %s.%s%s\n",
                       m->num_catches,
                       m->get_method()->get_class()->name->bytes,
                       m->get_method()->get_name()->bytes,
                       m->get_method()->get_descriptor());
            }
        }
        printf("(A total of %d methods)\n", num);
    }

    if(orp_print_total_stats_level > 2) {
        num = 0;
        printf("Methods unwinds (GC):\n");
        for(i = 0; i < _next_entry; i++) {
            JIT_Specific_Info *m = _table[i];
            if(m->num_unwind_java_frames_gc) {
                num++;
                printf("%8llu: %s.%s%s\n",
                       m->num_unwind_java_frames_gc,
                       m->get_method()->get_class()->name->bytes,
                       m->get_method()->get_name()->bytes,
                       m->get_method()->get_descriptor());
            }
        }
        printf("(A total of %d methods)\n", num);

        num = 0;
        printf("Methods unwinds (non-GC):\n");
        for(i = 0; i < _next_entry; i++) {
            JIT_Specific_Info *m = _table[i];
            if(m->num_unwind_java_frames_non_gc) {
                num++;
                printf("%8llu: %s.%s%s\n",
                       m->num_unwind_java_frames_non_gc,
                       m->get_method()->get_class()->name->bytes,
                       m->get_method()->get_name()->bytes,
                       m->get_method()->get_descriptor());
            }
        }
        printf("(A total of %d methods)\n", num);
    }

    printf("Total size of code blocks: %16d\n", code_block_size);
    printf("              data blocks: %16d\n", rw_data_block_size);
    printf("          JIT info blocks: %16d\n", jit_info_block_size);
} //Method_Lookup_Table::print_stats

#endif



JIT_Specific_Info *Method_Lookup_Table::get_first_method_jit(JIT *jit)
{
    for(unsigned i = 0; i < _next_entry; i++) {
        JIT_Specific_Info *m = _table[i];
        if(m && m->get_jit() == jit) {
            return m;
        }
    }
    return 0;
} //Method_Lookup_Table::get_first_method_jit



JIT_Specific_Info *Method_Lookup_Table::get_next_method_jit(JIT_Specific_Info *j)
{
    unsigned idx = find_index(j->get_code_block_addr());
    JIT *jit = j->get_jit();

    for(unsigned i = idx + 1; i < _next_entry; i++) {
        JIT_Specific_Info *m = _table[i];
        if(m && m->get_jit() == jit) {
            return m;
        }
    }
    return 0;
} //Method_Lookup_Table::get_next_method_jit



ORP_Code_Type orp_identify_eip(void *addr)
{
    JIT_Specific_Info *m = methods.find(addr);
    if(!m) {
        return ORP_TYPE_UNKNOWN;
    }

    if(m->get_method()->is_native()) {
        return ORP_TYPE_NATIVE_STUB;
    } else {
        return ORP_TYPE_JAVA;
    }
} //orp_identify_eip



ORP_Code_Type orp_identify_eip_deadlock_free(void *addr)
{
    JIT_Specific_Info *m = methods.find_deadlock_free(addr);
    if(!m) {
        return ORP_TYPE_UNKNOWN;
    }

    if(m->get_method()->is_native()) {
        return ORP_TYPE_NATIVE_STUB;
    } else {
        return ORP_TYPE_JAVA;
    }
} //orp_identify_eip



