// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_io_FileInputStream.cpp,v 1.7 2001/10/15 09:02:17 gwu2 Exp $
//


#include "platform.h"


#include <sys/types.h>
#include <sys/stat.h>

#ifdef ORP_NT
#include <io.h>
#endif

#include "native_utils.h"
#include "jni_direct.h"
#include "jni_utils.h"
#include "object_layout.h"
#include "exceptions.h"

#include <fcntl.h>

#ifdef ORP_NT
#include <direct.h>
#elif defined(ORP_POSIX)
#include <unistd.h>
#endif

#include <errno.h>

#include "java_io_FileInputStream.h"
#include "gnu_classpath_jni_utils.h"

/*
 * Class:     FileInputStream
 * Method:    closeInternal
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_java_io_FileInputStream_closeInternal
  (JNIEnv *jenv, jobject jobj, jint file_d)
{   
    if (file_d == -1)
        return;  // don't try to close an already closed fd

    //we don't close in/out/err, there're such attempts in FileXputStream.finalize()
	if(file_d <= 2) 
		return;

	jint ret = close(file_d);
	if (ret) { 
		// Throw IOException since write failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", "close() failed");
		return;
	}
	return;
}


/*
 * Class:     FileInputStream
 * Method:    getFileLength
 * Signature: (I)J
 */
JNIEXPORT jlong JNICALL Java_java_io_FileInputStream_getFileLength
  (JNIEnv *jenv, jobject jobj, jint file_d)
{
	// Determine the file size now.
	struct _stat buf;
	int result;

	// Get data associated with "fd"
#ifdef ORP_NT
	result = _fstat(file_d, &buf );
#else
	//result = 0; assert(0);
	result = fstat(file_d, &buf );
#endif

   // Check if statistics are valid
	if(result != 0) {
		perror("Bad file handle\n");
		orp_exit(-1);
	}

	jlong ret = buf.st_size;
	assert(ret >= 0);
	return ret;

}


/*
 * Class:     FileInputStream
 * Method:    open
 * Signature: (Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_java_io_FileInputStream_open
  (JNIEnv *jenv, jobject jobj, jstring jpath)
{
	jboolean is_copy;
	const char *path = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path); 
	
	int fmode = O_RDONLY;
	
#ifdef ORP_NT
	// Rope in _O_BINARY so that Ctrl-Z will not be interpreted as an EOF indicator, 
	// and translations are suppressed. Is it okay to have this ALWAYS ??? BUG?????
	fmode |= _O_BINARY;
#endif

	extern char *get_corrected_path(const char * path_name);
	char *ca = get_corrected_path(path);

	// Actually open the file; use default permissions
#ifdef ORP_NT
    int file_descr = _open(ca, fmode, 0755);
#elif defined (ORP_POSIX)
	int file_descr = open(ca, fmode, 0755); //change open(path...) to open(ca...);
#endif

	if (file_descr == -1) {
		char *msg = (char *) malloc(300);
#if 0 
		// Need to get strerror() working...23 compile errors!	
		sprintf(msg, "%s: %s", path, strerror (errno));
#else
		strcpy(msg, "Open failed???");
#endif

		throw_exception_from_jni(jenv, "java/io/FileNotFoundException", msg);
		free(msg);
	}
	return file_descr;
}


/* RNI format implementation for FileInputStream.readInternal().
 * Used to speed up reads from files.
 * The JNI implmentation is still kept around and 
 * called by Java_java_io_RandomAccessFile_readInternal.
 */


int java_io_FileInputStream_readInternal(Java_java_io_FileInputStream *pThis,
                                       int fd,
                                       JavaArrayOfByte *pArrayOfByte,
                                       int  offset,
                                       int len)
{
    int status;
	void *bufp = &pArrayOfByte->body[offset];

    status = read(fd, bufp, len);

    if (status < 0) {
        throw_java_exception("java/io/IOException");
    }
   
    if (status == 0) { 
        status = -1;
    }
    return status;
}


/*
 * Class:     FileInputStream
 * Method:    readInternal
 * Signature: (I[BII)I
 */
JNIEXPORT jint JNICALL Java_java_io_FileInputStream_readInternal
  (JNIEnv *jenv, jobject jobj, jint fd, jbyteArray barr, jint offset, jint count)
{
	// Read into byte array "count" bytes starting at index "offset"

	// Check if we have been passed a null pointer
	if (!barr) {
		throw_exception_from_jni(jenv, "java/lang/NullPointerException", 
					"Null pointer passed to read()");
		return 0;
	}

	// Do index check for the array.
	int array_len = jenv->GetArrayLength(barr);
	if (count < 0 || offset < 0 || offset + count > array_len) {
		throw_exception_from_jni(jenv, "java/lang/ArrayIndexOutOfBoundsException", 
					"Index check failed");
		return 0;
	}
	
	// Get the array reference.
	jboolean is_copy;
	jbyte *jbytes = jenv->GetByteArrayElements(barr, &is_copy);
	assert(jbytes);

	int ret = read (fd, &jbytes[offset], count);

    if (ret == 0) {
        jenv->ReleaseByteArrayElements(barr, jbytes, 0);
		return -1;
    }

#if 0 
	// Code below can be activated only when the native method currentThread() is 
	// implemented in java/lang/Thread.

	if (thread_was_interrupted(jenv)) {
		// Throw InterruptedIOException since thread was interrupted.
		throw_exception_from_jni(jenv, "java/io/InterruptedIOException",
						"read was interrupted");
		return 0;
	} else if (ret == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
		return 0;
	}
#else
	// No "interrupted" check for now.
	if (ret == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
		return 0;
	}
	assert(ret >= 0);
#endif

	// We need to copy back the modified array.
	jenv->ReleaseByteArrayElements(barr, jbytes, 0);
	return ret;
}




/*
 * Class:     FileInputStream
 * Method:    skipInternal
 * Signature: (IJ)J
 */
JNIEXPORT jlong JNICALL Java_java_io_FileInputStream_skipInternal
  (JNIEnv *jenv, jobject jobj, jint file_d, jlong skip_count)
{
#ifdef ORP_NT
    int64 oldpos = _lseek(file_d, 0L, SEEK_CUR);  // get current position
#elif defined (ORP_POSIX)
    int64 oldpos = lseek(file_d, 0L, SEEK_CUR);  // get current position
#endif
	if (oldpos == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
		return 0;
	}

#ifdef ORP_NT
    int64 pos = _lseek(file_d, (long)skip_count, SEEK_CUR); // go to destination
#elif defined (ORP_POSIX) 
    int64 pos = lseek(file_d, (off_t)skip_count, SEEK_CUR); // go to destination
#endif

	if (pos == -1 || pos < oldpos) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
		return 0;
	}
    return pos - oldpos;
}
